/************************************************************************
* Authors: Riley Barnard, Kamyar Izat, Alec Selfridge, Dylan Smith
* Date:    12/12/2015
* Rev:     1.6
* Project: RTOS Light Animations
* Notes:   The z-axis of the accelerometer is used to calculate the 
*          "effective" angle that the board is tilted. First, the Z data
*          (0-65) is converted to a (0-90) scale. Then the actual
*          manipulation occurs by injecting the acc data into a response
*          formula in the form of ax^2 - by + c. The resulting angle has
*          no sign but it's magnitude is used to determine the rate at 
*          which LEDs will "fill", thus simulating a gravity effect. To
*          account for both directions of tilt we check the polarity of 
*          the x-axis. A negative value indicates a tilt to the right
*          and vice versa.
*          A 5-axis joystick is used to mix R, G, and B values to create 
*          a custom color. The RGB LED is controled via PWM with a value
*          of 1.0 indicating "off" and vice versa.
*          The joystick itself is off-site on a slave board connected via
*          SPI. Each component is requested and applied when all 3 are
*          received.
*          SYSTEM:
*            Threads: 2
*            Timers:  2
*            ISRs:    1
*
************************************************************************/

#include "mbed.h"
#include "rtos.h"
#include "MMA7455.h"

// response formula coefficients
#define X2 0.0038
#define X  0.0299
#define C  0.1883

#define R_CODE 1
#define G_CODE 2
#define B_CODE 3

/*
******************************************************
                  Display Functions
******************************************************
*/
void displayBanner();
void eol();
void testLED(void);
void testRGB(void);
void LEDPush(void const *args);
void RGBupdate(void const *args);
void RGBPush(void);
unsigned char mapLEDS(double v);
/*
******************************************************
                  Utility Functions
******************************************************
*/
void SerialInit(void);
void AccInit(void);
void LEDsInit(void);
void RgbInit(void);
void accFeed(void);
void sampleAcc(int period, int32_t data[3]);
double getAngle(void);
int testSlave(void);
void checkSSEL_isr(void);
void unpackRGB(int tempR, int tempG, int tempB);
void packRGB (int R, int G, int B);
/*
******************************************************
                    Global Data
******************************************************
*/
const char banner[37] = "Serial Comm Established with LPC4088";
// x, y, z
// holds current position
int32_t accPos[3]           = {};
// holds up to 20 samples of data
int32_t accData[20][20][20] = {};
// holds calibration offsets
int32_t accCal[3]           = {};
// status register for R, G, and B
bool sentStatus[3]          = {};
// holds most recent, adjusted, acc value
double acc_adj              = 0.0;
// a value representing what's shown on the LEDs
uint8_t LEDvals             = 0;
// red value of RGB
float r                     = 0;
// green value of RGB
float g                     = 0;
// blue value of RGB
float b                     = 0;
// holds R value from slave
int Rtemp                   = 0;
// holds the G value from slave
int Gtemp                   = 0;
// holds the B value from the slave
int Btemp                   = 0;
/*
******************************************************
                     Objects
******************************************************
*/
// UART connection to PC
Serial terminal(USBTX, USBRX);
// accelerometer on I2C bus
MMA7455 acc(P0_27, P0_28);
// SPI interface for 8 LEDs & slave processor
SPI leds(p5, p6, p7);
// load for shift register
DigitalOut ld(p30);
// cs for SPI slave board
DigitalOut cs(p8);
// R of RGB LED
PwmOut R(p25);
// G of RGB LED
PwmOut G(p28);
// B of RGB LED
PwmOut B(p26);
// interrupt to check for SPI errors
Ticker tock;
/*
******************************************************
                   Main Execution
******************************************************
*/
int main() {
    SerialInit();
    AccInit(); 
    LEDsInit();
    testLED();
    RgbInit();
    testRGB();
    
    terminal.printf("Initializing slave...........");
    if(testSlave()) {
        terminal.printf("done.");
        eol();
    }
    else {
        terminal.printf("failed.");
        eol();
    }
    terminal.printf("Initializing timers..........");
    // timer responsible for updating the LEDs
    RtosTimer refresh_timer(LEDPush, osTimerPeriodic, (void *)0);
    // 16.7Hz timer (60ms or ~1/2 of 30fps)
    refresh_timer.start(60);
    RtosTimer update_timer(RGBupdate, osTimerPeriodic, (void *)0);
    // 10Hz timer (100ms)
    update_timer.start(100);
    terminal.printf("done.");
    eol();
    terminal.printf("Initializing interrupts......");
    tock.attach(checkSSEL_isr, 0.03);
    terminal.printf("done.");
    eol(); eol();
    terminal.printf("Initialization complete.");
    eol(); 
    
    // main's only job is to update the LED array with acc data
    while(true) {
        acc.read(accPos[0], accPos[1], accPos[2]);
        acc_adj = getAngle();
        LEDvals = mapLEDS(acc_adj);
    }
}
/*
******************************************************
                Function Definitions
******************************************************
*/
void SerialInit()
{
    // initialize connection to PC. default: 8N1
    terminal.baud(19200);
    displayBanner();
    eol();
}

void AccInit()
{
    // configure accelerometer for 2G range
    acc.setMode(MMA7455::ModeMeasurement);
    acc.setRange(MMA7455::Range_2g);
    terminal.printf("Calibrating accelerometer....");
    
    // if we can successfully calibrate the accelerometer...
    if(acc.calibrate()) {
        eol();
        acc.getCalibrationOffsets(accCal[0], accCal[1], accCal[2]);
        terminal.printf("  Offsets are (x,y,z): (%d, %d, %d)", accCal[0], accCal[1], accCal[2]);
        eol(); eol();
    }
    else {
        terminal.printf("failed.");
        eol(); eol();
    }
}

void LEDsInit()
{
    terminal.printf("Initializing LED array.......");
    leds.format(8, 3);      // 8-bit packet, polarity & phase mode 3
    leds.frequency(100000); // 1MHz
    ld = 1;
    LEDvals = 0x00;
    terminal.printf("done.");
    eol();
}

void RgbInit()
{
    terminal.printf("Initializing RGB LED.........");
    // 1KHz
    R.period(.001);
    R = 1.0;
    G.period(.001);
    G = 1.0;
    B.period(.001);
    B = 1.0;
    terminal.printf("done.");
    eol();
}

// inserts end-of-line
void eol()
{
    // newline = carriage return + line feed
    terminal.putc('\n');
    terminal.putc('\r');
}

/*
    Displays the following header:
************************************************                                
Serial Comm Established with LPC4088                                            
************************************************ 
*/
void displayBanner()
{
    int i = 0;
    for(int j = 0; j < 48; j++)
        terminal.putc('*');  
    eol();
          
    while(i != 36) {
        char c = banner[i];
        terminal.putc(c);
        i++;
    }
    eol();
    
    for(int j = 0; j < 48; j++)
        terminal.putc('*');    
}

// prints the current positional data from the accelerometer
void accFeed()
{
    // returns false if the mode is set to standby or unable to convert
    if(acc.read(accPos[0], accPos[1], accPos[2])) {             
        terminal.printf("x: %d y: %d z: %d", accPos[0], accPos[1], accPos[2]);
        eol();
    }
    else {
        terminal.printf("Unable to access MMA7455.");
        eol();
    }
}

/* 
    Samples the accelerometer in 1/4s intervals for the length of "period" (max 5).
    The results are placed in the 3D array. Additionally, the array passed in
    will hold the last reading.
*/
void sampleAcc(int period, int32_t data[3])
{   
    for(int i = 0; i < period*4; i++) {
        //load temps
        acc.read(data[0], data[1], data[2]);
        accData[i][0][0] = data[0]; // x
        accData[0][i][0] = data[1]; // y
        accData[0][0][i] = data[2]; // z
        wait(.25);
    }   
    // if we didn't fill the whole array, we'll clear it to avoid confusion later
    if(period < 5) {
        for(int i = period*4; i < 20; i++)
            accData[i][i][i] = 0;
    }
}

/* 
    -0-90 degrees = 0-65 units (deg/unit ratio)
    -90 - conversion = actual angle (a value of 65 means the device is flat)
     thus, Theta = 90 - (90/65 * x) (90/65 = 1.38)
    -Using ax^2 - by + c allows the data to fit into a custom model of about 1/3 scale
     Therefore, the final equation is: Theta = 90 - 3(ax^2 - by + c) 
*/ 
double getAngle(void)
{
    double deg = 1.38 * accPos[2];
    double cal = 1.38 * accCal[2];
    return ( 90 - (3.0 * ((X2*deg*deg) - (X*cal) + C)) );
}

// called by the refresh timer every 60ms
void LEDPush(void const *args)
{
    ld = 0; cs = 1; // active-low load, "open" lights, "lock" slave
    leds.write(LEDvals);
    ld = 1; cs = 0; // "lock" lights, "open" slave
}

// look-up table based on ranges
// this setup gives a nonlinear response from 0-90 degrees
unsigned char mapLEDS(double v)
{
    int angle = int(v);
    if(accPos[0] < 0) {
        if(angle < 8)
            return 0xFF;
        else if(angle >= 8 && angle < 15)
            return 0x7F;
        else if(angle >= 15 && angle < 25)
            return 0x3F;
        else if(angle >= 25 && angle < 35)
            return 0x1F;
        else if(angle >= 35 && angle < 40)
            return 0x0F;           
        else if(angle >= 40 && angle < 50)
            return 0x07; 
        else if(angle >= 50 && angle < 60)
            return 0x03;
        else if(angle >= 60 && angle < 70)
            return 0x01;
        else if(angle >= 70)
            return 0x00;
        else
            return 0xFF;
    }
    else {
        if(angle < 8)
            return 0xFF;
        else if(angle >= 8 && angle < 15)
            return 0xFE;
        else if(angle >= 15 && angle < 25)
            return 0xFC;
        else if(angle >= 25 && angle < 35)
            return 0xF8;
        else if(angle >= 35 && angle < 40)
            return 0xF0;           
        else if(angle >= 40 && angle < 50)
            return 0xE0; 
        else if(angle >= 50 && angle < 60)
            return 0xC0;
        else if(angle >= 60 && angle < 70)
            return 0x80;
        else if(angle >= 70)
            return 0x00;
        else
            return 0xFF;
    }
}

// "chase" pattern to verify LED array
void testLED(void)
{
    LEDvals = 0x80;
    LEDPush(0);
    wait_ms(75);
    for(int i = 0; i < 8; i++) {
        LEDvals = LEDvals >> 1;;
        LEDPush(0);
        wait_ms(75);
    }
    LEDvals = 0x01;
    LEDPush(0);
    wait_ms(75);
    for(int i = 0; i < 8; i++) {
        LEDvals = LEDvals << 1;;
        LEDPush(0);
        wait_ms(75);
    }
}

// cycles thru various colors to show functionality of an RGB LED
void testRGB(void)
{
    r = 1.0; g = 1.0; b = 1.0;
    RGBPush(); wait_ms(150);
    r = 0.0; g = 1.0; b = 1.0;
    RGBPush(); wait_ms(150);
    r = .6; g = 0.0; b = 0.0;
    RGBPush(); wait_ms(150);
    r = 1.0; g = 1.0; b = 0.0;
    RGBPush(); wait_ms(150);
    r = 1.0; g = 0.0; b = 0.0;
    RGBPush(); wait_ms(150);
    r = 1.0; g = 0.0; b = 1.0;
    RGBPush(); wait_ms(150);
    r = 1.0; g = 1.0; b = 0.0;
    RGBPush(); wait_ms(150);
    r = 0.0; g = 0.0; b = 1.0;
    RGBPush(); wait_ms(150);
    r = 0.0; g = 1.0; b = 1.0;
    RGBPush(); wait_ms(150);
    r = 1.0; g = 1.0; b = 1.0;
    RGBPush(); wait_ms(150);
}

// check if we get the proper reply from the slave.
// this tells us if we've got a good comm channel
int testSlave(void)
{
    int t               = 0;
    unsigned char reply = 0;
    // 3s timeout
    while(t < 300) {
        // keep sending init pattern until we get a reply
        cs = 0;
        reply = leds.write(0xAA);
        cs = 1;
        if(reply == 0x55)
            return 1;
        wait_ms(10);
        t++;
    }
    // timeout occured
    return 0;
}

// assigns the current value to the PWM object
void RGBPush(void)
{
    R = r; 
    G = g; 
    B = b;
}

/*
    Cycle thru RGB values using the status register (sentStatus) and
    request the next value which will be collected one cycle later.
    For example, we request red first but actually get red when we ask for
    green, and green when we ask for blue, and blue when we ask for red.
*/
void RGBupdate(void const *args){
    cs = 0; ld = 1;
    if(!sentStatus[0]) {
        sentStatus[0] = 1;
        Btemp = leds.write(R_CODE);
    }
    else if(!sentStatus[1]) {
        sentStatus[1] = 1;
        Rtemp = leds.write(G_CODE);
    }
    else if(!sentStatus[2]) {
        sentStatus[0] = 0;
        sentStatus[1] = 0;
        Gtemp = leds.write(B_CODE);
        unpackRGB(Rtemp, Gtemp, Btemp);
    } 
    cs = 1; ld = 0;
}

// convert an integer representation of a color to it's float equivalent
void unpackRGB(int tempR, int tempG, int tempB)
{
    r = (float(tempR))/255.0;
    g = (float(tempG))/255.0;
    b = (float(tempB))/255.0;
    RGBPush();
}

// simple SPI error checking 
// both SSEL's shouldn't be active at the same time
void checkSSEL_isr(){
    if((~(ld | cs)) == 0){
        terminal.printf("ERROR: Both SPI Slave Selects are active!");
        eol();
        }
    }