/* SpwfInterface NetworkSocketAPI Example Program
 * Copyright (c) 2015 ARM Limited
 * Copyright (c) 2017 KLIKA TECH, LLC
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 
 Contributors:
  *    Klika Tech - completely adopted to Amazon AWS IoT service

 */

#include "mbed.h"
#include "SpwfInterface.h"
#include "TCPSocket.h"
#include "MQTTClient.h"
#include "MQTTWiFi.h"
#include <ctype.h>
#include "x_nucleo_iks01a1.h"

//------------------------------------
// Hyperterminal default configuration
// 9600 bauds, 8-bit data, no parity
//------------------------------------
Serial pc(SERIAL_TX, SERIAL_RX); 
DigitalOut myled(LED2);
DigitalOut butled(LED3);
InterruptIn  mybutton(USER_BUTTON);

bool myButtonPressed = false;
    
#define MQTT_MAX_PACKET_SIZE 350
#define MQTT_MAX_PAYLOAD_SIZE 300 

#define AWS_IOT_MQTT_HOST              "a3t8vwpkw3sltg.iot.us-east-2.amazonaws.com" //Use your own host.
#define AWS_IOT_MQTT_PORT              8883
#define AWS_IOT_MQTT_CLIENT_ID         "Nucleo" //Should be kept if you are using same device clent.
#define AWS_IOT_MY_THING_NAME          "Nucleo" //Should be kept if you are using same device thing name.
#define AWS_IOT_MQTT_TOPIC_TEST		   "Nucleo/test"
#define AWS_IOT_MQTT_TOPIC_DATA		   "Nucleo/data"
#define AWS_IOT_MQTT_TOPIC_SHADOW	   "$aws/things/Nucleo/shadow/update"
#define AWS_IOT_ID ""
#define AWS_IOT_AUTH_TOKEN ""

// WiFi network credential
#define SSID   ""  // Network must be visible otherwise it can't connect
#define PASSW  ""
#error "Wifi SSID & password empty"

#include "stdint.h"


/**********************************************************************************************
***********************************************************************************************
													Root CA certificate: Never modify
***********************************************************************************************
***********************************************************************************************/

//This root CA can be used.
const uint8_t rootCA[] = "\
-----BEGIN CERTIFICATE-----\n\
MIIE0zCCA7ugAwIBAgIQGNrRniZ96LtKIVjNzGs7SjANBgkqhkiG9w0BAQUFADCB\
yjELMAkGA1UEBhMCVVMxFzAVBgNVBAoTDlZlcmlTaWduLCBJbmMuMR8wHQYDVQQL\
ExZWZXJpU2lnbiBUcnVzdCBOZXR3b3JrMTowOAYDVQQLEzEoYykgMjAwNiBWZXJp\
U2lnbiwgSW5jLiAtIEZvciBhdXRob3JpemVkIHVzZSBvbmx5MUUwQwYDVQQDEzxW\
ZXJpU2lnbiBDbGFzcyAzIFB1YmxpYyBQcmltYXJ5IENlcnRpZmljYXRpb24gQXV0\
aG9yaXR5IC0gRzUwHhcNMDYxMTA4MDAwMDAwWhcNMzYwNzE2MjM1OTU5WjCByjEL\
MAkGA1UEBhMCVVMxFzAVBgNVBAoTDlZlcmlTaWduLCBJbmMuMR8wHQYDVQQLExZW\
ZXJpU2lnbiBUcnVzdCBOZXR3b3JrMTowOAYDVQQLEzEoYykgMjAwNiBWZXJpU2ln\
biwgSW5jLiAtIEZvciBhdXRob3JpemVkIHVzZSBvbmx5MUUwQwYDVQQDEzxWZXJp\
U2lnbiBDbGFzcyAzIFB1YmxpYyBQcmltYXJ5IENlcnRpZmljYXRpb24gQXV0aG9y\
aXR5IC0gRzUwggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQCvJAgIKXo1\
nmAMqudLO07cfLw8RRy7K+D+KQL5VwijZIUVJ/XxrcgxiV0i6CqqpkKzj/i5Vbex\
t0uz/o9+B1fs70PbZmIVYc9gDaTY3vjgw2IIPVQT60nKWVSFJuUrjxuf6/WhkcIz\
SdhDY2pSS9KP6HBRTdGJaXvHcPaz3BJ023tdS1bTlr8Vd6Gw9KIl8q8ckmcY5fQG\
BO+QueQA5N06tRn/Arr0PO7gi+s3i+z016zy9vA9r911kTMZHRxAy3QkGSGT2RT+\
rCpSx4/VBEnkjWNHiDxpg8v+R70rfk/Fla4OndTRQ8Bnc+MUCH7lP59zuDMKz10/\
NIeWiu5T6CUVAgMBAAGjgbIwga8wDwYDVR0TAQH/BAUwAwEB/zAOBgNVHQ8BAf8E\
BAMCAQYwbQYIKwYBBQUHAQwEYTBfoV2gWzBZMFcwVRYJaW1hZ2UvZ2lmMCEwHzAH\
BgUrDgMCGgQUj+XTGoasjY5rw8+AatRIGCx7GS4wJRYjaHR0cDovL2xvZ28udmVy\
aXNpZ24uY29tL3ZzbG9nby5naWYwHQYDVR0OBBYEFH/TZafC3ey78DAJ80M5+gKv\
MzEzMA0GCSqGSIb3DQEBBQUAA4IBAQCTJEowX2LP2BqYLz3q3JktvXf2pXkiOOzE\
p6B4Eq1iDkVwZMXnl2YtmAl+X6/WzChl8gGqCBpH3vn5fJJaCGkgDdk+bW48DW7Y\
5gaRQBi5+MHt39tBquCWIMnNZBU4gcmU7qKEKQsTb47bDN0lAtukixlE0kF6BWlK\
WE9gyn6CagsCqiUXObXbf+eEZSqVir2G3l6BFoMtEMze/aiCKm0oHw0LxOXnGiYZ\
4fQRbxC1lfznQgUy286dUV4otp6F01vvpX1FQHKOtw5rDgb7MzVIcbidJ4vEZV8N\
hnacRHr2lVz2XTIIM6RUthg/aFzyQkqFOFSDX9HoLPKsEdao8WNq\n\
-----END CERTIFICATE-----\n";

/**********************************************************************************************
***********************************************************************************************
													Device Identity Certificates: Modify for your AWS IoT Thing
***********************************************************************************************
***********************************************************************************************/

/****************************************
(somecode)-certificate.pem.crt - Amazon signed PEM sertificate.
*****************************************/

//This Client cert is example. Use own instead.
const uint8_t clientCRT[] = "\
-----BEGIN CERTIFICATE-----\n\
MIIC8jCCAdqgAwIBAgIVAJrIfpHLnCshC2j/Tp0dBJlSgaFnMA0GCSqGSIb3DQEB\
CwUAME0xSzBJBgNVBAsMQkFtYXpvbiBXZWIgU2VydmljZXMgTz1BbWF6b24uY29t\
IEluYy4gTD1TZWF0dGxlIFNUPVdhc2hpbmd0b24gQz1VUzAeFw0xNzA5MTUxMjQ3\
NDBaFw00OTEyMzEyMzU5NTlaMIGAMQswCQYDVQQGEwJERTENMAsGA1UECBMERGVt\
bzENMAsGA1UEBxMERGVtbzENMAsGA1UEChMERGVtbzENMAsGA1UECxMERGVtbzEN\
MAsGA1UEAxMERGVtbzEmMCQGCSqGSIb3DQEJARYXcHNhdnloaW5Aa2xpa2EtdGVj\
aC5jb20wWTATBgcqhkjOPQIBBggqhkjOPQMBBwNCAAT5IihA21BQZFW0vSdVxNuD\
VKXAN7rI3Op3/MiWOlXqHEGHZeYs5ug8qEYkDZDkafhO87LNC0xhNSnGsNnNmyPI\
o2AwXjAfBgNVHSMEGDAWgBRF1n2grhwmYjwSZmF74bVqm/enfjAdBgNVHQ4EFgQU\
qSNNYMI1XGRMnnLenZlU1h/WNAkwDAYDVR0TAQH/BAIwADAOBgNVHQ8BAf8EBAMC\
B4AwDQYJKoZIhvcNAQELBQADggEBADTAhidWjd+MD6sLqr8+ZTdIcka0kT0tnMGy\
Chz5ixaDpNI/OS9fi+SfOAd1Dd+/panpNtvJ5OfN0wkYJRd+lhBaN8M5lWsIF7EM\
FvFtc+UV2cvGyYmSW47fFaV3DOv8vL068cmpNkd/HF8q9r0QNd0h2o97G99Xkk9k\
90DIOgzu0C3sSTy5xDankCvfWIM2ibh5Laz3NmIqVW9jnnkMpQ00xViR8IdnfR4g\
ke1C33ZQh1yTGNEE94nVGRMB2cPY62ChrM/ffgmUo4De0M45tX8ucFVL+ZwaCc3E\
pmjMxFza6yZU50a74zZESmWGR5HYp0PSovglr9Xc5jvktqSugKM=\
\n\
-----END CERTIFICATE-----\n";



/**********************************************************************************************
***********************************************************************************************
													Private Key: Modify for your AWS IoT Thing
***********************************************************************************************
***********************************************************************************************/

/********************************************************************8****************************************
nucleo.key.pem - client key generated according to readme.
**************************************************************************************************************/

//This Client Key is example. Use own instead.
const uint8_t clientKey[] ="\
-----BEGIN EC PARAMETERS-----\n\
BggqhkjOPQMBBw==\
-----END EC PARAMETERS-----\n\
-----BEGIN EC PRIVATE KEY-----\n\
MHcCAQEEIByuPtqukIClJ35+FA0gdvlMs7FmSFiOJGpaYsyQs4wwoAoGCCqGSM49\
AwEHoUQDQgAE+SIoQNtQUGRVtL0nVcTbg1SlwDe6yNzqd/zIljpV6hxBh2XmLObo\
PKhGJA2Q5Gn4TvOyzQtMYTUpxrDZzZsjyA==\
-----END EC PRIVATE KEY-----\n";

int connack_rc = 0; // MQTT connack return code
int connectTimeout = 1000;
int retryAttempt = 0;

PressureSensor *pressure_sensor;
HumiditySensor *humidity_sensor;
TempSensor *temp_sensor1;
MagneticSensor *magnetic_sensor;
GyroSensor     *gyro_sensor;
MotionSensor   *accel_sensor;

MQTT::Message message;
MQTTString TopicName= { AWS_IOT_MQTT_TOPIC_TEST };
MQTT::MessageData MsgData(TopicName, message);

void subscribe_cb(MQTT::MessageData & msgMQTT) {
    char msg[MQTT_MAX_PAYLOAD_SIZE];
    msg[0]='\0';
    strncat (msg, (char*)msgMQTT.message.payload, msgMQTT.message.payloadlen);
    printf ("--->>> subscribe_cb msg: %s\n\r", msg);
}

int subscribe(MQTT::Client<MQTTWiFi, Countdown, MQTT_MAX_PACKET_SIZE>* client, MQTTWiFi* ipstack)
{
    char* pubTopic = AWS_IOT_MQTT_TOPIC_TEST;
    return client->subscribe(pubTopic, MQTT::QOS0, subscribe_cb);
}

int connect(MQTT::Client<MQTTWiFi, Countdown, MQTT_MAX_PACKET_SIZE>* client, MQTTWiFi* ipstack)
{ 
	SpwfSAInterface& WiFi = ipstack->getWiFi();

	// Network debug statements
	LOG("=====================================\n\r");
	LOG("Connecting WiFi.\n\r");
	LOG("Nucleo IP ADDRESS: %s\n\r", WiFi.get_ip_address());
	LOG("Nucleo MAC ADDRESS: %s\n\r", WiFi.get_mac_address());
	LOG("Server Hostname: %s port: %d\n\r", AWS_IOT_MQTT_HOST, AWS_IOT_MQTT_PORT);
	LOG("Client ID: %s\n\r", AWS_IOT_MQTT_CLIENT_ID);
	//LOG("Topic: %s\n\r", AWS_IOT_MQTT_TOPIC_TEST);
	//LOG("Subscription URL: %s\n\r", subscription_url);
	LOG("=====================================\n\r");
    
    ipstack->open(&ipstack->getWiFi());

    int rc=ipstack->getNTPtime();

    if (rc != 0)
	{
    	WARN("Get NTP time error: %d\n", rc);
		return rc;
	}

    rc = WiFi.setSocketClientSecurity((uint8_t *)"m", (uint8_t *)rootCA, (uint8_t *)clientCRT, (uint8_t *)clientKey, (uint8_t *)AWS_IOT_MQTT_HOST, ipstack->getTime());

    if (rc != 0)
	{
		WARN("Set security params error: %d\n", rc);
		return rc;
	}

    rc = ipstack->connect(AWS_IOT_MQTT_HOST, AWS_IOT_MQTT_PORT, connectTimeout);

    if (rc != 0)
    {
    	WARN("IP Stack connect returned: %d\n\r", rc);
        return rc;
    }

    printf ("--->TCP Connected\n\r");

    // MQTT Connect
    MQTTPacket_connectData data = MQTTPacket_connectData_initializer;
    data.MQTTVersion = 4;
    data.struct_version=0;
    data.clientID.cstring = AWS_IOT_MQTT_CLIENT_ID;
    //data.username.cstring = "use-token-auth";
    //data.password.cstring = AWS_IOT_AUTH_TOKEN;

    if ((rc = client->connect(data)) == 0) 
    {
        printf ("--->MQTT Connected\n\r");

        if (!subscribe(client, ipstack)) printf ("--->>>MQTT subscribed to: %s\n\r",AWS_IOT_MQTT_TOPIC_TEST);
    }
    else
    {
        WARN("MQTT connect returned %d\n", rc);        
    }
    if (rc >= 0)
        connack_rc = rc;
    return rc;
}

int getConnTimeout(int attemptNumber)
{  // First 10 attempts try within 3 seconds, next 10 attempts retry after every 1 minute
   // after 20 attempts, retry every 10 minutes
    return (attemptNumber < 10) ? 3 : (attemptNumber < 20) ? 60 : 600;
}

void attemptConnect(MQTT::Client<MQTTWiFi, Countdown, MQTT_MAX_PACKET_SIZE>* client, MQTTWiFi* ipstack)
{
    while (connect(client, ipstack) != MQTT_CONNECTION_ACCEPTED) 
    {    
        if (connack_rc == MQTT_NOT_AUTHORIZED || connack_rc == MQTT_BAD_USERNAME_OR_PASSWORD)
        {
            printf ("File: %s, Line: %d Error: %d\n\r",__FILE__,__LINE__, connack_rc);        
            return; // don't reattempt to connect if credentials are wrong
        } 

        int timeout = getConnTimeout(++retryAttempt);
        WARN("Retry attempt number %d waiting %d\n\r", retryAttempt, timeout);
        
        // if ipstack and client were on the heap we could deconstruct and goto a label where they are constructed
        //  or maybe just add the proper members to do this disconnect and call attemptConnect(...)        
        // this works - reset the system when the retry count gets to a threshold
        if (retryAttempt == 2)
        {
        	ipstack->getWiFi().reset_chip();
            NVIC_SystemReset();
        }
        else
            wait(timeout);
    }
}

int publish(MQTT::Client<MQTTWiFi, Countdown, MQTT_MAX_PACKET_SIZE>* client, MQTTWiFi* ipstack)
{
    MQTT::Message message;
    char* pubTopic = AWS_IOT_MQTT_TOPIC_SHADOW;
            
    char buf[MQTT_MAX_PAYLOAD_SIZE];
    float temp, press, hum;
    int32_t magnet[3];
    int32_t gyro[3];
    int32_t accel[3];

    temp_sensor1->GetTemperature(&temp);
    pressure_sensor->GetPressure(&press);
    humidity_sensor->GetHumidity(&hum);
    magnetic_sensor->Get_M_Axes(magnet);
    gyro_sensor->Get_G_Axes(gyro);
    accel_sensor->Get_X_Axes(accel);
    
    if (!myButtonPressed)
    {
    	butled = 1;
    	sprintf(buf, "{\"state\": {\"reported\": {\"temperature\": %f, \"humidity\": %f, \"pressure\": %f, \"accelerometer\": [%f, %f, %f], \"gyroscope\": [%f, %f, %f], \"magnetometer\": [%f, %f, %f]}}}",
    			temp, hum, press, accel[0]/1000.0, accel[1]/1000.0, accel[2]/1000.0, gyro[0]/1000.0, gyro[1]/1000.0, gyro[2]/1000.0, magnet[0]/10.0, magnet[1]/10.0, magnet[2]/10.0);
    }
    else
    {
    	myButtonPressed = false; // reset state
    	butled = 0;

    	sprintf(buf, "{\"temperature\": %f, \"humidity\": %f, \"pressure\": %f, \"accelerometer\": [%f, %f, %f], \"gyroscope\": [%f, %f, %f], \"magnetometer\": [%f, %f, %f], \"marker\": true}",
    	    			temp, hum, press, accel[0]/1000.0, accel[1]/1000.0, accel[2]/1000.0, gyro[0]/1000.0, gyro[1]/1000.0, gyro[2]/1000.0, magnet[0]/10.0, magnet[1]/10.0, magnet[2]/10.0);
    	pubTopic = AWS_IOT_MQTT_TOPIC_DATA;
    }

    message.qos = MQTT::QOS0;
    message.retained = false;
    message.dup = false;
    message.payload = (void*)buf;
    message.payloadlen = strlen(buf);
    
    printf("Length - %d, Publishing %s\n\r", strlen(buf), buf);

    return client->publish(pubTopic, message);
} 

void pressed()
{
	myButtonPressed = true;
}

int main()
{
    const char * ssid = SSID; // Network must be visible otherwise it can't connect
    const char * seckey = PASSW;

    pc.baud(115200);

    SpwfSAInterface spwf(D8, D2, true);
    
    myled=0;
    DevI2C *i2c = new DevI2C(I2C_SDA, I2C_SCL);
    i2c->frequency(400000);    
    
    mybutton.fall(&pressed);

    X_NUCLEO_IKS01A1 *mems_expansion_board = X_NUCLEO_IKS01A1::Instance(i2c);   
    pressure_sensor = mems_expansion_board->pt_sensor;
    temp_sensor1 =    mems_expansion_board->ht_sensor;
    humidity_sensor = mems_expansion_board->ht_sensor;
    magnetic_sensor = mems_expansion_board->magnetometer;
    gyro_sensor     = mems_expansion_board->GetGyroscope();
    accel_sensor    = mems_expansion_board->GetAccelerometer();
    
    // Due to bug in mbed this workaround is needed to avoid Nucleo hang up when lsm6ds3 is absent
    if (mems_expansion_board->gyro_lsm6ds3 == NULL)
    {    
    	NVIC_DisableIRQ(EXTI4_IRQn);
    	NVIC_ClearPendingIRQ(EXTI4_IRQn);
    }
    
    pc.printf("\r\nX-NUCLEO-IDW01M1 mbed Application\r\n");     
    pc.printf("\r\nconnecting to AP\r\n");            

	MQTTWiFi ipstack(spwf, ssid, seckey, NSAPI_SECURITY_WPA2);

	LOG("Connected to WiFI.\r\n");
	
	spwf.set_debug(false);

	MQTT::Client<MQTTWiFi, Countdown, MQTT_MAX_PACKET_SIZE> client(ipstack, 5000);

	attemptConnect(&client, &ipstack);

	if (connack_rc == MQTT_NOT_AUTHORIZED || connack_rc == MQTT_BAD_USERNAME_OR_PASSWORD)
	{
	  while (true)
	  wait(1.0); // Permanent failures - don't retry
	}

	myled=1;

	int count = 0;

	while (true)
	{
		if (++count == 1)
		{
			myled = 0;
			// Publish a message every second
			if (publish(&client, &ipstack) != 0)
			{
				myled=0;
				ipstack.getWiFi().reset_chip();
				NVIC_SystemReset();
				attemptConnect(&client, &ipstack);   // if we have lost the connection
			}
			else myled=1;

			count = 0;
		}

		client.yield(1000);  // allow the MQTT client to receive messages
	}
}
