/**
 *@section DESCRIPTION
 * mbed NRF2401+  Library
 *@section LICENSE
 * Copyright (c) 2015, Malcolm McCulloch
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 * @file "NRF2401P.h"
 */
 
/** \brief A library to drive the nRF24l01+
* 
*   This library is a simple way to get the nRF24l01+ up and running.
*
*- Use case: For a simple transmitter 
*@code
*int main() {
*
*  long long addr1=0xAB00CD; // setup address - any 5 byte number - same as RX
*  int channel =0x12;  // [0-126] setup channel, must be same as RX
*  bool txOK;
*  char msg[32];
*  char ackData[32];
*  char len;
*
*  // Setup 
*  NRF2401P nrf1(PTD6,PTD7, PTD5,PTD4, PTC12); //mosi, miso, sclk, csn, ce)
*  nrf1.quickTxSetup(channel, addr1); // sets nrf24l01+ as transmitter
*
*  // transmit
*  strcpy (msg, "Hello"); 
*  txOK= nrf1.transmitData(msg,strlen(msg));
*
*  // read ack data if available
*  if (nrf1.isAckData()) { 
*      len= nrf1.getRxData(ackData); // len is number of bytes in ackData
*   }
*}
*@endcode
*- Use case: For a simple receiver 
*@code
*int main(){
*        
*  long long addr1=0xAB00CD; // setup address - any 5 byte number - same as TX
*  int channel =0x12;  // [0-126] setup channel, must be same as TX
*  bool txOK;
*  char msg[32];
*  char ackData[32];
*  char len;
*
*  // Setup 
*  NRF2401P nrf1(PTD6,PTD7, PTD5,PTD4, PTC12); //mosi, miso, sclk, csn, ce)
*  nrf1.quickRxSetup(channel, addr1); // sets nrf24l01+ as  receiver, using pipe 1
*
*  // set ack data
*  sprintf(ackData,"Ack data");
*  nrf1.acknowledgeData(ackData, strlen(ackData),1); // ack for pipe 1
*    
*  // receive
*  while (! nrf1.isRxData()); // note this blocks until RX data
*  len= nrf1.getRxData(msg); // gets the message, len is length of msg
*
*}
*@endcode
*/
class NRF2401P
{
public:
    SPI *spi;
    DigitalOut csn,ce;
    char addressWidth;
    char logMsg[80];
    char status;
    char statusS[32];
    char pipe0Add[5];
    char txAdd[5];
    bool dynamic,debug;
    Serial *pc;

    NRF2401P (PinName mosi, PinName miso, PinName sclk, PinName _csn, PinName _ce);

    char acknowledgeData( char *data, char width, char pipe);
    char checkStatus();
    bool clearStatus();
    bool flushRx();
    bool flushTx();
    char getRxData(char * buffer);
    char getRxWidth();
    bool isAckData();
    bool isRxData();
    void log (char *msg);
    void quickRxSetup(int channel,long long addr);
    void quickTxSetup(int channel,long long addr);
    char readReg(char address, char *data);
    char testReceive();
    char testTransmit();
    bool transmitData( char *data, char width );

    bool setAddressWidth(char width);
    char setChannel(char chan);
    void setDynamicPayload();
    bool setPwrUp();
    char setRadio(char speed,char power);
    char setRxAddress(char *address, char pipe);
    char setRxAddress(long long address, char pipe);
    bool setRxMode();
    char setTxAddress(char *address);
    char setTxAddress(long long address);
    bool setTxMode();
    void setTxRetry(char delay, char numTries);
    void start();
    char * statusString();
    char writeReg(char address, char *data, char width);
    char writeReg(char address, char data);

    void scratch();


};