#include "mbed.h"
#include "sx1276-hal.h"

#define RX_TIMEOUT_VALUE 3500
#define RF_FREQUENCY 868000000

#define LORA_BANDWIDTH                              0         // [0: 125 kHz,
                                                              //  1: 250 kHz,
                                                              //  2: 500 kHz,
                                                              //  3: Reserved]
#define LORA_SPREADING_FACTOR                       9         // [SF7..SF12]
#define LORA_CODINGRATE                             1         // [1: 4/5,
                                                              //  2: 4/6,
                                                              //  3: 4/7,
                                                              //  4: 4/8]
#define LORA_PREAMBLE_LENGTH                        8         // Same for Tx and Rx
#define LORA_SYMBOL_TIMEOUT                         5         // Symbols
#define LORA_FIX_LENGTH_PAYLOAD_ON                  false
#define LORA_FHSS_ENABLED                           false  
#define LORA_NB_SYMB_HOP                            4         // Symbols
#define LORA_IQ_INVERSION_ON                        false
#define LORA_CRC_ENABLED                            true

#define RADIO_STATE_RX 0
#define RADIO_STATE_TX_CW 1

static int RadioState = RADIO_STATE_RX;

static RadioEvents_t RadioEvents;

static struct
{
    uint32_t RfFrequency;
    uint32_t TxOutputPower;
} TxCwParams;

SX1276MB1xAS Radio( NULL );

static char buffer[256];

Serial pc(USBTX, USBRX); // tx, rx

void OnRadioTxTimeout( void )
{
    pc.printf("#TX Timeout\r\n");
    if (RadioState == RADIO_STATE_TX_CW)
    {
        // Restarts continuous wave transmission when timeout expires after 65535 seconds
        Radio.SetTxContinuousWave( TxCwParams.RfFrequency, TxCwParams.TxOutputPower, 65535 );
    }
}

void OnRadioRxTimeout( void )
{
    pc.printf("#RX Timeout\r\n");
    if (RadioState == RADIO_STATE_RX)
    {
        Radio.Rx(RX_TIMEOUT_VALUE);
    }
}

int main( void )
{
    // Radio initialization
    RadioEvents.RxTimeout = OnRadioRxTimeout;
    RadioEvents.TxTimeout = OnRadioTxTimeout;
    Radio.Init( &RadioEvents );

    // verify the connection with the board
    while( Radio.Read( REG_VERSION ) == 0x00  ) {
        debug( "Radio could not be detected!\n\r", NULL );
        wait( 1 );
    }
    
    Radio.SetChannel( RF_FREQUENCY );
    
    Radio.SetRxConfig( MODEM_LORA, LORA_BANDWIDTH, LORA_SPREADING_FACTOR,
                                   LORA_CODINGRATE, 0, LORA_PREAMBLE_LENGTH,
                                   LORA_SYMBOL_TIMEOUT, LORA_FIX_LENGTH_PAYLOAD_ON,
                                   0, LORA_CRC_ENABLED, 0, 0, LORA_IQ_INVERSION_ON, true );

    Radio.Rx(RX_TIMEOUT_VALUE);
    
    pc.printf("#Started\r\n");

    while (1)
    {
        int offset = 0;
        
        while (offset < sizeof(buffer))
        {
            char ch = buffer[offset] = pc.getc();
            if (ch == '\r')
            {
                buffer[offset] = '\0';
                break;
            }
            else if (ch == '\n')
            {
            }
            else
                offset++;
        }
        
        if (strncmp(buffer, ">RX", 3) == 0)
        {
            int RfFreq;
            sscanf(buffer, ">RX %d", &RfFreq);
            
            Radio.SetChannel( RF_FREQUENCY );
            Radio.Rx(RX_TIMEOUT_VALUE);
            
            pc.printf("#Going into RX\r\n");
            
            RadioState = RADIO_STATE_RX;
        }
        else if (strncmp(buffer, ">CW", 3) == 0)
        {
            int RfFreq, TxPower;
            
            sscanf(buffer, ">CW %d, %d", &RfFreq, &TxPower);
            TxCwParams.RfFrequency = RfFreq;
            TxCwParams.TxOutputPower = TxPower;
            
            RadioState = RADIO_STATE_TX_CW;
            
            Radio.SetTxContinuousWave( TxCwParams.RfFrequency, TxCwParams.TxOutputPower, 65535 );
            pc.printf("#Going into TX CW\r\n");
        }
        else if (strncmp(buffer, ">RSSI", 5) == 0)
        {
            int rssi = Radio.GetRssi(MODEM_LORA);
            printf("<%d\r\n", rssi);
            pc.printf("#Measuring RSSI\r\n");
        }
        else
            pc.printf("#Unknown command\r\n");
    }
}
