/**
* @author Satoshi Komorita
*
* @section LICENSE
*
* Copyright (c) 2014 Satoshi Komorita
*
* Permission is hereby granted, free of charge, to any person obtaining a copy
* of this software and associated documentation files (the "Software"), to deal
* in the Software without restriction, including without limitation the rights
* to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
* copies of the Software, and to permit persons to whom the Software is
* furnished to do so, subject to the following conditions:
*
* The above copyright notice and this permission notice shall be included in
* all copies or substantial portions of the Software.
*
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
* IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
* FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
* AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
* LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
* OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
* THE SOFTWARE.
*
* @section DESCRIPTION
*    Simple websocket client
*
*/

#ifndef DXCLIENT_H
#define DXCLIENT_H

#include "mbed.h"
#include "EthernetInterface.h"
#include "Websocket.h"

#define MAX_USERNAME 32
#define MAX_PASSWORD 32
#define MAX_DEVICEID 64

#define MAX_PROPLEN 128
#define MAX_PROPNAMELEN 32
#define MAX_PROPSVALLEN 64

#define MAX_MESSAGELEN 512
#define MAX_MESSAGEIDLEN 32

#define MAX_DEV_DESCRIPTION 64
#define MAX_DEV_NAME 32

typedef enum {
    DX_STRING,
    DX_INTEGER,
    DX_FLOAT,
    DX_BOOLEAN
} dx_prop_type;
typedef enum {
    DX_UPONLY,
    DX_DOWNONLY,
    DX_UPDOWN
} dx_prop_direction;
typedef enum {
    DX_WRITEONLY,
    DX_READONLY,
    DX_READWRITE
} dx_prop_mode;
typedef struct {
    char name[MAX_PROPNAMELEN];
    dx_prop_type type;
    dx_prop_mode mode;
    dx_prop_direction direction;
    char s_val[MAX_PROPSVALLEN];
    float f_val;
    bool b_val;
} dx_prop;

typedef struct {
    int numofprops;
    dx_prop *props;
} dx_props;
 
typedef bool ( *REQUEST_HANDLER )( dx_props *props); 

class DxClient
{
    public:
 
        /**
        * Constructor
        * @param url The DxClient url in the form "ws://ip_domain[:port]/path" (by default: port = 80)
        * @param deviceid is uniqued ID identifing this device.
        */
        DxClient(char * url, char* deviceid, float seed);

        /*
         *  Set Auth User and Password
         *  this method should be called before connect();
         */
        void set_user(char* user, char *pass);

        void set_device_description(char* desc);
        void set_device_name(char* name);
        
        /**
        * Connect to the websocket url
        *@return true if the connection is established, false otherwise
        */
        bool connect();

        /**
        *  close websocket connection
        *@return true if the connection is established, false otherwise
        */
        bool close();
        
        /*
         *  send device_register_requsest message to server
         */
        bool register_device(dx_props* props);

        /*
         *  send device_unregister_requsest message to server
         */
        bool deregister_device();

        /*
         *  send device_update_request to server
         */
        bool update_device(dx_props *props);

        /*
         *  send device_keepalive_request to server
         */
        bool keepalive_device(void);

        /*
         *  handle device_get_request from server
         */
        void set_get_requset_handler(REQUEST_HANDLER handler);

        /*
         *  handle device_get_request from server
         */
        void set_set_requset_handler(REQUEST_HANDLER handler);
        
        
        bool handle_messages(void);

    private:
    
        Websocket m_ws;
        
        char    m_user[MAX_USERNAME];
        char    m_pass[MAX_PASSWORD];
        char    m_deviceid[MAX_DEVICEID];
        char    m_dev_description[MAX_DEV_DESCRIPTION];
        char    m_dev_name[MAX_DEV_NAME];
        
        REQUEST_HANDLER m_func_get_request;
        REQUEST_HANDLER m_func_set_request;
//        bool (*m_func_set_handler)(dx_props *props);
        
        bool dx_user_auth_request();
        bool dx_device_set_response( dx_props *props, const char* mesid);
        bool dx_device_get_response( dx_props *props, const char* mesid);
        bool dx_error_response( const char* mesid);
        
    
};


#endif
