#ifndef LTC294X_H
#define LTC294X_H

#include "mbed.h"

#define LTC_CONSOLE_OUTOUT 1 /* Set this if you need debug messages on the console;    */
                            /* it will have an impact on code-size and power consumption. */

#if LTC_CONSOLE_OUTOUT
//Serial usb(USBTX, USBRX); // tx, rx
extern Serial  debug;
#define DEBUG(...) { debug.printf(__VA_ARGS__); }
#else
#define DEBUG(...) /* nothing */
#endif /* #if NEED_CONSOLE_OUTPUT */

//gets a bit position by shifting 1
#define _BV(x) (1 << x)

// sbi sets a particular bit and cbi clears the bit (target byte-word, bit position)
#define sbi(y,x); y|=_BV(x)
#define cbi(y,x); y&=~_BV(x)


//BIT 7-6 ADC MODE SELECTOR
#define ADC_AUTO    0x00
#define ADC_VOLT    0x01
#define ADC_TEMP    0x02
#define ADC_SLEEP   0x03

//BIT 2-1 AL/CC MODE SELECTOR
#define ALCC_ALERT  0x04
#define ALCC_CCOMP  0x05
#define ALCC_OFF    0x06

//STATUS REGISTER.. READ BIT VALUES ONLY
#define CHIP_ID             _BV(7)
#define RESERVE             _BV(6)
#define CHARGE_OVUV_ALERT   _BV(5)
#define TEMPERATURE_ALERT   _BV(4)
#define CHARGE_HIGH_ALERT   _BV(3)
#define CHARGE_LOW_ALERT    _BV(2)
#define VOLT_ALERT          _BV(1)
#define UVLO_ALERT          _BV(0)


//prescaler options
#define PRESCALE_128               7
#define PRESCALE_64                6
#define PRESCALE_32                5
#define PRESCALE_16                4
#define PRESCALE_8                 3
#define PRESCALE_4                 2
#define PRESCALE_2                 1
#define PRESCALE_1                 0

//CONTROL REGISTER.. BIT LOCATIONS
#define ADC_VOLT_MODE       7
#define ADC_TEMP_MODE       6
#define PRESCALER_2         5
#define PRESCALER_1         4
#define PRESCALER_0         3
#define ALCC_ALERT_MODE     2
#define ALCC_CHRGC_MODE     1
#define IC_SHUTDOWN         0

//I2C REGISTER ADDRESSES
#define REG_STATUS          0x00    //  00h A Status R See Below
#define REG_CONTROL         0x01    //  01h B Control R/W 3Ch
#define ACC_MSB             0x02    //  02h C Accumulated Charge MSB R/W 7Fh
#define ACC_LSB             0x03    //  03h D Accumulated Charge LSB R/W FFh
#define CHRG_THH_MSB        0x04    //  04h E Charge Threshold High MSB R/W FFh
#define CHRG_THH_LSB        0x05    //  05h F Charge Threshold High LSB R/W FFh
#define CHRG_THL_MSB        0x06    //  06h G Charge Threshold Low MSB R/W 00h
#define CHRG_THL_LSB        0x07    //  07h H Charge Threshold Low LSB R/W 00h
#define VOLT_MSB            0x08    //  08h I Voltage MSB R XXh
#define VOLT_LSB            0x09    //  09h J Voltage LSB R XXh
#define VOLT_THH            0x0A    //  0Ah K Voltage Threshold High R/W FFh
#define VOLT_THL            0x0B    //  0Bh L Voltage Threshold Low R/W 00h
#define TEMP_MSB            0x0C    //  0Ch M Temperature MSB R XXh
#define TEMP_LSB            0x0D    //  0Dh N Temperature LSB R XXh
#define TEMPERATURE_THH     0x0E    //  0Eh O Temperature Threshold High R/W FFh
#define TEMPERATURE_THL     0x0F    //  0Fh P Temperature Threshold Low R/W 00h


/** LTC2942 class.
 *  interface LTC2942 coloumb counter on i2c port with interrupt, function name (and battery capacity in Ah Amp-hour).
 *
 * Example:
 * @code
 * #include "mbed.h"
 * #include "LTC2942.h"
 *
 * LTC2942 ltc(PTC9, PTC8, PTA4, &interrDOWN, 0.560);
 * 
 * int main() {
 *     float chLevel = ltc.accumulatedCharge();
 * }
 * @endcode
 */
 
class LTC2942{
public:
    /** create LTC2942 instance based on I2C, WARNING! use setPrescAndBattCap() to set Battery Capacity
     *
     * @param pin sda
     * @param pin scl
     * @param pin al/cc interrupt
     */
    LTC2942(PinName, PinName, PinName, void (*interruptFunc)(void));

    /** create LTC2942 instance based on I2C, 
     *  this will computer internal M prescale value and set internal max charge level
     *
     * @param pin sda
     * @param pin scl
     * @param pin al/cc interrupt
     * @param float battery capacity in Ah (Amp-hour)
     */
    LTC2942(PinName, PinName, PinName, void (*interruptFunc)(void), float);
    
    /** select an ADC measurement option for voltage / temperature
     *
     * @param ADC_x
     */
    void setADCMode(int);

    /** select a prescaler value (not required if object initialized with battery capacity 
     *
     * @param PRESCALE_x
     */
    void setPrescaler(int);

    /** returns prescale value in integer
     *
     * @returns int prescale value
     */
    int getPrescaler();

    /** configure the AL/CC interrupt: TODO works as Alarm only
     *
     * @param ALCC_x
     */
    void configALCC(int);

    /** shutsdown the analog section of LTC2942, also used to read accumulated charge registers
     *
     * @param none
     */
    void shutdown();

    /** wakes the analog section of LTC2942
     *
     * @param none
     */
    void wake();

    /** wakes the analog section of LTC2942
     *
     * @param none
     */
  
    /** amend the battery capacity, also computers prescaler M and adjusts internal Max battery capacity
     *
     * @param battery capacity in Ah (Amp-hour)
     */
    void setPrescAndBattCap(float);

    /** returns the 2 Byte accumulated charge register
     *
     * @return accumulated charge read as 2 byte hex value
     */
    int accumulatedChargeReg();

    /** write into the 2 Byte accumulated charge register
     *
     * @param accumulated charge write as 2 byte hex value
     */
    void accumulatedChargeReg(int);

    /** reads the accumulated charge register as a percent value computed from max battery capacity
     *
     * @return accumulated charge read as percent
     */
    float accumulatedCharge();

    /** writes the accumulated charge register as a percent value computed from max battery capacity
     *  use this to initialise the battery capacity if known, 
     *  or set it to 50% or set it to 100% if battery capacity reaches max
     * 
     *  @param accumulated charge as percent (0-100)
     */
    void accumulatedCharge(float);

    /** set low battery capacity threshhold
     *
     * @param float percent remaining value(0-100)
     */
    void setChargeThresholdLow(float);

    /** set high battery capacity threshhold
     *
     * @param float percent remaining value(0-100)
     */
    void setChargeThresholdHigh(float);    
    
    /** obtains battery voltage at (sense-) pin
     *  if required ADC is set 
     *
     * @return battery level in Volts
     */
    float voltage();

    /** set low battery level threshhold
     *
     * @param Voltage level in Volts
     */
    void setVoltageThresholdLow(float);

    /** set high battery level threshhold
     *
     * @param Voltage level in Volts
     */
    void setVoltageThresholdHigh(float);
    
    /** obtains IC temperature, if required ADC is set 
     *
     * @return temperature in centigrade
     */
    float temperature();

    /** set IC temperature low threshold
     *
     * @param temperauture in centigrade
     */
    void setTemperatureThresholdLow(float);

    /** set IC temperature high threshold
     *
     * @param temperauture in centigrade
     */
    void setTemperatureThresholdHigh(float);
    

    /** reads all LTC2942 register values prints on serial terminal
     *
     * @param none
     */
    void readAll();

    /** reads the status register first
     *  uses the ARA alert response protocol for SMBus to clear AL/CC pin
     *   
     *  int resp = ic.alertResponse();
     *      if((resp & TEMPERATURE_ALERT) == TEMPERATURE_ALERT) {   }
     *
     * @return STATUS register
     */
    int alertResponse();

private:
    char getStatusReg();
    char getControlReg();
    void setControlReg(char);

    I2C _i2c;
    InterruptIn _alcc;
    
    char _addr;
    float _battCap;
    float _battMax;
    float _presc;
};


#endif