/**
* @file MIP8F_SPI.h
* @brief ver4.0 Library header file: Class for JDI MIP8 display
* @details  
* Ver4.0 Addtional function is Line buffer version
* Ver3.0 Addtional function is font display
* ver2.0 Addtional function is Monochome display by 1bit mode of SPI transfer.
*
* <license>
* Copyright 2018 Japan Display Inc.
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*     http://www.apache.org/licenses/LICENSE-2.0
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*/

#include "mbed.h"
/*****************************************
* Select compile option
******************************************/
/**
* @def LINEBUFF_MODE 0
* @brief If you will use a Line Buffer mode,you must define LINEBUFF_MODE 1
*/
#define LINEBUFF_MODE 1
/**
* @def FRAMEBUFF_MODE 1
* @brief If you will use a Frame Buffer mode,you must define LINEBUFF_MODE 1
*/
#define FRAMEBUFF_MODE 0
/*****************************************
******************************************/
#if LINEBUFF_MODE
    //  1dot = 4bit
    #define LINE_SIZE  320  // 640 / 2
#endif
#if FRAMEBUFF_MODE
    //#define FRAME_SIZE 9328  //1flame = 212dot X 88dot  1dot = 4bit
    #define FRAME_SIZE  153600  //  153600 640 x 480        103600  400dot x 200dot    1dot = 4bit
#endif

// RGB color definitions       /*  R, G, B  */
#define Black           0x00    /*  0  0  0  */
#define Blue            0x02    /*  0  0  1  */
#define Green           0x04    /*  0  1  0  */
#define Cyan            0x06    /*  0  1  1  */
#define Red             0x08    /*  1  0  0  */
#define Magenta         0x0a    /*  1  0  1  */ 
#define Yellow          0x0c    /*  1  1  0  */
#define White           0x0e    /*  1  1  1  */

//transfer mode
#define TrBIT4    0x00
#define TrBIT3    0x01
#define TrBIT1    0x02
    
#define RGB8(r,g,b)   (((r & 0x80) >>4) | ((g & 0x80)>>5) | ((b & 0x80)>>6) ) & 0x0E  //24bit->4bit ??6bit(8bit)?? MIP  MASK 0000 1110??

/**
* @brief display class for JDI MIP8(memory in pixel 8 color display)
* @details spi-transfer has 3 mode. 
* 4bit mode is color display, this bit arrange is  R,G,B,x.   R,G,B = R,G,B subpixel bit.  a x bit is Dummy.
* No ues(3bit mode is color display, this bit arrange is  R,G,B.   R,G,B = R,G,B subpixel bit.  No bit is Dummy.)
* 1bit mode is monocrome display,high speed refresh mode. a only Green subpixel of bitmap data is transfered.
*/ 
class memLCD8 : public Stream {
public:

    unsigned char* font;

    /**
    * @brief Constructor : Set MPU pin names
    */
    memLCD8(PinName mosi,PinName miso,PinName sclk,PinName cs,PinName disp,PinName power); 
    void  locate(int x, int y);
    void set_font(unsigned char* f);
    void setWH(int width, int height);
    void SwDisp(bool ONorOFF);
    void foreground(uint8_t colour);
    void background(uint8_t colour);
    void command(char command);
    //ver2.0
    void SetTransfermode(int transfermode);
  
    //ver3.0
    void set_FixedFontWidth( unsigned char width );
    void set_ActualFontWidth(void);
    unsigned char get_Background(void);

#if LINEBUFF_MODE
    //ver3.0
    void  pixel(int x, uint8_t color);
    void  clsLINEBUF(void);
    int*  GetPixelValueFromLineBuffer(int _x,uint8_t* buff);
    void  writeDISPLinebuffer(void);
    void  writeDISP(int line,int transfermode);
#endif

#if FRAMEBUFF_MODE
    void  pixel(int x, int y, uint8_t color);
    void  writeDISP(void);
    void  writeDISP(int transfermode);// transfermode : 4bit,3bit,1bit
    void  writeDISP(int startline , int endline , int transfermode);// refresh display selected line
    void  clsBUF(void);
    void setmarge(bool ifMarge);
    void character(int x, int y, int c);
    void circle(int x0, int y0, int r, uint8_t color);
    void fillcircle(int x0, int y0, int r, uint8_t color);
    void hline(int x0, int x1, int y, uint8_t color);
    void vline(int x, int y0, int y1, uint8_t color);
    void line(int x0, int y0, int x1, int y1, uint8_t color);
    void rect(int x0, int y0, int x1, int y1, uint8_t color);
    void fillrect(int x0, int y0, int x1, int y1, uint8_t color);
    void Symbol(unsigned int x, unsigned int y, unsigned char *symbol);
    //ver2.0
    int* GetPixelValue(int x , int y , uint8_t* buff);
    //ver3.0
    int  textout(int x,int y,char* text);
    void oblique(int x, int y, int c);
    int  obliqueout(int x,int y,char* text);
#endif

protected:
    virtual int _putc(int value);
    virtual int _getc();

    //! SPI class
    SPI _spi;
    //! pin class , SPI line 
    DigitalOut _cs;
    //! display on/off
    DigitalOut _disp;
    //! diplay power on/off
    DigitalOut _power;
        
    //! foreground color of display
    char _foreground;
    //! background color of display
    char _background;
    bool _ifMarge;
    char _if16;

    //ver3.0
    //! Fixed Font width size  if _FixedFontWidth==0, use actual Font width size.
    unsigned char _FixedFontWidth;
    
#if FRAMEBUFF_MODE
    //! frame buffer for display
    uint8_t _dispBUF[FRAME_SIZE];
    //uint8_t _LayerBUF[FRAME_SIZE];
#endif
#if LINEBUFF_MODE
    //uint8_t *_pLineBuf;
    //! line buffer for display
    uint8_t _dispLINEBUF[LINE_SIZE];
#endif

    //! height,diplay pixel size 
    int _height;
    //! width,diplay pixel size 
    int _width;
    unsigned int char_x;
    unsigned int char_y;
    
    //ver2.0
    //! SPI transfer mode command to MIP8 diplay
    char TrModeCommand;
    int TrAdd;
    int TrValue[3];
    int TrValNum;
};
