/* FXAS21002CQ sensor driver
 * Copyright (c) 2015 WD
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#ifndef FXAS21002CQ_H
#define FXAS21002CQ_H

#include "mbed.h"
#include "MotionSensor.h"

// FXAS21002CQ I2C address
#define FXAS21002CQ_SLAVE_ADDR0 (0x20<<1) // with pins SA0=0, 
#define FXAS21002CQ_SLAVE_ADDR1 (0x21<<1) // with pins SA0=1, 
// FXAS21002CQ internal register addresses
#define FXAS21002CQ_STATUS 0x00
#define FXAS21002CQ_OUT_X_MSB 0x01
#define FXAS21002CQ_OUT_X_LSB 0x02
#define FXAS21002CQ_OUT_Y_MSB 0x03
#define FXAS21002CQ_OUT_Y_LSB 0x04
#define FXAS21002CQ_OUT_Z_MSB 0x05
#define FXAS21002CQ_OUT_Z_LSB 0x06
#define FXAS21002CQ_DR_STATUS 0x07
#define FXAS21002CQ_F_STATUS 0x08
#define FXAS21002CQ_F_SETUP 0x09
#define FXAS21002CQ_F_EVENT 0x0A
#define FXAS21002CQ_INT_SRC_FLAG 0x0B
#define FXAS21002CQ_WHO_AM_I 0x0C
#define FXAS21002CQ_CTRL_REG0 0x0D
#define FXAS21002CQ_RT_CFG 0x0E
#define FXAS21002CQ_RT_SRC 0x0F
#define FXAS21002CQ_RT_THS 0x10
#define FXAS21002CQ_RT_COUNT 0x11
#define FXAS21002CQ_TEMP 0x12
#define FXAS21002CQ_CTRL_REG1 0x13
#define FXAS21002CQ_CTRL_REG2 0x14
#define FXAS21002CQ_CTRL_REG3 0x15

enum FXAS21002CQ_F_SETUP_struct_F_MODE {DISABLED_BUFFER=0, CIRCULAR_BUFFER=1, STOP_BUFFER=2};

struct FXAS21002CQ_F_SETUP_struct
{
    enum FXAS21002CQ_F_SETUP_struct_F_MODE F_MODE;  // Bit 7,6
    uint8_t F_WMRK;                                 // Bit 5-0 0- LSB    
};

enum FXAS21002CQ_CTRL_REG0_struct_BW{HIGH_BW=0, MEDIUM_BW=1, LOW_BW=2};
enum FXAS21002CQ_CTRL_REG0_struct_SPIW{SPI4=0, SPI3=1};
enum FXAS21002CQ_CTRL_REG0_struct_SEL{HIGH_HP=0, MEDIUM_HIGH_HP=1, MEDIUM_LOW_HP=2, LOW_HP=3};
enum FXAS21002CQ_CTRL_REG0_struct_HPF_EN{DISABLED_HP=0, ENABLED_HP=1};
enum FXAS21002CQ_CTRL_REG0_struct_FS{RANGE_2000=0, RANGE_1000=1, RANGE_500=2, RANGE_250=3};

struct FXAS21002CQ_CTRL_REG0_struct
{
    enum FXAS21002CQ_CTRL_REG0_struct_BW        BW;         // Bit 7, 6
    enum FXAS21002CQ_CTRL_REG0_struct_SPIW      SPIW;       // Bit 5
    enum FXAS21002CQ_CTRL_REG0_struct_SEL       SEL;        // Bit 4, 3
    enum FXAS21002CQ_CTRL_REG0_struct_HPF_EN    HPF_EN;     // Bit 2
    enum FXAS21002CQ_CTRL_REG0_struct_FS        FS;         // Bit 1, 0
};

enum FXAS21002CQ_RT_CFG_struct_ELE{DISABLED_ELE=0, ENABLED_ELE=1};
enum FXAS21002CQ_RT_CFG_struct_ZTEVE{DISABLED_ZTEVE=0, ENABLED_ZTEVE=1};
enum FXAS21002CQ_RT_CFG_struct_YTEVE{DISABLED_YTEVE=0, ENABLED_YTEVE=1};
enum FXAS21002CQ_RT_CFG_struct_XTEVE{DISABLED_XTEVE=0, ENABLED_XTEVE=1};

struct FXAS21002CQ_RT_CFG_struct
{
    enum FXAS21002CQ_RT_CFG_struct_ELE      ELE;    // Bit 3
    enum FXAS21002CQ_RT_CFG_struct_ZTEVE    ZTEVE;  // Bit 2
    enum FXAS21002CQ_RT_CFG_struct_YTEVE    YTEVE;  // Bit 1
    enum FXAS21002CQ_RT_CFG_struct_XTEVE    XTEVE;  // Bit 2
};

enum FXAS21002CQ_RT_THS_struct_DBCNTM {CLEAR_CNT=0, DECREMENT_CNT=1};

struct FXAS21002CQ_RT_THS_struct
{
    enum FXAS21002CQ_RT_THS_struct_DBCNTM DBCNTM;   // Bit 7
    uint8_t THS;                                    // Bit 6-0
};

struct FXAS21002CQ_RT_COUNT_struct
{
    uint8_t D;                                      // Bit 7-0
};

enum FXAS21002CQ_CTRL_REG1_struct_RST {NOT_RST=0, CHIP_RST=1};
enum FXAS21002CQ_CTRL_REG1_struct_ST {NOT_SELF_TEST=0, SELF_TEST=1};
enum FXAS21002CQ_CTRL_REG1_struct_DR {ODR_800=0, ODR_400=1, ODR_200=2, ODR_100=3, ODR_50=4, ODR_25=5, ODR_12_5=6, ODR_12_5_7=7};
enum FXAS21002CQ_CTRL_REG1_struct_ACTIVE {NOT_ACTIVE_GYRO=0, ACTIVE_GYRO=1};
enum FXAS21002CQ_CTRL_REG1_struct_READY{STANDBY_GYRO=0, READY_GYRO=1};

struct FXAS21002CQ_CTRL_REG1_struct
{
    enum FXAS21002CQ_CTRL_REG1_struct_RST       RST;        // Bit 6
    enum FXAS21002CQ_CTRL_REG1_struct_ST        ST;         // Bit 5
    enum FXAS21002CQ_CTRL_REG1_struct_DR        DR;         // Bit 4-2
    enum FXAS21002CQ_CTRL_REG1_struct_ACTIVE    ACTIVE;     // Bit 1
    enum FXAS21002CQ_CTRL_REG1_struct_READY     READY;      // Bit 0
};

enum FXAS21002CQ_CTRL_REG2_struct_INT_CFG_FIFO{FIFO_TO_INT2=0, FIFO_TO_INT1=1};
enum FXAS21002CQ_CTRL_REG2_struct_INT_EN_FIFO{FIFO_INT_DISABLE=0, FIFO_INT_ENABLE=1};
enum FXAS21002CQ_CTRL_REG2_struct_INT_CFG_RT{RATE_TO_INT2=0, RATE_TO_INT1=1};
enum FXAS21002CQ_CTRL_REG2_struct_INT_EN_RT{RATE_INT_DISABLE=0, RATE_INT_ENABLE=1};
enum FXAS21002CQ_CTRL_REG2_struct_INT_CFG_DRDY{DRDY_TO_INT2=0, DRDY_TO_INT1=1};
enum FXAS21002CQ_CTRL_REG2_struct_INT_EN_DRDY{DRDY_INT_DISABLE=0, DRDY_INT_ENABLE=1};
enum FXAS21002CQ_CTRL_REG2_struct_IPOL{INT_ACT_LOW=0, INT_ACT_HIGH=1};
enum FXAS21002CQ_CTRL_REG2_struct_PP_OD{INT_PUSH_PULL=0, INT_OPEN_DRAIN=1};

struct FXAS21002CQ_CTRL_REG2_struct
{
    enum FXAS21002CQ_CTRL_REG2_struct_INT_CFG_FIFO  INT_CFG_FIFO;   // Bit 7
    enum FXAS21002CQ_CTRL_REG2_struct_INT_EN_FIFO   INT_EN_FIFO;    // Bit 6
    enum FXAS21002CQ_CTRL_REG2_struct_INT_CFG_RT    INT_CFG_RT;     // Bit 5
    enum FXAS21002CQ_CTRL_REG2_struct_INT_EN_RT     INT_EN_RT;      // Bit 4
    enum FXAS21002CQ_CTRL_REG2_struct_INT_CFG_DRDY  INT_CFG_DRDY;   // Bit 3
    enum FXAS21002CQ_CTRL_REG2_struct_INT_EN_DRDY   INT_EN_DRDY;    // Bit 2
    enum FXAS21002CQ_CTRL_REG2_struct_IPOL          IPOL;           // Bit 1
    enum FXAS21002CQ_CTRL_REG2_struct_PP_OD         PP_OD;          // Bit 0
};

enum FXAS21002CQ_CTRL_REG3_struct_WRAPTOONE {WRAP_TO_ZERO=0, WRAP_TO_ONE=1};
enum FXAS21002CQ_CTRL_REG3_struct_EXTCTRLEN {INT2_IS_INTERUPT=0, INT2_IS_POWER_CTRL=1};
enum FXAS21002CQ_CTRL_REG3_struct_FS_DOUBLE {NOT_DOUBLE_RANGE=0, DOUBLE_RANGE=1};

struct FXAS21002CQ_CTRL_REG3_struct
{
    enum FXAS21002CQ_CTRL_REG3_struct_WRAPTOONE WRAPTOONE;  // Bit 3
    enum FXAS21002CQ_CTRL_REG3_struct_EXTCTRLEN EXTCTRLEN;  // Bit 2
    enum FXAS21002CQ_CTRL_REG3_struct_FS_DOUBLE FS_DOUBLE;  // Bit 0
};

struct FXAS21002CQ_Config_struct 
{
    struct FXAS21002CQ_F_SETUP_struct F_SETUP;
    struct FXAS21002CQ_CTRL_REG0_struct CTRL_REG0;
    struct FXAS21002CQ_RT_CFG_struct RT_CFG;
    struct FXAS21002CQ_RT_THS_struct RT_THS;
    struct FXAS21002CQ_RT_COUNT_struct RT_COUNT;
    struct FXAS21002CQ_CTRL_REG1_struct CTRL_REG1;
    struct FXAS21002CQ_CTRL_REG2_struct CTRL_REG2;
    struct FXAS21002CQ_CTRL_REG3_struct CTRL_REG3;
};

extern struct FXAS21002CQ_Config_struct FXAS21002CQ_Config;

/** FXAS21002CQ giro example
    @code
    #include "mbed.h"
    #include "FXAS21002CQ.h"
    I2C i2c(PTE25, PTE24);
    FXAS21002CQ_Giro giro(i2c, FXAS21002CQ_SLAVE_ADDR0);    // Configured for the STTBC-AGM01
    int main(void)
    {
        motion_data_units_t giro_data;
        motion_data_counts_t giro_raw;
        float fgX, fgY, fgZ, tmp_float;
        int16_t rgX, rgY, rgZ, tmp_int;
        giro.enable();
        while (true) {
            // counts based results
            giro.getAxis(giro_raw);
            giro.getX(rgX);
            giro.getY(rgY);
            giro.getZ(rgZ);
            // unit based results
            giro.getAxis(giro_data);
            giro.getX(fgX);
            giro.getY(fgY);
            giro.getZ(fgZ);
            wait(0.1f);
        }
    }
    @endcode
 */

/** FXAS21002CQ driver class
 */
class FXAS21002CQ : public MotionSensor
{
public:

    /** Read a device register
        @param addr The address to read from
        @param data The data to read from it
        @param len The amount of data to read from it
        @return 0 if successful, negative number otherwise
     */
    void readRegs(uint8_t addr, uint8_t *data, uint32_t len) ;

    /** Read the ID from a whoAmI register
        @return The device whoAmI register contents
     */
    virtual uint8_t whoAmI(void) ;

    virtual void enable(void) const;
    virtual void disable(void) const;
    virtual uint32_t sampleRate(uint32_t frequency) const;
    virtual uint32_t dataReady(void) const;

    virtual int16_t getX(int16_t &x) const;
    virtual int16_t getY(int16_t &y) const;
    virtual int16_t getZ(int16_t &z) const;
    virtual float getX(float &x) const;
    virtual float getY(float &y) const;
    virtual float getZ(float &z) const;
    virtual void getAxis(motion_data_counts_t &xyz) const;
    virtual void getAxis(motion_data_units_t &xyz) const;
    
    int8_t getTemperature(void);

    /** FXAS21002CQ constructor
        @param i2c a configured i2c object
        @param addr addr of the I2C peripheral as wired
     */
    FXAS21002CQ(I2C &i2c, const uint8_t addr);
    FXAS21002CQ(I2C &i2c, const uint8_t addr, const struct FXAS21002CQ_Config_struct &FXAS21002CQ_Config);

    /** FXAS21002CQ deconstructor
     */
    ~FXAS21002CQ();
    
protected:
    I2C *_i2c;
    uint8_t _addr;
    
    void writeRegs(uint8_t *data, uint32_t len) const;
    int16_t getSensorAxis(uint8_t addr) const;
};

#endif
