/* 
 * Copyright (c) 2016 Delta Electronics(Japan), Inc
 *
 */
#ifndef US_TICKER_RTC_H
#define US_TICKER_RTC_H

#include <time.h>

#ifdef __cplusplus
extern "C" {
#endif

/** 
 *
 * Provides Real-Time Clock (RTC) functions based on us_ticker.
 * 
 * us_ticker count up 32 bit value in microseconds.
 * So it is overflow in 4294.967295 seconds = 71 minutes.
 * Need to call set_time() before overflow.
 *
 *
 * Example:
 * @code
 * #include "mbed.h"
 * #include "us_ticker_rtc.h"
 *
 * int main() {
 *
 *     //  Attach us_ticker base rtc functions to be used for the C time functions
 *     attach_rtc(us_ticker_rtc_read, us_ticker_rtc_write, us_ticker_rtc_init, us_ticker_rtc_isenabled);
 *
 *     time_t init_seconds = 1256729737;  // Set RTC time to Wed, 28 Oct 2009 11:35:37
 *     set_time(init_seconds);
 *
 *     while(1) {
 *         time_t seconds = time(NULL);
 *         if ((seconds - init_seconds) > 3600)
 *         {
 *             set_time(seconds);
 *             init_seconds = seconds;
 *         }
 *
 *         printf("Time as seconds since January 1, 1970 = %d\n", seconds);
 *
 *         printf("Time as a basic string = %s", ctime(&seconds));
 *
 *         char buffer[32];
 *         strftime(buffer, 32, "%I:%M %p\n", localtime(&seconds));
 *         printf("Time as a custom formatted string = %s", buffer);
 *
 *         wait(1);
 *     }
 * }
 * @endcode
 */

void us_ticker_rtc_init(void);
void us_ticker_rtc_free(void);
int us_ticker_rtc_isenabled(void);

time_t us_ticker_rtc_read(void);
void us_ticker_rtc_write(time_t t);

#ifdef __cplusplus
}
#endif

#endif
