#include "sensor_board.h"

#include <ctime>

bool send_from_digital_sensor = false;
bool send_from_analog_sensor = false;
bool update_analog_sensor = false;

void setSentFromDigitalSensor() {
    send_from_digital_sensor = true;
}

void setSentFromAnalogSensor() {
    send_from_analog_sensor = true;
}

void setUpdateAnalogSensor() {
    update_analog_sensor = true;
}

const int Board::MAX_RETRY = 3;
const float Board::UPDATE_INTERVAL = 0.1;

Board::Board(unsigned long long rx_address, DigitalSensor* digital_sensor,
             AnalogSensor* analog_sensor, unsigned long long tx_address) :
                pc_(USBTX, USBRX), radio_(PB_15, PB_14, PB_13, PB_12, PB_1, PB_2),
                analog_sensor_(analog_sensor), digital_sensor_(digital_sensor),
                analog_ticker_(Ticker()), digital_ticker_(Ticker()), analog_update_(Ticker()) {

    std::srand(std::time(0));
    pc_.baud(115200);
    radio_init(&radio_, rx_address, tx_address);
    displayRadioSetup();
    analog_ticker_.attach(&setSentFromAnalogSensor, analog_sensor->SEND_INTERVAL);
    digital_ticker_.attach(&setSentFromDigitalSensor, digital_sensor->SEND_INTERVAL);
    analog_update_.attach(&setUpdateAnalogSensor, UPDATE_INTERVAL);
}

void Board::updateSensor(Sensor* sensor, bool* should_send_data) {
    if (sensor != NULL) {
        if (update_analog_sensor) {
            sensor->updateState();
            update_analog_sensor = false;
        }
        if (*should_send_data) {
            Data data_to_send = sensor->dataToSend();
            if (send(sensor->getId(), data_to_send)) {
                *should_send_data = false;
                sensor->resetState();
            }
        }
    }
}

void Board::run() {
    while(true) {
        updateSensor(analog_sensor_, &send_from_analog_sensor);
        updateSensor(digital_sensor_, &send_from_digital_sensor);
    }
}

void Board::displayRadioSetup() {
    // Display the (default) setup of the nRF24L01+ chip
    pc_.printf( "nRF24L01+ Frequency    : %d MHz\r\n",  radio_.getRfFrequency() );
    pc_.printf( "nRF24L01+ Output power : %d dBm\r\n",  radio_.getRfOutputPower() );
    pc_.printf( "nRF24L01+ Data Rate    : %d kbps\r\n", radio_.getAirDataRate() );
    pc_.printf( "nRF24L01+ TX Address   : 0x%010llX\r\n", radio_.getTxAddress() );
    pc_.printf( "nRF24L01+ RX0 Address   : 0x%010llX\r\n", radio_.getRxAddress(NRF24L01P_PIPE_P0) );
    pc_.printf( "nRF24L01+ RX1 Address   : 0x%010llX\r\n", radio_.getRxAddress(NRF24L01P_PIPE_P1) );
}

int Board::getRandomWaitMs() {
    return std::rand() % 5000 + 1000;
}

bool Board::sendByRadio(char message[TRANSFER_SIZE]) {
    int retry_counter = 0;
    while (retry_counter < Board::MAX_RETRY) {
        wait_ms(getRandomWaitMs());
        radio_.enablePipeRX(NRF24L01P_PIPE_P0);
        int tx_bytes = radio_.write(NRF24L01P_PIPE_P0, message, TRANSFER_SIZE);
        radio_.disablePipeRX(NRF24L01P_PIPE_P0);
        if (tx_bytes >= 0) {
            pc_.printf("Board retry %d,  radio retry %d\r\n", retry_counter, radio_.getRetrCount());
            return true;
        }
        retry_counter++;
    }
    pc_.printf("sending failed\r\n");
    return false;
}

bool Board::send(int id, Data data) {
    //pc_.printf("value: %f \r\n", data.value.f * 3.3);
    std::string serialized_data = data.serialize();

    std::string output = str_hex(serialized_data.c_str(), serialized_data.size());
    pc_.printf("id %d string data '%s', len %d\r\n", id, output.c_str(), serialized_data.size());

    char message[TRANSFER_SIZE];
    memset(message, 0, sizeof(message));
    memcpy(message, serialized_data.c_str(), serialized_data.length());
    return sendByRadio(message);
}
