#include "mbed.h"
#include "BLEDevice.h"
#include <string>
#include "UARTService.h"
 
#define NEED_CONSOLE_OUTPUT 1 /* Set this if you need debug messages on the console;
                               * it will have an impact on code-size and power consumption. */
Serial pc(USBTX, USBRX);

#if NEED_CONSOLE_OUTPUT
#define DEBUG(...) { pc.printf(__VA_ARGS__); }
#else
#define DEBUG(...) /* nothing */
#endif /* #if NEED_CONSOLE_OUTPUT */
 
BLEDevice  ble;
DigitalOut led1(LED1);
DigitalOut led2(LED2);

DigitalOut powerSwitch(P0_0);
 
AnalogIn tempPin(P0_1);
 
UARTService *uartServicePtr;
 
void disconnectionCallback(Gap::Handle_t handle, Gap::DisconnectionReason_t reason)
{
    DEBUG("Disconnected!\n\r");
    DEBUG("Restarting the advertising process\n\r");
    ble.startAdvertising();
}
 
void onDataWritten(const GattCharacteristicWriteCBParams *params)
{
    if ((uartServicePtr != NULL) && (params->charHandle == uartServicePtr->getTXCharacteristicHandle())) {
        float temp = tempPin.read(); // temperature voltage
        
        float tempC = ((temp*3.0)-0.5)*100.0; // conversion to celcius
        float tempF = (tempC*9.0/5.0)+32.0;                 // conversion to fahrenheit
        DEBUG("temp[V]:%f\ttemp[C]:%f\ttemp[F]:%f\n",temp,tempC,tempF);
        //DEBUG(params->data, "%
        uint16_t bytesRead = params->len;
        DEBUG("received %u bytes\n\r", bytesRead);
        DEBUG("first byte received: %u\n\r", params->data[0]);
        uint8_t on = '1';
        uint8_t off = '0';
        if (params->data[0] == on) {
            led2 = 1;
            powerSwitch = 1;
        } else if (params->data[0] == off) {
            led2 = 0;
            powerSwitch = 0;
        }
        //ble.updateCharacteristicValue(uartServicePtr->getRXCharacteristicHandle(), params->data, bytesRead);
        char buffer[9];
        sprintf(buffer,"%4.4f",tempF);
        ble.updateCharacteristicValue(uartServicePtr->getRXCharacteristicHandle(), (uint8_t *)buffer, sizeof(&buffer));

    }
}
 
void periodicCallback(void)
{
    led1 = !led1;
}
 
int main(void)
{
    led1 = 1;
    led2 = 0;
    powerSwitch = 0;
    Ticker ticker;
    ticker.attach(periodicCallback, 1);
 
    DEBUG("Initialising the nRF51822\n\r");
    ble.init();
    ble.onDisconnection(disconnectionCallback);
    ble.onDataWritten(onDataWritten); // called when someone writes to me
 
    /* setup advertising */
    ble.accumulateAdvertisingPayload(GapAdvertisingData::BREDR_NOT_SUPPORTED);
    ble.setAdvertisingType(GapAdvertisingParams::ADV_CONNECTABLE_UNDIRECTED);
    ble.accumulateAdvertisingPayload(GapAdvertisingData::SHORTENED_LOCAL_NAME,
                                     (const uint8_t *)"pgao", sizeof("pgao") - 1);
    ble.accumulateAdvertisingPayload(GapAdvertisingData::COMPLETE_LIST_128BIT_SERVICE_IDS,
                                     (const uint8_t *)UARTServiceUUID_reversed, sizeof(UARTServiceUUID_reversed));
 
    ble.setAdvertisingInterval(160); /* 100ms; in multiples of 0.625ms. */
    ble.startAdvertising();
 
    UARTService uartService(ble);
    uartServicePtr = &uartService;
 
    while (true) {
        ble.waitForEvent();
    }
}
 