/**
 * Class KeyManager
 *
 * This class tracks the state of the seven input
 * buttons. It exposes (void) callback methods to
 * flag each button as active or not and provides
 * a mapping between key combinations and the
 * alphabet for text input.
 *
 */

#ifndef KEYMANAGER
#define KEYMANAGER

#include <string>
#include <ctype.h>

class KeyManager {
private:
    // key definitions
    bool keyModifier;
    bool keySpace;
    bool key0;
    bool key1;
    bool key2;
    bool key3;
    bool key4;
    
    // static character map and constants
    static char characters[27];
    static char space;
    static char unknown;
        
public:
    
    // constructor
    KeyManager() {
        keyModifier = false;
        keySpace = false;
        key0 = false;
        key1 = false;
        key2 = false;
        key3 = false;
        key4 = false;
    }
    
    // DEBUG: return the status of which keys are pressed
    char* getStatusString() {
        std::string status = "the following keys are pressed:";
        if (key0) {
           status += " 0";
        }
        if (key1) {
           status += " 1";
        }
        if (key2) {
           status += " 2";
        }
        if (key3) {
           status += " 3";
        }
        if (key4) {
           status += " 4";
        }
        status += "\n";
        return (char*)status.c_str(); 
    }
    
    // get the numeric code corresponding to
    // the currently pressed keys.
    int getCode() {
        int result = 0;
        if (key0) {
           result = result | 1;
        }
        if (key1) {
           result = result | 2;
        }
        if (key2) {
           result = result | 4;
        }
        if (key3) {
           result = result | 8;
        }
        if (key4) {
           result = result | 16;
        }
        return result;
    }
    
    // map the current code to its character
    char getCharacter() {
        if (keySpace) {
            return space;
        } else {
            int code = this->getCode();
            if (code < 28 && !keyModifier) {
                return characters[code-1];   
            } else if (code < 28 && keyModifier) {
                return toupper(characters[code-1]);
            } else {
                return unknown;   
            }
        }
    }
    
    // check whether any of the character keys or the space
    // key are pressed
    bool keysPressed() {
        if (key0 || key1 || key2 || key3 || key4 || keySpace) {
            return true;   
        }
        return false;
    }
    
    // Interrupt callback functions
    void keyMOn() { keyModifier = true; }
    void keyMOff() { keyModifier = false; }
    void keySpaceOn() { keySpace = true; }
    void keySpaceOff() { keySpace = false; }
    void key0On() { key0 = true; }
    void key0Off() { key0 = false; }
    void key1On() { key1 = true; }
    void key1Off() { key1 = false; }
    void key2On() { key2 = true; }
    void key2Off() { key2 = false; }
    void key3On() { key3 = true; }
    void key3Off() { key3 = false; }
    void key4On() { key4 = true; }
    void key4Off() { key4 = false; }
};

// define statics
char KeyManager::characters[27] = {'a','b','c','d','e','f','g',
    'h','i','j','k','l','m','n','o','p','q','r','s','t',
    'u','v','w','x','y','z','.' };
char KeyManager::space = ' ';
char KeyManager::unknown = '?';

#endif