/**
 *  \brief CS294-84 Chat Demo \author Ben Zhang, Antonio Iannopollo, Bjoern Hartmann
 *
 * This sampel code illustrates how to connect the mbed KL25Z platform to internet
 * thorugh the CC3000 wifi breakout board (http://www.adafruit.com/product/1469).
 * Tap the slider past the midway point to generate an "ON" message; 
 * release to generate an "OFF" message.
 *
 * Connections between the KL25Z and the CC3000 are made according to the
 * guide at https://learn.adafruit.com/adafruit-cc3000-wifi -- KL25Z and arduino
 * UNO are pin to pin compatible --
 *
 * This application uses the following libraries:
 * - cc3000_hostdriver_mbedsocket
 *   (http://developer.mbed.org/users/Kojto/code/cc3000_hostdriver_mbedsocket/)
 * - HTTPClient (http://developer.mbed.org/users/donatien/code/HTTPClient/)
*  - tsi_sensor
 */

#include "mbed.h"
#include "cc3000.h"
#include "HTTPClient.h"
#include "tsi_sensor.h"


/* TSI slider configuration - This defines will be replaced by PinNames soon */
#if defined (TARGET_KL25Z) || defined (TARGET_KL46Z)
  #define ELEC0 9
  #define ELEC1 10
#elif defined (TARGET_KL05Z)
  #define ELEC0 9
  #define ELEC1 8
#else
  #error TARGET NOT DEFINED
#endif



// KL25Z wifi connection
// we need to define connection pins for:
// - IRQ      => (pin D3)
// - Enable   => (pin D5)
// - SPI CS   => (pin D10)
// - SPI MOSI => (pin D11)
// - SPI MISO => (pin D12)
// - SPI CLK  => (pin D13)
// plus wifi network SSID, password, security level and smart-configuration flag.
mbed_cc3000::cc3000 wifi(D3, D5, D10, SPI(D11, D12, D13),
                         "EECS-PSK", "Thequickbrown", WPA2, false);

// create an http instance
HTTPClient http;

// str is used to hold the response data
char str[512];
// setup the serial connection, and LEDs
Serial pc(USBTX, USBRX);
DigitalOut led_red(LED_RED);
DigitalOut led_green(LED_GREEN);

TSIAnalogSlider tsi(TSI_ELEC0, TSI_ELEC1, 40);
int tsiState=0;

int main()
{
  // by default, it's red
  led_red = 0;
  led_green = 1;

  // print message to indicate the program has started
  pc.printf("CC3000 Chat Program\r\n");
  wifi.init();

  while(1) {
    // continuosly check connection status
    if(wifi.is_connected() == false) {
      // try to connect
      if (wifi.connect() == -1) {
        pc.printf("Failed to connect."
               "Please verify connection details and try again. \r\n");
      } else {
        pc.printf("IP address: %s \r\n", wifi.getIPAddress());

        //once connected, turn green LED on and red LED off
        led_red = 1;
        led_green = 0;
      }
    } else {
      // send a chat message if slider is tapped or released
      float position = tsi.readPercentage();
      if(position>0.5 and tsiState==0) {
        tsiState=1;
        char url[]="http://husk.eecs.berkeley.edu/projects/cc3000/sendchat.php?msg=ON";
        pc.printf(url);
        pc.printf("\r\n");
        int ret = http.get(url, str, 128);
        pc.printf("Status: %d\r\n",ret);
      } else if(position<0.5 and tsiState==1) {
          tsiState=0;
          char url[]="http://husk.eecs.berkeley.edu/projects/cc3000/sendchat.php?msg=OFF";
          pc.printf(url);
          pc.printf("\r\n");
          int ret = http.get(url, str, 128);
          pc.printf("Status: %d\r\n",ret);
      }
    }
  }
}
