/* mbed Microcontroller Library
 * Copyright (c) 2006-2013 ARM Limited
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "mbed.h"
#include "BLEDevice.h"
#include <string.h>

#include "UARTService.h"

#define NEED_CONSOLE_OUTPUT 1 /* Set this if you need debug messages on the console;
                               * it will have an impact on code-size and power consumption. */

#if NEED_CONSOLE_OUTPUT
#define DEBUG(...) { printf(__VA_ARGS__); }
#else
#define DEBUG(...) /* nothing */
#endif /* #if NEED_CONSOLE_OUTPUT */

#define ON 1960
#define OFF 40
#define PULSE_WIDTH 2000
#define OUTPUT_LENGTH 35

BLEDevice  ble;
DigitalOut led1(LED1);
DigitalOut led2(LED2);
DigitalOut pwmOut(P0_30);

AnalogIn ldr0(P0_1);
AnalogIn ldr1(P0_2);
AnalogIn ldr2(P0_3);
AnalogIn ldr3(P0_4);

int threshold = 0;

//Output string to indicate the status of the LDRs
char output[50] = "LDR0 = w x y z";

UARTService *uartServicePtr;
#define TIMER 0
//Creating timer Instance

Ticker ticker;
#if TIMER
long int timerCount = 0;
#endif

void disconnectionCallback(Gap::Handle_t handle, Gap::DisconnectionReason_t reason)
{
    DEBUG("Disconnected!\n\r");
    DEBUG("Restarting the advertising process\n\r");
    ble.startAdvertising();
}

//Thresholding the values of the LDRs
void getValuesofLDR()
{
    if (ldr0.read() < 0.52f)
        output[7] = '0';
    else
        output[7] = '1';
    if (ldr0.read() < 0.52f)
        output[9] = '0';
    else
        output[9] = '1';
    if (ldr0.read() < 0.52f)
        output[11] = '0';
    else
        output[11] = '1';
    if (ldr0.read() < 0.52f)
        output[13] = '0';
    else
        output[13] = '1';
 
}

//Preparation of output string
void onDataWritten(const GattCharacteristicWriteCBParams *params)
{
    char *str = 0;
    int count = 0;

    if ((uartServicePtr != NULL) && (params->charHandle == uartServicePtr->getTXCharacteristicHandle())) {
        uint16_t bytesRead = params->len;
        DEBUG("received %u bytes\n\r", bytesRead);
        
        //Obtaining user command
        str = (char *)malloc((sizeof(char) * bytesRead) + 1);
        while(count <= bytesRead)
        {
            *(str + count) = *(params->data + count);
            count++;
        }
        count--;
        *(str + count) = '\0';
        DEBUG("payload = %s\n\r",str);
        
        //Comparing user input against keywords
        if (strncmp(str,"On",bytesRead) == 0)
        {
            led2 = 0;
            pwmOut = 0;
            bytesRead = 7;
            strncpy((char *)params->data,"Lamp On", bytesRead);
        }    
        else if (strncmp(str,"Off",bytesRead) == 0)
        {
            led2 = 1;
            pwmOut = 1;
            bytesRead = 8;
            strncpy((char *)params->data,"Lamp Off", bytesRead);
        }
        else if (strncmp(str,"Read",bytesRead) == 0)
        {
            bytesRead = 13;
            getValuesofLDR();    
            strncpy((char *)params->data,output,bytesRead);
            
        }
        else//None of the commands match. Invalid entry
        {
            bytesRead = 14;
            strncpy((char *)params->data,"Invalid Entry", bytesRead);
        }    
        free(str);
        ble.updateCharacteristicValue(uartServicePtr->getRXCharacteristicHandle(), params->data, bytesRead);
    }
}

void periodicCallback(void)
{
#if TIMER
    timerCount++;
    if ((timerCount < threshold))
    {
            led1 = 1;
            pwm_out = 1;
    }
    else
    {
            led1 = 0;    
            pwm_out = 0;
    }       
    if (timerCount == PULSE_WIDTH)
        timerCount = 0;
#else
    led1 = !led1;
#endif

}

int main(void)
{
    led1 = 1;
    DEBUG("Initialising the nRF51822\n\r");
    //ticker.attach(periodicCallback, 0.00001f);
    ble.init();
    
    ble.onDisconnection(disconnectionCallback);

    ble.onDataWritten(onDataWritten);
    
    /* setup advertising */
    ble.accumulateAdvertisingPayload(GapAdvertisingData::BREDR_NOT_SUPPORTED);
    ble.setAdvertisingType(GapAdvertisingParams::ADV_CONNECTABLE_UNDIRECTED);
    ble.accumulateAdvertisingPayload(GapAdvertisingData::SHORTENED_LOCAL_NAME,
                                     (const uint8_t *)"AdarshBLE UART", sizeof("AdarshBLE UART") - 1);
    ble.accumulateAdvertisingPayload(GapAdvertisingData::COMPLETE_LIST_128BIT_SERVICE_IDS,
                                     (const uint8_t *)UARTServiceUUID_reversed, sizeof(UARTServiceUUID_reversed));

    ble.setAdvertisingInterval(160); /* 100ms; in multiples of 0.625ms. */
    ble.startAdvertising();

    UARTService uartService(ble);
    uartServicePtr = &uartService;
    
    while (true) {
        ble.waitForEvent();
    }
}
