/** --- Includes --- */
#include "mbed.h"
#include "TimeManager.h"
#include "UartReceiver.h"
#include "CommandParser.h"
#include "global.h"
#include "string.h"

static int sampleHanlder(CommandParser *pC, char *arg, int exarg);
static int softResetHanlder(CommandParser *pC, char *arg, int exarg);
static int setTimeHanlder(CommandParser *pC, char *arg, int exarg);
static int getTimeHanlder(CommandParser *pC, char *arg, int exarg);
static int startSDStore(CommandParser *pC, char *arg, int exarg);
static int endSDStore(CommandParser *pC, char *arg, int exarg);
static int writeSDStore(CommandParser *pC, char *arg, int exarg);

static int configIDHandler(CommandParser *pC, char *arg, int exarg);
static int configWriteHandler(CommandParser *pC, char *arg, int exarg);
static int configReadHandler(CommandParser *pC, char *arg, int exarg);

static int startRunHandler(CommandParser *pC, char *arg, int exarg);
static int stopReqHandler(CommandParser *pC, char *arg, int exarg);
static int checkRunStatHandler(CommandParser *pC, char *arg, int exarg);
static int setRepeatCountHandler(CommandParser *pC, char *arg, int exarg);
static int setRepeatStrideHandler(CommandParser *pC, char *arg, int exarg);
static int setRepeatSingleShotMode(CommandParser *pC, char *arg, int exarg);

/* Event Hankder */
CmdParseRule rules[] = {
    /* -- sample -- */
    {"CMD", sampleHanlder, 0},
    /* Control SD-Card(NOT-SDHC) */
    {"SDS", startSDStore, 0},
    {"SDE", endSDStore, 0},
    {"SDW", writeSDStore, 0}, /* for TEST */
    /* Get Timestamp */
    {"GTS", getTimeHanlder, 0},
    /* Set Timestamp */
    {"TYR", setTimeHanlder, TimeManager::SetTimeMethod::Year},
    {"TMO", setTimeHanlder, TimeManager::SetTimeMethod::Month},
    {"TDA", setTimeHanlder, TimeManager::SetTimeMethod::Day},
    {"THR", setTimeHanlder, TimeManager::SetTimeMethod::Hour},
    {"TMI", setTimeHanlder, TimeManager::SetTimeMethod::Min},
    {"TSE", setTimeHanlder, TimeManager::SetTimeMethod::Sec},
    /* Sensing Control */
    {"CRS", startRunHandler, 0},        /* Run(start) */
    {"CRE", stopReqHandler, 0},         /* Stop -- req */
    {"CPS", checkRunStatHandler, 0},    /* Check ReadyToRun */
    {"RRP", setRepeatCountHandler, 0},  /* set repeat setting => Count */
    {"RMC", setRepeatStrideHandler, 0}, /* set repeat setting => Stride */
    {"RSH", setRepeatSingleShotMode, 0}, /* set repeat setting => SingleShot */
    /* Device Control */
    {"SRS", softResetHanlder, 0},     /* TODO: Software Reset */
    {"CID", configIDHandler, 0},      /* Config ID Access */
    {"CFW", configWriteHandler, 0},   /* Config Write */
    {"CFR", configReadHandler, 0},    /* Config Read */
    };

int getNumOfRules = sizeof(rules)/sizeof(CmdParseRule);

/****************************************************/
/* Event Handlers (Device Control) */
/****************************************************/
static int startRunHandler(CommandParser *pC, char *arg, int exarg)
{
    bool success;
    success = pDevRept->start();
    if (!success) {
        pC->reply(false, CommandParser::NakCode::NAK_IN_RUNNING);
        return 1;
    }
    pC->reply();
    return 0;    
}

static int stopReqHandler(CommandParser *pC, char *arg, int exarg)
{
    bool success;
    success = pDevRept->stop();
    if (!success) {
        pC->reply(false, CommandParser::NakCode::NAK_IN_STOPPED);
        return 1;
    }
    pC->reply();
    return 0;    
}

static int checkRunStatHandler(CommandParser *pC, char *arg, int exarg)
{
    bool success;
    success = pDevRept->readyToStart();
    if (!success) {
        pC->reply(false, CommandParser::NakCode::NAK_IN_RUNNING);
        return 1;
    }
    pC->reply();
    return 0;    
}

static int setRepeatCountHandler(CommandParser *pC, char *arg, int exarg)
{
    bool success;
    int setvalue = atoi(arg);
    success = pDevRept->setRepeatCount(setvalue);
    if (!success) {
        pC->reply(false, CommandParser::NakCode::NAK_IN_RUNNING);
        return 1;
    }
    pC->reply();
    return 0;    
}
static int setRepeatStrideHandler(CommandParser *pC, char *arg, int exarg)
{
    bool success;
    char msflag = arg[3];
    char valsrc[4] = {0};
    int setvalue;
    
    /* this Argment is "xxxs" or "xxxm" */
    memcpy(valsrc, arg, 3);
    setvalue = atoi(valsrc);
    
    if (msflag == 'm') {
        setvalue *= 60; /* setvalue x 60(min to sec)*/
    } else if (msflag == 's') {
        /* NOP setvalue = setvalue */
    } else {
        /* invalid format */
        pC->reply(false, CommandParser::NakCode::NAK_INVAL_PARAM);
        return 1;
    }
    success = pDevRept->setRepeatStride(setvalue);
    if (!success) {
        pC->reply(false, CommandParser::NakCode::NAK_IN_RUNNING);
        return 1;
    }
    pC->reply();
    return 0;    
}

static int setRepeatSingleShotMode(CommandParser *pC, char *arg, int exarg)
{
    bool setMode = true;
    bool success;
    if (strcmp("SET1", arg) == 0) {
        setMode = true;
    } else if (strcmp("CLR0", arg) == 0) {
        setMode = false;
    } else {
        /* INVALID COMMAND */
        pC->reply(false, CommandParser::NakCode::NAK_INVAL_PARAM);
        return 1;        
    }
    success = pDevRept->setRepeatSingleShot(setMode);
    if (success != true) {
        /* INVALID COMMAND in SetMode */
        pC->reply(false, CommandParser::NakCode::NAK_IN_RUNNING);
        return 1;        
    }
    pC->reply();
    return 0;
}

/****************************************************/
/* Event Handlers (Device Control) */
/****************************************************/
static int configIDHandler(CommandParser *pC, char *arg, int exarg)
{
    bool success;
    int setvalue = strtol(arg, NULL, 16);
    success = pDevRept->setConfigId(setvalue);
    if (!success) {
        pC->reply(false,  CommandParser::NakCode::NAK_INVAL_SEQ);
        return 1;
    }
    pC->reply();
    return 0;    
}

static int configWriteHandler(CommandParser *pC, char *arg, int exarg)
{
    bool success;
    int setvalue = strtol(arg, NULL, 16);
    success = pDevRept->setConfigValue(setvalue);
    if (!success) {
        pC->reply(false, CommandParser::NakCode::NAK_INVAL_SEQ);
        return 1;
    }
    pC->reply();
    return 0;
}

static int configReadHandler(CommandParser *pC, char *arg, int exarg)
{
    bool success;
    int getvalue;
    Serial *pUart = pC->getCurrentUart();
    
    success = pDevRept->getConfigValue(&getvalue);
    if (!success) {
        pC->reply(false, CommandParser::NakCode::NAK_INVAL_SEQ);
        return 1;
    }
    uprintf(":%d CFG 0004 %04x\n", pC->getDeviceID(), getvalue);
    return 0;
}

static int softResetHanlder(CommandParser *pC, char *arg, int exarg)
{   
    bool success;
    success = pDevRept->resetAllStatus();
    if (!success) {
        pC->reply(false, 3);
        return 1;
    }
    pC->reply();
    return 0;
}

/****************************************************/
/* Event Handlers */
/****************************************************/
/* Sample Command */
static int sampleHanlder(CommandParser *pC, char *arg, int exarg)
{   
    wait(1);
    NVIC_SystemReset();
    /* Never Reached. */
    pC->reply(false, CommandParser::NakCode::NAK_INVAL_SEQ);
    return 1;
}

/****************************************************/
/* Event Handlers (SD Control) */
/****************************************************/
static int startSDStore(CommandParser *pC, char *arg, int exarg)
{
    if (pSds->startFileWithTimeStamp("","txt") != true) {
        pC->reply(false, CommandParser::NakCode::NAK_INTERNAL_ERR);
        return 1;
    }
    pC->reply();
    return 0;
}

static int endSDStore(CommandParser *pC, char *arg, int exarg)
{
    FILE *fp;
    if ((fp = pSds->getFilePointer()) == NULL) {
        /* NOP */
        pC->reply(false, 2);
        return 2;
    }
    fprintf(fp, "call endSDStore(%s)\n", pSds->getFileName());    
    pSds->syncFile();
    pSds->closeFile();
    pC->reply();
    return 0;
}

static int writeSDStore(CommandParser *pC, char *arg, int exarg)
{
    FILE *fp;
    char curr[TimeManager::TimeStampLength + 1] = {0};
    
    if ((fp = pSds->getFilePointer()) == NULL) {
        /* NOP */
        pC->reply(false, CommandParser::NakCode::NAK_INVAL_SEQ);
        return 2;
    }
    pTM->getTimeStamp(curr);
    fprintf(fp, "call writeSDStore at %s\n", curr);    
    pSds->syncFile();
    pC->reply();
    return 0;
}

/****************************************************/
/* Event Handlers (Time Control) */
/****************************************************/
/* Time Set Command */
static int setTimeHanlder(CommandParser *pC, char *arg, int exarg)
{   
    bool success = false;
    int setvalue = atoi(arg);
    
    switch (exarg) {
    case TimeManager::SetTimeMethod::Year:
    case TimeManager::SetTimeMethod::Month:
    case TimeManager::SetTimeMethod::Day:
    case TimeManager::SetTimeMethod::Hour:
    case TimeManager::SetTimeMethod::Min:
    case TimeManager::SetTimeMethod::Sec:
        success = pTM->setCurrentTime(exarg,setvalue);
        break;
    }
    pC->reply(success, (success)? 0 : CommandParser::NakCode::NAK_INVAL_PARAM );
    return 0;
}

/* Time Get Command */
static int getTimeHanlder(CommandParser *pC, char *arg, int exarg)
{
    int len;
    char timestamp[TimeManager::TimeStampLength + 1] = {0};
    Serial *pUart = pC->getCurrentUart();
    len = pTM->getTimeStamp(timestamp);
    uprintf(":%d RTS %04d %s\n", pC->getDeviceID(), len, timestamp);
    return 0;
}