/* mbed Microcontroller Library
 * Copyright (c) 2006-2013 ARM Limited
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "mbed.h"
#include "BLEDevice.h"
#include "CycleSpeedCadenceService.h"
#include "DeviceInformationService.h"

/* Enable the following if you need to throttle the connection interval. This has
 * the effect of reducing energy consumption after a connection is made;
 * particularly for applications where the central may want a fast connection
 * interval.*/
#define UPDATE_PARAMS_FOR_LONGER_CONNECTION_INTERVAL 0
#define TICK_TIME 1.0/1024.0
#define DEBOUNCE_TIME 50
#define BAUD_RATE 9600

BLEDevice  ble;
DigitalOut led1(LED1);

const static char     DEVICE_NAME[]        = "CSC tracker";
static const uint16_t uuid16_list[]        = {GattService::UUID_CYCLING_SPEED_AND_CADENCE,
                                              GattService::UUID_DEVICE_INFORMATION_SERVICE};
static volatile bool updateWheel = false; 
static volatile bool updateCrank = false;

InterruptIn wheelSwitch(P0_08);
InterruptIn crankSwitch(P0_14);

Timer wheelDebounce;
Timer crankDebounce;


static volatile uint32_t wheelCounter;
static volatile uint16_t wheelEvent;
static volatile uint16_t crankCounter;
static volatile uint16_t crankEvent;
static volatile uint16_t counter;

void disconnectionCallback(Gap::Handle_t handle, Gap::DisconnectionReason_t reason)
{
    ble.startAdvertising(); // restart advertising
}

void periodicCallback(void)
{
    counter++;
}

void wheelSwitchFall() {
    int time = wheelDebounce.read_ms();
    wheelDebounce.stop();
    wheelDebounce.reset();
    if (time>DEBOUNCE_TIME){ //only allow toggle if debounce timer has passed 200ms
        //restart timer when toggle is performed        
        led1= !led1;
        wheelCounter++;
        wheelEvent=counter;
        updateWheel=true;
    }
    
}

void wheelSwitchRise() {    
    wheelDebounce.reset();       //restart timer when toggle is performed    
    wheelDebounce.start();
}

void crankSwithFall() {
    int time = crankDebounce.read_ms();
//    printf("rise: %d \r\n",time);
    crankDebounce.stop();
    crankDebounce.reset();
    if (time>DEBOUNCE_TIME){ //only allow toggle if debounce timer has passed 200ms
//        printf("TICK!!!!!!!!!");        
//        led1= !led1;       //restart timer when toggle is performed
        crankCounter++;
        crankEvent=counter;        
        updateCrank=true;
    }
    
}

void crankSwithRise() {
//    printf("fall: %d \r\n",crankDebounce.read_ms());
    crankDebounce.reset();
    crankDebounce.start();
}



int main(void)
{
    
//    Serial s(P0_5, P0_6); //default for nrf51 is p0.09 p0.11
//    s.baud(BAUD_RATE);
    
//    led1 = 1;
    Ticker ticker;
    ticker.attach(periodicCallback, TICK_TIME); 

    wheelDebounce.start();
    crankDebounce.start();
    
    wheelSwitch.rise(&wheelSwitchRise); 
    wheelSwitch.fall(&wheelSwitchFall);
    
    crankSwitch.rise(&crankSwithRise);
    crankSwitch.fall(&crankSwithFall);
    
    
    ble.init();
    ble.onDisconnection(disconnectionCallback);

    
    
    CycleSpeedCadenceService speedService(ble, wheelCounter,wheelEvent,crankCounter,crankEvent, CycleSpeedCadenceService::LOCATION_REAR_DROPOUT);

    /* Setup auxiliary service. */
    DeviceInformationService deviceInfo(ble, "ARM", "Model1", "SN1", "hw-rev1", "fw-rev1", "soft-rev1");

    /* Setup advertising. */
    ble.accumulateAdvertisingPayload(GapAdvertisingData::BREDR_NOT_SUPPORTED | GapAdvertisingData::LE_GENERAL_DISCOVERABLE);
    ble.accumulateAdvertisingPayload(GapAdvertisingData::COMPLETE_LIST_16BIT_SERVICE_IDS, (uint8_t *)uuid16_list, sizeof(uuid16_list));
    ble.accumulateAdvertisingPayload(GapAdvertisingData::CYCLING_SPEED_AND_CADENCE_SENSOR);
    ble.accumulateAdvertisingPayload(GapAdvertisingData::COMPLETE_LOCAL_NAME, (uint8_t *)DEVICE_NAME, sizeof(DEVICE_NAME));
    ble.setAdvertisingType(GapAdvertisingParams::ADV_CONNECTABLE_UNDIRECTED);
    ble.setAdvertisingInterval(1000);
    ble.startAdvertising();
    
//    printf("starting\n\r");
    // infinite loop
    while (1) {
        // check for trigger from periodicCallback()
        if ((updateWheel || updateCrank) && ble.getGapState().connected) {      
            
            if (updateWheel){
                speedService.updateWheel(wheelCounter,wheelEvent);
                updateWheel=false;
            }
            if (updateCrank){
                speedService.updateCrank(crankCounter,crankEvent);
                updateCrank=false;
            }
            
//            printf("wheel: %d crank: %d\r\n",wheelCounter,crankCounter);
            
        } else {
            ble.waitForEvent(); // low power wait for event
        }
    }
}
