/*
Copyright (c) 2012-2014 RedBearLab

Permission is hereby granted, free of charge, to any person obtaining a copy of this software
and associated documentation files (the "Software"), to deal in the Software without restriction,
including without limitation the rights to use, copy, modify, merge, publish, distribute, sublicense,
and/or sell copies of the Software, and to permit persons to whom the Software is furnished to do so,
subject to the following conditions:
The above copyright notice and this permission notice shall be included in all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED,
INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE
FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

*/

#include "mbed.h"
#include "ble/BLE.h"
#include "GattCallbackParamTypes.h"

#define BLE_UUID_TXRX_SERVICE            0x0000 /**< The UUID of the Nordic UART Service. */
#define BLE_UUID_TX_CHARACTERISTIC       0x0002 /**< The UUID of the TX Characteristic. */
#define BLE_UUIDS_RX_CHARACTERISTIC      0x0003 /**< The UUID of the RX Characteristic. */

#define TXRX_BUF_LEN                     20

#define ON                              1
#define OFF                             0
#define IN                              1
#define OUT                             0
#define PWM                             2
#define NONE                            -1

//static int MODE_MOTE;
//Solo puede estar definido un solo Board
//#define nRF51-DK_board
#define redBearLab_board

#ifdef nRF51-DK_board
//Aqui se define el hardware del nRF51-DK

//Analog in
#define A0                               P0_1   //Analog 0
#define A1                               P0_2   //Analog 1
#define A2                               P0_3   //Analog 2
#define A3                               P0_4   //Analog 3
#define A4                               P0_5   //Analog 4
#define A5                               P0_6   //Analog 5

//Digital

#define D0                               P0_14  //Digital I/O            
#define D1                               P0_12  //Digital I/O
#define D2                               P0_13  //Digital I/O 
#define D3                               P0_15  //Digital I/O
#define D4                               P0_21  //Digital I/O  (LED 1)
#define D5                               P0_22  //Digital I/O  (LED 2)
#define D6                               P0_23  //Digital I/O  (LED 3)
#define D7                               P0_24  //Digital I/O  (LED 4)
#define D8                               P0_17  //Digital I/O  (BUTTON 1) 
#define D9                               P0_19  //Digital I/O  (BUTTON 3)
#define D10                              P0_18  //Digital I/O  (BUTTON 2)
#define D11                              P0_20  //Digital I/O  (BUTTON 4)
#define D12                              P0_7   //Digital I/O
#define D13                              P0_30  //Digital I/O
#define D14                              P0_8   //Digital I/O
#define D15                              P0_16  //Digital I/O

//Aplicacion
#define LED_1                            P0_21  //Digital I/O  (LED 1)
#define LED_2                            P0_22  //Digital I/O  (LED 2)
#define LED_3                            P0_23  //Digital I/O  (LED 3)              
#define LED_4                            P0_24  //Digital I/O  (LED 4)

#define BTN_1                            P0_17  //Digital I/O  (BUTTON 1)
#define BTN_2                            P0_18  //Digital I/O  (BUTTON 2)
#define BTN_3                            P0_19  //Digital I/O  (BUTTON 3)
#define BTN_4                            P0_20  //Digital I/O  (BUTTON 4)

// Aqui se acaba la configuracion del board NRF51-DK

#else
//Aqui se define el hardware del redBearLab
//Analog in
#define A0                               P0_1   //Analog 0
#define A1                               P0_2   //Analog 0
#define A2                               P0_3   //Analog 0
#define A3                               P0_4   //Analog 0
#define A4                               P0_5   //Analog 0
#define A5                               P0_6   //Analog 0

//Digital
#define D0                               P0_11  //Digital I/O             
#define D1                               P0_9   //Digital I/O
#define D2                               P0_10  //Digital I/O
#define D3                               P0_8   //Digital I/O
#define D4                               P0_21  //Digital I/O
#define D5                               P0_23  //Digital I/O (PWM RBL)
#define D6                               P0_16  //Digital I/O (PWM RBL)
#define D7                               P0_17  //Digital I/O  
#define D8                               P0_19  //Digital I/O             
#define D9                               P0_18  //Digital I/O (PWM RBL)
#define D10                              P0_14  //Digital I/O
#define D11                              P0_12  //Digital I/O
#define D12                              P0_13  //Digital I/O (LED RBL)
#define D13                              P0_15  //Digital I/O 
#define D14                              P0_29  //Digital I/O
#define D15                              P0_28  //Digital I/O

//Aplicacion
#define LED_1                            P0_21  //Digital I/O  (LED 1)
#define LED_2                            P0_23  //Digital I/O  (LED 2)
#define LED_3                            P0_16  //Digital I/O  (LED 3)              
#define LED_4                            P0_17  //Digital I/O  (LED 4)

#define BTN_1                            P0_11  //Digital I/O  (BUTTON 1)
#define BTN_2                            P0_9   //Digital I/O  (BUTTON 2)
#define BTN_3                            P0_10  //Digital I/O  (BUTTON 3)
#define BTN_4                            P0_8   //Digital I/O  (BUTTON 4)

#endif


// Aqui se acabo la definicion del board REDBEARLAB

/****  Aqui se definen los pines necesarios independientemente del board seleccionado ******************

                     Instrucciones:
Pines digitales -->
                    ON  --> Usado
                    OFF --> No sera usado
                    IN  --> Entrada
                    OUT --> Salida

Pines Analogicos -->
                    ON  --> Usado
                    OFF --> No sera usado
                    Valor del TED en numero reales, ejemplo 39.1, sino se va a usar se queda con 0
********************************************************************************************************/

#define A0_USO                           OFF     //-> Divisor de voltaje para medir bateria VDD
#define A1_USO                           ON     //-> Temperatura PT1000 (Borneras Shield de pruebas) Borneras
#define A2_USO                           OFF     //-> Temperatura PT1000 (Borneras Shield de pruebas)
#define A3_USO                           OFF    //-> Humedad (Borneras Shield de pruebas)
#define A4_USO                           OFF     //-> Humedad (Borneras Shield de pruebas)
#define A5_USO                           OFF     //-> Conectarlo al LM35

#define A0_TED                           39.10  //Volts,  bateria
#define A1_TED                           39.10   // ºC  temperatura PT 1000
#define A2_TED                           39.10   // ºC  temperatura PT 1000
#define A3_TED                           39.10
#define A4_TED                           39.10
#define A5_TED                           39.10   // ºC  temperatura ambiental con el LM35

#define D0_TYPE                          NONE    // Ejemplo: OUT
#define D1_TYPE                          NONE     // Ejemplo: IN
#define D2_TYPE                          NONE    // Para medir Humedad con Resistencia AC
#define D3_TYPE                          NONE    // Led encienden con 1
#define D4_TYPE                          OUT    // Led encienden con 0
#define D5_TYPE                          NONE    // Led encienden con 0
#define D6_TYPE                          PWM // Led encienden con 1
#define D7_TYPE                          OUT  // Pushbuttons  (Activos en bajo)
#define D8_TYPE                          IN   // Pushbuttons  (Activos en bajo)
#define D9_TYPE                          NONE    // DIP Switch 1 (Activos en bajo)
#define D10_TYPE                         NONE     // DIP Switch 2 (Activos en bajo)
#define D11_TYPE                         NONE     // DIP Switch 3 (Activos en bajo)
#define D12_TYPE                         NONE    // Para medir Humedad con Resistencia AC
#define D13_TYPE                         NONE     // DIP Switch 4 (Activos en bajo)
#define D14_TYPE                         NONE    // Para medir Humedad con Resistencia AC
#define D15_TYPE                         NONE    // Para medir Humedad con Resistencia AC

//Macros de mbed
/*** Aqui Se definen todas macros en funcion de si el bit esta ON u OFF   ************

Estos son las macors ya definidas:
DigitalOut      LED_SET(DIGITAL_OUT_PIN);   //Modo de uso --> LED_SET = 1 o LED_SET = 0;
DigitalIn       BUTTON(DIGITAL_IN_PIN);     //Modo de uso --> if (BUTTON != old_state)
PwmOut          PWM(PWM_PIN);               //Modo de uso -->  PWM = value;
AnalogIn        ANALOGTEMP(ANALOG_IN_PIN);  //Modo de uso -> float s = ANALOG_A0;
AnalogIn        ANALOGBAT(ANALOG_IN_BAT);
Servo           MYSERVO(SERVO_PIN);
*************************************************************************************/

//Macros analogicas
AnalogIn        ANALOG_A0(A0);
AnalogIn        ANALOG_A1(A1);
AnalogIn        ANALOG_A2(A2);
AnalogIn        ANALOG_A3(A3);
AnalogIn        ANALOG_A4(A4);
AnalogIn        ANALOG_A5(A5);   //Temperatura ambiental con LM35

// Indica si alguien esta conectado al dispositivo.
DigitalOut PIN_CONNECT(D13);

#if D2_TYPE == IN
    DigitalIn PIN_D2(D2);
#elif  D2_TYPE == OUT
    DigitalOut PIN_D2(D2);
#else
    int PIN_D2 = 0;
#endif

#if D3_TYPE == IN
    DigitalIn PIN_D3(D3);
#elif D3_TYPE == OUT
    DigitalOut PIN_D3(D3);
#else
    int PIN_D3 = 0;
#endif

#if D4_TYPE == NONE
    static int PIN_D4 = 0;
#elif D4_TYPE == IN
    DigitalIn PIN_D4(D4);
#elif D4_TYPE == OUT
    DigitalOut PIN_D4(D4);
#else
    int PIN_D4 = 0;
#endif

#if D5_TYPE == IN
    DigitalIn PIN_D5(D5);
#elif D5_TYPE == OUT
    DigitalOut PIN_D5(D5);
#elif D5_TYPE == PWM
    PwmOut PIN_D5(D5);
#else
    int PIN_D5 = 0;
#endif

#if D6_TYPE == IN
    DigitalIn PIN_D6(D6);
#elif D6_TYPE == OUT
    DigitalOut PIN_D6(D6);
#elif D6_TYPE == PWM
    PwmOut PIN_D6(D6);
#else
    int PIN_D6 = 0;
#endif

#if D7_TYPE == IN
    DigitalIn PIN_D7(D7);
#elif D7_TYPE == OUT
    DigitalOut PIN_D7(D7);
#else
    int PIN_D7 = 0;
#endif

#if D8_TYPE == IN
    DigitalIn PIN_D8(D8);
#elif D8_TYPE == OUT
    DigitalOut PIN_D8(D8);
#else
    int PIN_D8 = 0;
#endif

#if D9_TYPE == IN
    DigitalIn PIN_D9(D9);
#elif D9_TYPE == OUT
    DigitalOut PIN_D9(D9);
#elif D9_TYPE == PWM
    PwmOut PIN_D9(D9);
#else
    int PIN_D9 = 0;
#endif


#if D10_TYPE == IN
    DigitalIn PIN_D10(D10);
#elif D10_TYPE == OUT
    DigitalOut PIN_D10(D10);
#else
    int PIN_D10 = 0;
#endif

#if D11_TYPE == IN
    DigitalIn PIN_D11(D11);
#elif D11_TYPE == OUT
    DigitalOut PIN_D11(D11);
#else
    int PIN_D11 = 0;
#endif

#if D12_TYPE == IN
    DigitalIn PIN_D12(D12);
#elif D12_TYPE == OUT
    DigitalOut PIN_D12(D12);
#else
    int PIN_D12 = 0;
#endif

#if D13_TYPE == IN
    DigitalIn PIN_D13(D13);
#elif D13_TYPE == OUT
    DigitalOut PIN_D13(D13);
#else
    int PIN_D13 = 0;
#endif

#if D14_TYPE == IN
    DigitalIn PIN_D14(D14);
#elif D14_TYPE == OUT
    DigitalOut PIN_D14(D14);
#else
    int PIN_D14 = 0;
#endif

#if D15_TYPE == IN
    DigitalIn PIN_D15(D15);
#elif D15_TYPE == OUT
    DigitalOut PIN_D15(D15);
#else
    int PIN_D15 = 0;
#endif


//Hasta aqui la definicion de macos
//Digital Inputs --> DATA VALUE
uint16_t DigitalInput_DATA = 0x0000;            //  Mapa de bits con los valores de la entrada digital segun la posicion
uint16_t DigitalOutput_DATA = 0x0000;           //  Mapa de bits con los valores de la entrada digital segun la posicion
uint16_t ESTADO_ENTRADAS_DIGITALES = 0x0000;    //  Mapa de bits de los estados de las entradas digitales
uint16_t ESTADO_SALIDAS_DIGITALES = 0x0000;     //  Mapa de bits de las salidas digitales
uint16_t ESTADO_ENTRADAS_DIGITALES_OLD = 0x0000;//  Almacena temporalmente el estado de las salidas digitales.
uint16_t PAQUETE_ID  = 0;
static bool CONNECTED;
float AnalogInput_Ted [6] = {0.0, 0.0, 0.0, 0.0, 0.0, 0.0};  //Si el TED es cero, es que no está en uso

//Digital input, cantidad, posiciones --> de todos, ted = 0, tipico hasta 16
uint16_t DigitalInput_Pos = 0x0000;    //Mapa de bits, seran usados los bytes del TED 16 bits

//Digital output, cantidad, pocisiones --> de todos, ted = 0,  tipico hasta 16
uint16_t DigitalOutput_Pos = 0x0000;    //Mapa de bits, seran usados los bytes del TED 16 bits

uint16_t DigitalPwm_Pos = 0x0000;       //Mapa de bits, seran usados los bytes del TED 16 bits

//Digital input, cantidad, posiciones --> de todos, ted = 0, tipico hasta 16
uint8_t AnalogInput_Pos = 0x0000;    //Mapa de bits, seran usados los bytes del TED 8 bits

//Digital output, cantidad, pocisiones --> de todos, ted = 0,  tipico hasta 16
uint8_t AnalogOutput_Pos = 0x0000;   //Mapa de bits, seran usados los bytes del TED 8 bits


// Declarando los pines
//static int32_t send_config = 0;

static int8_t SEND_CONFIG_GENERAL = 0;
static int8_t SEND_CONFIG_ANALOG_0, SEND_CONFIG_ANALOG_1, SEND_CONFIG_ANALOG_2, SEND_CONFIG_ANALOG_3, SEND_CONFIG_ANALOG_4, SEND_CONFIG_ANALOG_5;
static bool publish_D0;     //  Indica si es necesario informar al gateway sobre estado de las salidas digitales
static bool publish_D1;     //  Indica si es necesario informar al gateway sobre estado de las entradas digitales

BLE ble;

// Permite imprimir mensajes en la consola
Serial pc(USBTX, USBRX);

//  Para las entradas digitales (Botones)

//  Para las entradas analogicas (Sensores)
static float value_A5, value_A4, value_A3, value_A2, value_A1, value_A0;


// The Nordic UART Service
static const uint8_t uart_base_uuid[] = {0x71, 0x3D, 0, 0, 0x50, 0x3E, 0x4C, 0x75, 0xBA, 0x94, 0x31, 0x48, 0xF1, 0x8D, 0x94, 0x1E};
static const uint8_t uart_tx_uuid[]   = {0x71, 0x3D, 0, 3, 0x50, 0x3E, 0x4C, 0x75, 0xBA, 0x94, 0x31, 0x48, 0xF1, 0x8D, 0x94, 0x1E};
static const uint8_t uart_rx_uuid[]   = {0x71, 0x3D, 0, 2, 0x50, 0x3E, 0x4C, 0x75, 0xBA, 0x94, 0x31, 0x48, 0xF1, 0x8D, 0x94, 0x1E};
static const uint8_t uart_base_uuid_rev[] = {0x1E, 0x94, 0x8D, 0xF1, 0x48, 0x31, 0x94, 0xBA, 0x75, 0x4C, 0x3E, 0x50, 0, 0, 0x3D, 0x71};

// Trama de Configuracion de los Pines (a).
static uint8_t TRAMA_CONFIG_GENERAL[] = {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 , 0x00};        //  Length 12
static uint8_t TRAMA_CONFIG_ANALOG[] = {0x00, 0x00, 0x00, 0x00, 0x00, 0x00};    //  Length 6
static uint8_t LECTURA_DIGITAL[] = {0x00, 0x00, 0x00, 0x00, 0x00, 0x00};        //  Length 6
static uint8_t LECTURA_ANALOGICA[] = {0x00, 0x00, 0x00, 0x00, 0x00, 0x00};      //  Length 5

uint8_t txPayload[TXRX_BUF_LEN] = {0,};
uint8_t rxPayload[TXRX_BUF_LEN] = {0,};

GattCharacteristic  txCharacteristic (uart_tx_uuid, txPayload, 1, TXRX_BUF_LEN, GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_WRITE | GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_WRITE_WITHOUT_RESPONSE);
GattCharacteristic  rxCharacteristic (uart_rx_uuid, rxPayload, 1, TXRX_BUF_LEN, GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_NOTIFY);
GattCharacteristic *uartChars[] = {&txCharacteristic, &rxCharacteristic};
GattService         uartService(uart_base_uuid, uartChars, sizeof(uartChars) / sizeof(GattCharacteristic *));


// https://developer.mbed.org/forum/repo-61676-BLE_GAP_Example-community/topic/17193/
void disconnectionCallback(const Gap::DisconnectionCallbackParams_t *)
{
    BLE& ble = BLE::Instance(BLE::DEFAULT_INSTANCE);
    pc.printf("Disconnected \r\n");
    pc.printf("Restart advertising \r\n");
    ble.startAdvertising();

    SEND_CONFIG_GENERAL = 0;

    //  En caso de no completarse el envio de la configuraciones las detenemos por completo, y dehabilitamos el envio.
    SEND_CONFIG_ANALOG_0 = OFF;
    SEND_CONFIG_ANALOG_1 = OFF;
    SEND_CONFIG_ANALOG_2 = OFF;
    SEND_CONFIG_ANALOG_3 = OFF;
    SEND_CONFIG_ANALOG_4 = OFF;
    SEND_CONFIG_ANALOG_5 = OFF;
    CONNECTED = false;
    PIN_CONNECT = 0;
}

//  Carga la configuración de las entradas digitales del MOTE.
void readDigitalInputs_Value ()
{
    DigitalInput_DATA = (uint16_t) (((D15_TYPE == IN ? 1:0) << 15) | ((D14_TYPE == IN?1:0)<< 14) | ((D13_TYPE == IN ?1:0) << 13) | ((D12_TYPE == IN ?1:0) << 12) | ((D11_TYPE == IN ?1:0) << 11) | ((D10_TYPE == IN? 1:0) << 10) | ((D9_TYPE == IN? 1:0) << 9) | ((D8_TYPE == IN ? 1:0) << 8) | ((D7_TYPE == IN ? 1:0) << 7) | ((D6_TYPE == IN ? 1:0) << 6) |((D5_TYPE == IN ? 1:0) << 5) | ((D4_TYPE == IN ? 1:0) << 4) | ((D3_TYPE == IN ? 1:0) << 3) | ((D2_TYPE == IN ? 1:0) << 2));
}
//  Crea el Mapa de bits de las posiciones ocupadas de los pines digitales de salidas.
void readActuadoresDigitales()
{
    DigitalOutput_DATA = (uint16_t)   ((D15_TYPE == OUT ? 1 : 0) << 15) | ((D14_TYPE == OUT ? 1 : 0) << 14) | ((D13_TYPE == OUT ? 1 : 0) << 13) | ((D12_TYPE == OUT ? 1 : 0) << 12);
    DigitalOutput_DATA |= (uint16_t)  ((D11_TYPE == OUT ? 1 : 0) << 11) | ((D10_TYPE == OUT ? 1 : 0) << 10) | ((D9_TYPE == OUT ? 1 : 0) << 9)   | ((D8_TYPE == OUT ? 1 : 0) << 8);
    DigitalOutput_DATA |= (uint16_t)  ((D7_TYPE == OUT ? 1 : 0) << 7)   | ((D6_TYPE == OUT ? 1 : 0) << 6)   | ((D5_TYPE == OUT ? 1 : 0) << 5) | ((D4_TYPE == OUT ? 1 : 0) << 4);
    DigitalOutput_DATA |= (uint16_t)  ((D3_TYPE == OUT ? 1 : 0) << 3)   | ((D2_TYPE == OUT ? 1 : 0) << 2) ;
}

//  Crea el Mapa de bits de los estados de los pines digitales de entrada.
void cargarEstadoEntradasDigitales()
{
    ESTADO_ENTRADAS_DIGITALES = (uint16_t)   ((D15_TYPE == IN ? PIN_D15 : 0) << 15) | ((D14_TYPE == IN ? PIN_D14 : 0) << 14) | ((D13_TYPE == IN ? PIN_D13 : 0) << 13) | ((D12_TYPE == IN ? PIN_D12 : 0) << 12);
    ESTADO_ENTRADAS_DIGITALES |= (uint16_t)  ((D11_TYPE == IN ? PIN_D11 : 0) << 11) | ((D10_TYPE == IN ? PIN_D10 : 0) << 10) | ((D9_TYPE == IN ? PIN_D9 : 0) << 9)   | ((D8_TYPE == IN ? PIN_D8 : 0) << 8);
    ESTADO_ENTRADAS_DIGITALES |= (uint16_t)  ((D7_TYPE == IN ? PIN_D7 : 0) << 7)   | ((D6_TYPE == IN ? PIN_D6 : 0) << 6)   | ((D5_TYPE == IN ? PIN_D5 : 0) << 5) | ((D4_TYPE == IN ? PIN_D4 : 0) << 4);
    ESTADO_ENTRADAS_DIGITALES |= (uint16_t)  ((D3_TYPE == IN ? PIN_D3 : 0) << 3)   | ((D2_TYPE == IN ? PIN_D2 : 0) << 2) ;
}

//  Crea el Mapa de bits de los estados de los pines digitales de salida.
void cargarEstadosSalidasDigitales()
{
    ESTADO_SALIDAS_DIGITALES = (uint16_t)   ((D15_TYPE == OUT ? PIN_D15 : 0) << 15) | ((D14_TYPE == OUT ? PIN_D14 : 0) << 14) | ((D13_TYPE == OUT ? PIN_D13 : 0) << 13) | ((D12_TYPE == OUT ? PIN_D12 : 0) << 12);
    ESTADO_SALIDAS_DIGITALES |= (uint16_t)  ((D11_TYPE == OUT ? PIN_D11 : 0) << 11) | ((D10_TYPE == OUT ? PIN_D10 : 0) << 10) | ((D9_TYPE == OUT ? PIN_D9 : 0) << 9)   | ((D8_TYPE == OUT ? PIN_D8 : 0) << 8);
    ESTADO_SALIDAS_DIGITALES |= (uint16_t)  ((D7_TYPE == OUT ? PIN_D7 : 0) << 7)   | ((D6_TYPE == OUT ? PIN_D6 : 0) << 6)   | ((D5_TYPE == OUT ? PIN_D5 : 0) << 5) | ((D4_TYPE == OUT ? PIN_D4 : 0) << 4);
    ESTADO_SALIDAS_DIGITALES |= (uint16_t)  ((D3_TYPE == OUT ? PIN_D3 : 0) << 3)   | ((D2_TYPE == OUT ? PIN_D2 : 0) << 2) ;
}

//Funcion para crear los extra bytes
void makeExtraBytes_CONFIG ()
{
    //  Teds de los Analog inputs
    AnalogInput_Ted [0] = A0_TED;
    AnalogInput_Ted [1] = A1_TED;
    AnalogInput_Ted [2] = A2_TED;
    AnalogInput_Ted [3] = A3_TED;
    AnalogInput_Ted [4] = A4_TED;
    AnalogInput_Ted [5] = A5_TED;

    DigitalInput_Pos =   (uint16_t) ((D15_TYPE == IN ? 1:0) << 15) | ((D14_TYPE == IN ? 1:0) << 14) | ((D13_TYPE == IN ? 1:0) << 13)| ((D12_TYPE == IN ? 1:0) << 12);
    DigitalInput_Pos |=  (uint16_t) ((D11_TYPE == IN ? 1:0) << 11) | ((D10_TYPE == IN ? 1:0) << 10) | ((D9_TYPE == IN ? 1:0) << 9)  | ((D8_TYPE == IN ? 1:0) << 8);
    DigitalInput_Pos |=  (uint16_t) ((D7_TYPE == IN ? 1:0) << 7)   | ((D6_TYPE == IN ? 1:0) << 6)   | ((D5_TYPE == IN ? 1:0) << 5)  | ((D4_TYPE == IN ? 1:0) << 4);
    DigitalInput_Pos |=  (uint16_t) ((D3_TYPE == IN ? 1:0) << 3)   | ((D2_TYPE == IN ? 1:0) << 2)   | ((D13_TYPE == IN ? 1:0) << 1) | ((D12_TYPE == IN ? 1:0));

    DigitalOutput_Pos =   (uint16_t) ((D15_TYPE == OUT ? 1:0) << 15) | ((D14_TYPE == OUT ? 1:0) << 14) | ((D13_TYPE == OUT ? 1:0) << 13)| ((D12_TYPE == OUT ? 1:0) << 12);
    DigitalOutput_Pos |=  (uint16_t) ((D11_TYPE == OUT ? 1:0) << 11) | ((D10_TYPE == OUT ? 1:0) << 10) | ((D9_TYPE == OUT ? 1:0) << 9)  | ((D8_TYPE == OUT ? 1:0) << 8);
    DigitalOutput_Pos |=  (uint16_t) ((D7_TYPE == OUT ? 1:0) << 7)   | ((D6_TYPE == OUT ? 1:0) << 6)   | ((D5_TYPE == OUT ? 1:0) << 5)  | ((D4_TYPE == OUT ? 1:0) << 4);
    DigitalOutput_Pos |=  (uint16_t) ((D3_TYPE == OUT ? 1:0) << 3)   | ((D2_TYPE == OUT ? 1:0) << 2)   | ((D13_TYPE == OUT ? 1:0) << 1) | ((D12_TYPE == OUT ? 1:0));

    DigitalPwm_Pos  =   (uint16_t) ((D15_TYPE == PWM ? 1:0) << 15)| ((D14_TYPE == PWM ? 1:0) << 14) | ((D13_TYPE == PWM ? 1:0) << 13)| ((D12_TYPE == PWM ? 1:0) << 12);
    DigitalPwm_Pos |=  (uint16_t) ((D11_TYPE == PWM ? 1:0) << 11) | ((D10_TYPE == PWM ? 1:0) << 10) | ((D9_TYPE == PWM ? 1:0) << 9)  | ((D8_TYPE == PWM ? 1:0) << 8);
    DigitalPwm_Pos |=  (uint16_t) ((D7_TYPE == PWM ? 1:0) << 7)   | ((D6_TYPE == PWM ? 1:0) << 6)   | ((D5_TYPE == PWM ? 1:0) << 5)  | ((D4_TYPE == PWM ? 1:0) << 4);
    DigitalPwm_Pos |=  (uint16_t) ((D3_TYPE == PWM ? 1:0) << 3)   | ((D2_TYPE == PWM ? 1:0) << 2)   | ((D13_TYPE == PWM ? 1:0) << 1) | ((D12_TYPE == PWM ? 1:0));
    
    AnalogInput_Pos = (uint16_t) ((A5_USO << 5) | (A4_USO << 4) | (A3_USO << 3) | (A2_USO << 2) | (A1_USO << 1) | (A0_USO << 0));
    

    //para probar
    pc.printf("DigitalInput_Pos = %x \r\n", DigitalInput_Pos);
    pc.printf("DigitalOutput_Pos = %x \r\n", DigitalOutput_Pos);
    pc.printf("AnalogInput_Pos = %x \r\n", AnalogInput_Pos);
    pc.printf("AnalogOutput_Pos = %x \r\n", DigitalPwm_Pos);
    pc.printf("---------------------------\r\n");

    //  Definimos la trama de configuracion general
    TRAMA_CONFIG_GENERAL [0] = 0xC1;                    //  Codigo de configuracion general
    TRAMA_CONFIG_GENERAL [1] = 0xA1;                    //  Codigo de entradas analogicas
    TRAMA_CONFIG_GENERAL [2] = AnalogInput_Pos;                    //  Valor de las entradas analogicas
    TRAMA_CONFIG_GENERAL [3] = 0xA0;                    //  Codigo de las salidas analogicas
    TRAMA_CONFIG_GENERAL [4] = (DigitalPwm_Pos >> 8);                    //  Valor de las salidas analogicas
    TRAMA_CONFIG_GENERAL [5] = (DigitalPwm_Pos);                    //  Valor de las salidas analogicas
    TRAMA_CONFIG_GENERAL [6] = 0xD1;                    //  Codigo de las entradas digitales
    TRAMA_CONFIG_GENERAL [7] = (DigitalInput_Pos >> 8);
    TRAMA_CONFIG_GENERAL [8] = DigitalInput_Pos;
    TRAMA_CONFIG_GENERAL [9] = 0xD0;                    //  Codigo de las salidas difitales
    TRAMA_CONFIG_GENERAL [10] = (DigitalOutput_Pos>> 8);
    TRAMA_CONFIG_GENERAL [11] = DigitalOutput_Pos;
}



// Ingresa por este metdo unicamente la primera vez que se conecta al mote.
// Tomado desde: https://developer.mbed.org/teams/Bluetooth-Low-Energy/code/BLE_LEDBlinker/file/dc392bde2b3c/main.cpp
void connectionCallback(const Gap::ConnectionCallbackParams_t *)
{
    pc.printf("connectionCallback \r\n");
    PIN_CONNECT = 1;

    makeExtraBytes_CONFIG();
    SEND_CONFIG_GENERAL = 1;
    CONNECTED = true;
}

// Recepta las caracteristicas que se desea escribir en el mote.
void WrittenHandler(const GattWriteCallbackParams *Handler)
{
    //pc.printf("WrittenHandler(const GattWriteCallbackParams *Handler) \r\n");
    uint8_t buf[TXRX_BUF_LEN];
    uint16_t bytesRead, index;

    if (Handler->handle == txCharacteristic.getValueAttribute().getHandle()) {
        ble.readCharacteristicValue(txCharacteristic.getValueAttribute().getHandle(), buf, &bytesRead);
        memset(txPayload, 0, TXRX_BUF_LEN);
        memcpy(txPayload, buf, TXRX_BUF_LEN);

        for(index=0; index<bytesRead; index++)
            pc.putc(buf[index]);

        //pc.printf("Leemos la trama: \r\n");
        //for(index=0; index<bytesRead; index++) {
            //pc.printf("buf[%02x]: %02x\r\n", index, buf[index]);
        //}

        // Desde el telefono desactiva el envio de tramas de configuracion general o de cada sensor.
        if (buf[0] == 0xDC) {              //  Dato Configuracion
            if(buf[1] == 0xC1) {            //  Hace referencia a la configuración general
                SEND_CONFIG_GENERAL = buf[2];      // Debe ser cero, hace que ya no se vuelva a enviar la conf general del mote

                //  Comprueba si estan habilitados las salidas analogicas, y las
                //  inicializa a 1 para que mas adelante envie la configuracion
                //  de cada sensor
                SEND_CONFIG_ANALOG_0 = (A0_USO == ON)?1:0;
                SEND_CONFIG_ANALOG_1 = (A1_USO == ON)?1:0;
                SEND_CONFIG_ANALOG_2 = (A2_USO == ON)?1:0;
                SEND_CONFIG_ANALOG_3 = (A3_USO == ON)?1:0;
                SEND_CONFIG_ANALOG_4 = (A4_USO == ON)?1:0;
                SEND_CONFIG_ANALOG_5 = (A5_USO == ON)?1:0;
                
                publish_D1 = true;     //  bandera a true, para que el Mote deba obligatoriamente enviar al gateway el estado de las entradas digitales.
                publish_D0 = true;     //  bandera a true, para que el Mote deba obligatoriamente enviar al gateway el estado de las entradas digitales.
            }

            //  Maneja las Tramas de confirmacion de la configuracion detallada de cada senssor analogico
            if(buf[1] == 0xC2) {            //  Hace referencia a la configuracion detallada por sensor.
                switch (buf[2]) {           //  Evalua la posición
                    case 0x00:              //  0xA0
                        SEND_CONFIG_ANALOG_0 = buf[3];      //  buf[3] debe ser 0, para detenerse el envió SEND_CONFIG_ANALOG_0
                        break;
                    case 0x01:              //  0xA1
                        SEND_CONFIG_ANALOG_1 = buf[3];      //  buf[3] debe ser 0, para detenerse el envió SEND_CONFIG_ANALOG_1
                        break;
                    case 0x02:              //  0xA2
                        SEND_CONFIG_ANALOG_2 = buf[3];      //  buf[3] debe ser 0, para detenerse el envió SEND_CONFIG_ANALOG_2
                        break;
                    case 0x03:              //  0xA3
                        SEND_CONFIG_ANALOG_3 = buf[3];      //  buf[3] debe ser 0, para detenerse el envió SEND_CONFIG_ANALOG_3
                        break;
                    case 0x04:              //  0xA4
                        SEND_CONFIG_ANALOG_4 = buf[3];      //  buf[3] debe ser 0, para detenerse el envió SEND_CONFIG_ANALOG_4
                        break;
                    case 0x05:              //  0xA5
                        SEND_CONFIG_ANALOG_5 = buf[3];      //  buf[3] debe ser 0, para detenerse el envió SEND_CONFIG_ANALOG_5
                        break;
                }
            }
        }else if (buf[0] == 0xAD) {
            // Desde el gateway asignamos el nuevo valor del paquete id
            //  de esta manera al ser el paquete id = 0, no volvera hacer enviado de nuevo las lecturas de digital out al gateway
            PAQUETE_ID = (((buf[1] << 8) & 0xFF00) | (buf[2] & 0xFF));
        }


        // Verifico si es una trama de Escitura.
        if(buf[0] == 0xEE) {
            PAQUETE_ID = (((buf[1] << 8) & 0xFF00) | (buf[2] & 0xFF));
            //pc.printf("PAQUETE_ID: %d\r\n", PAQUETE_ID);
            // Verifico si es un signal Digital Out
            if(buf[3] == 0xD0) {
                publish_D0 = true;     //  indica que el satado de las salidas digitales deben ser enviadas para informar al gateway el estado de las mismas.
                //  Evaluo sobre que pin se debe actuar.
                switch (buf[4]) {
                    case 0x02:
                        #if D2_TYPE == OUT
                            PIN_D2 = (buf[5] == 0x01) ? 1:0;
                        #endif
                        break;
                    case 0x03:
                        #if D3_TYPE == OUT
                            PIN_D3 = (buf[5] == 0x01) ? 1:0;
                        #endif
                        break;
                    case 0x04:
                        #if D4_TYPE == OUT
                            PIN_D4 = (buf[5] == 0x01) ? 1:0;
                        #endif
                        break;
                    case 0x05:
                        #if D5_TYPE == OUT
                            PIN_D5 = (buf[5] == 0x01) ? 1:0;
                        #endif
                        break;
                    case 0x06:
                        #if D6_TYPE == OUT
                            PIN_D6 = (buf[5] == 0x01) ? 1:0;
                        #endif
                        break;
                    case 0x07:
                        #if D7_TYPE == OUT
                            PIN_D7 = (buf[5] == 0x01) ? 1:0;
                        #endif
                        break;
                    case 0x08:
                        #if D8_TYPE == OUT
                            PIN_D8 = (buf[5] == 0x01) ? 1:0;
                        #endif
                        break;
                    case 0x09:
                        #if D9_TYPE == OUT
                            PIN_D9 = (buf[5] == 0x01) ? 1:0;
                        #endif
                        break;
                    case 0x10:
                        #if D10_TYPE == OUT
                            PIN_D10 = (buf[5] == 0x01) ? 1:0;
                        #endif
                        break;
                    case 0x11:
                        #if D11_TYPE == OUT
                            PIN_D11 = (buf[5] == 0x01) ? 1:0;
                        #endif
                        break;
                    case 0x12:
                        #if D12_TYPE == OUT
                            PIN_D12 = (buf[5] == 0x01) ? 1:0;
                        #endif
                        break;
                    case 0x13:
                        #if D13_TYPE == OUT
                            PIN_D13 = (buf[5] == 0x01) ? 1:0;
                        #endif
                        break;
                    case 0x14:
                        #if D14_TYPE == OUT
                            PIN_D14 = (buf[5] == 0x01) ? 1:0;
                        #endif
                        break;
                    case 0x15:
                        #if D15_TYPE == OUT
                            PIN_D15 = (buf[5] == 0x01) ? 1:0;
                        #endif
                        break;
                }
                // Verifico si es un signal Analog out
            } else if(buf[3] == 0xA0) {
                float value = (float)buf[5]/255;
                bool enviar_trama = false;
                int posicion = 0x00;
                switch (buf[4]) {
                    case 0x05:
                        #if D5_TYPE == PWM
                            PIN_D5 = value;
                            posicion = 0x05;
                            enviar_trama = true;
                        #endif
                        break;
                    case 0x06:
                        #if D6_TYPE == PWM
                            PIN_D6 = value;
                            posicion = 0x06;
                            enviar_trama = true;
                        #endif
                        break;
                    case 0x09:
                        #if D9_TYPE == PWM
                            PIN_D9 = value;
                            posicion = 0x09;
                            enviar_trama = true;
                        #endif
                        break;
                }
                
                if (enviar_trama){
                    LECTURA_DIGITAL[0] = 0xDD;                                      //  Codigo Datos
                    LECTURA_DIGITAL[1] = 0xA0;                                      //  A0  --> Codigo Senial (Analog Output)
                    LECTURA_DIGITAL[2] = posicion;                                      //  Posicion de la interfaz
                    LECTURA_DIGITAL[3] = (int8_t)(PAQUETE_ID >> 8);                 //  primera parte del paquete
                    LECTURA_DIGITAL[4] = (int8_t)PAQUETE_ID ;                       //  segunda parte del paquete
                    LECTURA_DIGITAL[5] = buf[5];                                     //  Valor de las salidas digitales
                    PAQUETE_ID = 0;
                    pc.printf("PWM LECTURA_DIGITAL %x \r\n", LECTURA_DIGITAL);
                    ble.updateCharacteristicValue(rxCharacteristic.getValueAttribute().getHandle(), LECTURA_DIGITAL, 6);    // Para el RTD
                }
            }
        }
    }
}


/*
* Desde este metodo envia las tramas al Gateway.
*/
void m_status_check_handle(void)
{   
    //  Verifica si alguien esta conectado para enviar las tramas a través de Bluetooth.
    if (CONNECTED){
        if (SEND_CONFIG_GENERAL == 1) {
            //  Envia la configuracion Genaral del Mote.
            ble.updateCharacteristicValue(rxCharacteristic.getValueAttribute().getHandle(), TRAMA_CONFIG_GENERAL, 12);
        }
    
    //  Envio de las tramas de Configuracion de los sensores.
        //  Evalua si se encuentra pendiente enviar la configuracion de los sensores.
        if (SEND_CONFIG_ANALOG_0 == ON||SEND_CONFIG_ANALOG_1 == ON||SEND_CONFIG_ANALOG_2 == ON||SEND_CONFIG_ANALOG_3 == ON||SEND_CONFIG_ANALOG_4 == ON||SEND_CONFIG_ANALOG_5 == ON) {
    
            TRAMA_CONFIG_ANALOG[0] = (0xC2);                //  Codigo que indica que la configuracion sera por cada Pin.
            TRAMA_CONFIG_ANALOG[1] = (0x05);                //  Categoria puede ser: A (Actuador) | 5 (Sensor)
            TRAMA_CONFIG_ANALOG[2] = (0xAA);                //  Tipo de Signal AA | DD
    
            if (A0_USO == ON && SEND_CONFIG_ANALOG_0 == ON) {
                TRAMA_CONFIG_ANALOG[3] = 0x00;              //  Posicion que ocupa en el mote
                int16_t TED_PF = (int16_t) (256 * A0_TED);       //  Conversion Punto Fijo
                TRAMA_CONFIG_ANALOG[4] = (int8_t)(TED_PF >> 8);
                TRAMA_CONFIG_ANALOG[5] = (int8_t)(TED_PF);
                pc.printf("SEND_CONFIG_ANALOG_0 \r\n");
                ble.updateCharacteristicValue(rxCharacteristic.getValueAttribute().getHandle(), TRAMA_CONFIG_ANALOG, 6);
    
            } else if (A1_USO == ON && SEND_CONFIG_ANALOG_1 == ON) {
                TRAMA_CONFIG_ANALOG[3] = 0x01;              //  Posicion que ocupa en el mote
                int16_t TED_PF = (int16_t) (256 * A1_TED);       //  Conversion Punto Fijo
                TRAMA_CONFIG_ANALOG[4] = (int8_t)(TED_PF >> 8);
                TRAMA_CONFIG_ANALOG[5] = (int8_t)(TED_PF);
                pc.printf("SEND_CONFIG_ANALOG_1 \r\n");
                ble.updateCharacteristicValue(rxCharacteristic.getValueAttribute().getHandle(), TRAMA_CONFIG_ANALOG, 6);
    
            } else if (A2_USO == ON && SEND_CONFIG_ANALOG_2 == ON) {
                TRAMA_CONFIG_ANALOG[3] = 0x02;              //  Posicion que ocupa en el mote
                int16_t TED_PF = (int16_t) (256 * A2_TED);       //  Conversion Punto Fijo
                TRAMA_CONFIG_ANALOG[4] = (int8_t)(TED_PF >> 8);
                TRAMA_CONFIG_ANALOG[5] = (int8_t)(TED_PF);
                pc.printf("SEND_CONFIG_ANALOG_2 \r\n");
                ble.updateCharacteristicValue(rxCharacteristic.getValueAttribute().getHandle(), TRAMA_CONFIG_ANALOG, 6);
    
            } else if (A3_USO == ON && SEND_CONFIG_ANALOG_3 == ON) {
                TRAMA_CONFIG_ANALOG[3] = 0x03;              //  Posicion que ocupa en el mote
                int16_t TED_PF = (int16_t) (256 * A3_TED);       //  Conversion Punto Fijo
                TRAMA_CONFIG_ANALOG[4] = (int8_t)(TED_PF >> 8);
                TRAMA_CONFIG_ANALOG[5] = (int8_t)(TED_PF);
                pc.printf("SEND_CONFIG_ANALOG_3 \r\n");
                ble.updateCharacteristicValue(rxCharacteristic.getValueAttribute().getHandle(), TRAMA_CONFIG_ANALOG, 6);
    
            } else if (A4_USO == ON && SEND_CONFIG_ANALOG_4 == ON) {
                TRAMA_CONFIG_ANALOG[3] = 0x04;              //  Posicion que ocupa en el mote
                int16_t TED_PF = (int16_t) (256 * A4_TED);       //  Conversion Punto Fijo
                TRAMA_CONFIG_ANALOG[4] = (int8_t)(TED_PF >> 8);
                TRAMA_CONFIG_ANALOG[5] = (int8_t)(TED_PF);
                pc.printf("SEND_CONFIG_ANALOG_4 \r\n");
                ble.updateCharacteristicValue(rxCharacteristic.getValueAttribute().getHandle(), TRAMA_CONFIG_ANALOG, 6);
    
            } else if (A5_USO == ON && SEND_CONFIG_ANALOG_5 == ON) {
                TRAMA_CONFIG_ANALOG[3] = 0x05;              //  Posicion que ocupa en el mote
                int16_t TED_PF = (int16_t) (256 * A5_TED);  //  Conversion Punto Fijo
                TRAMA_CONFIG_ANALOG[4] = (int8_t)(TED_PF >> 8);
                TRAMA_CONFIG_ANALOG[5] = (int8_t)(TED_PF);
                pc.printf("SEND_CONFIG_ANALOG_5 %x , %x \r\n", TRAMA_CONFIG_ANALOG[4], TRAMA_CONFIG_ANALOG[5]);
                //pc.printf("SEND_CONFIG_ANALOG_6 %x, %x, %x, %x, %x, %x, %x \r\n", TRAMA_CONFIG_ANALOG[0], TRAMA_CONFIG_ANALOG[1], TRAMA_CONFIG_ANALOG[2], TRAMA_CONFIG_ANALOG[3], TRAMA_CONFIG_ANALOG[4], TRAMA_CONFIG_ANALOG[5], TRAMA_CONFIG_ANALOG[6] );
                ble.updateCharacteristicValue(rxCharacteristic.getValueAttribute().getHandle(), TRAMA_CONFIG_ANALOG, 6);
            }
        }
        //  Envio de las tramas de lecturas del Mote.
        cargarEstadoEntradasDigitales();                      //  Leemos los estados digitales.
        //pc.printf("ESTADO_ENTRADAS_DIGITALES_OLD %x   -  ESTADO_ENTRADAS_DIGITALES %x \r\n", ESTADO_ENTRADAS_DIGITALES_OLD, ESTADO_ENTRADAS_DIGITALES);
        if (ESTADO_ENTRADAS_DIGITALES_OLD != ESTADO_ENTRADAS_DIGITALES || publish_D1) {
            if (publish_D1){
                publish_D1 = false;
            }
            ESTADO_ENTRADAS_DIGITALES_OLD = ESTADO_ENTRADAS_DIGITALES;
    
            LECTURA_DIGITAL[0] = 0xDD;                    //  Codigo de lectura de datos
            LECTURA_DIGITAL[1] = 0xD1;                      //  Digital Input
            LECTURA_DIGITAL[2] = (int8_t)(ESTADO_ENTRADAS_DIGITALES >> 8);         //  valores de las interfaces digitales
            LECTURA_DIGITAL[3] = (int8_t)(ESTADO_ENTRADAS_DIGITALES);         //  valores de las interfaces digitales
            pc.printf("Envio LECTURA_DIGITAL %x \r\n", ESTADO_ENTRADAS_DIGITALES);  // Imprimo en terminal lo que esta enviando desde el mote.
    
            ble.updateCharacteristicValue(rxCharacteristic.getValueAttribute().getHandle(), LECTURA_DIGITAL, 4);    // Para el RTD
            
        }else if (PAQUETE_ID != 0 || publish_D0) {
            if (publish_D0){
                publish_D0 = false;
            }
            //  Lecturas digitales
            cargarEstadosSalidasDigitales();
    
            pc.printf("ESTADO_SALIDAS_DIGITALES: %x \r\n", ESTADO_SALIDAS_DIGITALES);
            LECTURA_DIGITAL[0] = 0xDD;                                      //  Codigo
            LECTURA_DIGITAL[1] = 0xD0;                                      //  D0  --> Digital Output
            LECTURA_DIGITAL[2] = (int8_t)(PAQUETE_ID >> 8);                         //  primera parte del paquete
            LECTURA_DIGITAL[3] = (int8_t)PAQUETE_ID ;                               //  segunda parte del paquete
            LECTURA_DIGITAL[4] = (int8_t)(ESTADO_SALIDAS_DIGITALES >> 8);                  //  Valor de las salidas digitales
            LECTURA_DIGITAL[5] = (int8_t)(ESTADO_SALIDAS_DIGITALES);                       //  Valor de las salidas digitales
    
            ble.updateCharacteristicValue(rxCharacteristic.getValueAttribute().getHandle(), LECTURA_DIGITAL, 6);    // Para el RTD
        }
    
    //  Envio de las tramas de las lecturas de los sensores habilitados.
        if (SEND_CONFIG_GENERAL == 0) {
    
            LECTURA_ANALOGICA[0] = 0xDD;        //  Codigo Datos
            LECTURA_ANALOGICA[1] = 0xA1;        //  --> Analog Input (A1)
                    
    
            //  Si la configuracion ha sido enviada, envio las lecturas. Esta para cumplir con una secuencia.
            if (A5_USO == ON) {
                float s = ANALOG_A5;
                if (value_A5 != s) {
                    value_A5 = s;
    
                    uint16_t value = s * 1024;
    
                    float Vtemp = value * 0.0032080078125;
                    float Rtemp = (15111 / Vtemp) - 4630;
                    float Tempfinal = ((Rtemp - 1000) / 3.850);
    
                    //2- Conversion flotante a punto fijo
                    int16_t Temperature_PF = (int16_t) (256 * Tempfinal);
    
                    LECTURA_ANALOGICA[2] = 0x05;        //  --> Posicion del sensor
                    LECTURA_ANALOGICA[3] = (int8_t)(Temperature_PF >> 8);
                    LECTURA_ANALOGICA[4] = (int8_t)(Temperature_PF);
    
                    pc.printf("ANALOG_A5 (Format HEX): %x , LECTURA_ANALOGICA: %x \r\n", Temperature_PF, LECTURA_ANALOGICA);
                    ble.updateCharacteristicValue(rxCharacteristic.getValueAttribute().getHandle(), LECTURA_ANALOGICA, 5);    // Para el RTD
                }
            }
            if (A4_USO == ON) {
                float s = ANALOG_A4;
                if (value_A4 != s) {
                    value_A4 = s;
    
                    uint16_t value = s * 1024;
    
                    float Vtemp = value * 0.0032080078125;
                    float Rtemp = (15111 / Vtemp) - 4630;
                    float Tempfinal = ((Rtemp - 1000) / 3.850);
    
                    //2- Conversion flotante a punto fijo
                    int16_t Temperature_PF = (int16_t) (256 * Tempfinal);
    
                    LECTURA_ANALOGICA[2] = 0x04;        //  --> Posicion del sensor
                    LECTURA_ANALOGICA[3] = (int8_t)(Temperature_PF >> 8);
                    LECTURA_ANALOGICA[4] = (int8_t)(Temperature_PF);
    
                    pc.printf("ANALOG_A4 (Format HEX): %x , LECTURA_ANALOGICA: %x \r\n", Temperature_PF, LECTURA_ANALOGICA);
                    ble.updateCharacteristicValue(rxCharacteristic.getValueAttribute().getHandle(), LECTURA_ANALOGICA, 5);    // Para el RTD
                }
            }
            if (A3_USO == ON) {
                float s = ANALOG_A3;
                if (value_A3 != s) {
                    value_A3 = s;
    
                    uint16_t value = s * 1024;
    
                    float Vtemp = value * 0.0032080078125;
                    float Rtemp = (15111 / Vtemp) - 4630;
                    float Tempfinal = ((Rtemp - 1000) / 3.850);
    
                    //2- Conversion flotante a punto fijo
                    int16_t Temperature_PF = (int16_t) (256 * Tempfinal);
    
                    LECTURA_ANALOGICA[2] = 0x03;        //  --> Posicion del sensor
                    LECTURA_ANALOGICA[3] = (int8_t)(Temperature_PF >> 8);
                    LECTURA_ANALOGICA[4] = (int8_t)(Temperature_PF);
    
                    pc.printf("ANALOG_A3 (Format HEX): %x , LECTURA_ANALOGICA: %x \r\n", Temperature_PF, LECTURA_ANALOGICA);
                    ble.updateCharacteristicValue(rxCharacteristic.getValueAttribute().getHandle(), LECTURA_ANALOGICA, 5);    // Para el RTD
                }
            }
            if (A2_USO == ON) {
                float s = ANALOG_A2;
                if (value_A2 != s) {
                    value_A2 = s;
    
                    uint16_t value = s * 1024;
    
                    float Vtemp = value * 0.0032080078125;
                    float Rtemp = (15111 / Vtemp) - 4630;
                    float Tempfinal = ((Rtemp - 1000) / 3.850);
    
                    //2- Conversion flotante a punto fijo
                    int16_t Temperature_PF = (int16_t) (256 * Tempfinal);
    
                    LECTURA_ANALOGICA[2] = 0x02;        //  --> Posicion del sensor
                    LECTURA_ANALOGICA[3] = (int8_t)(Temperature_PF >> 8);
                    LECTURA_ANALOGICA[4] = (int8_t)(Temperature_PF);
    
                    pc.printf("ANALOG_A2 (Format HEX): %x , LECTURA_ANALOGICA: %x \r\n", Temperature_PF, LECTURA_ANALOGICA);
                    ble.updateCharacteristicValue(rxCharacteristic.getValueAttribute().getHandle(), LECTURA_ANALOGICA, 5);    // Para el RTD
                }
            }
            if (A1_USO == ON) {
                float s = ANALOG_A1;
                if (value_A1 != s) {
                    value_A1 = s;
    
                    uint16_t value = s * 1024;
    
                    float Vtemp = value * 0.0032080078125;
                    float Rtemp = (15111 / Vtemp) - 4630;
                    float Tempfinal = ((Rtemp - 1000) / 3.850);
    
                    //2- Conversion flotante a punto fijo
                    int16_t Temperature_PF = (int16_t) (256 * Tempfinal);
    
                    LECTURA_ANALOGICA[2] = 0x01;        //  --> Posicion del sensor
                    LECTURA_ANALOGICA[3] = (int8_t)(Temperature_PF >> 8);
                    LECTURA_ANALOGICA[4] = (int8_t)(Temperature_PF);
    
                    pc.printf("ANALOG_A1 (Format HEX): %x , LECTURA_ANALOGICA: %x \r\n", Temperature_PF, LECTURA_ANALOGICA);
                    ble.updateCharacteristicValue(rxCharacteristic.getValueAttribute().getHandle(), LECTURA_ANALOGICA, 5);    // Para el RTD
                }
            }
            if (A0_USO == ON) {
                float s = ANALOG_A0;
                if (value_A0 != s) {
                    value_A0 = s;
    
                    uint16_t value = s * 1024;
    
                    float Vtemp = value * 0.0032080078125;
                    float Rtemp = (15111 / Vtemp) - 4630;
                    float Tempfinal = ((Rtemp - 1000) / 3.850);
    
                    //2- Conversion flotante a punto fijo
                    int16_t Temperature_PF = (int16_t) (256 * Tempfinal);
    
                    LECTURA_ANALOGICA[2] = 0x00;        //  --> Posicion del sensor
                    LECTURA_ANALOGICA[3] = (int8_t)(Temperature_PF >> 8);
                    LECTURA_ANALOGICA[4] = (int8_t)(Temperature_PF);
    
                    pc.printf("ANALOG_A0 (Format HEX): %x , LECTURA_ANALOGICA: %x \r\n", Temperature_PF, LECTURA_ANALOGICA);
                    ble.updateCharacteristicValue(rxCharacteristic.getValueAttribute().getHandle(), LECTURA_ANALOGICA, 5);    // Para el RTD
                }
            }
        }
    }
}

void apagarLeds()
{
    //  SOLO PARA NRF51822
    if (D3_TYPE == OUT) {
        //PIN_D3 = 0;
    }
    if (D4_TYPE == OUT) {
        //PIN_D4 = 1;
    }
    if (D5_TYPE == OUT) {
        //PIN_D5 = 1;
    }
    if (D6_TYPE == OUT) {
        //PIN_D6 = 0;
    }
}

int main(void)
{
    Ticker ticker;
    ticker.attach_us(m_status_check_handle, 200000);

    ble.init();
    ble.onDisconnection(disconnectionCallback);
    ble.onConnection(connectionCallback);
    ble.onDataWritten(WrittenHandler);

    pc.baud(9600);
    pc.printf("SimpleChat Init \r\n");
    //pc.attach( uartCB , pc.RxIrq);

    // setup advertising
    ble.accumulateAdvertisingPayload(GapAdvertisingData::BREDR_NOT_SUPPORTED);
    ble.setAdvertisingType(GapAdvertisingParams::ADV_CONNECTABLE_UNDIRECTED);
    ble.accumulateAdvertisingPayload(GapAdvertisingData::SHORTENED_LOCAL_NAME, (const uint8_t *)"Biscuit2", sizeof("Biscuit2") - 1);   // Original:  Biscuit
    ble.accumulateAdvertisingPayload(GapAdvertisingData::COMPLETE_LIST_128BIT_SERVICE_IDS, (const uint8_t *)uart_base_uuid_rev, sizeof(uart_base_uuid));

    ble.setAdvertisingInterval(160);

    ble.addService(uartService);

    ble.startAdvertising();
    pc.printf("Advertising Start \r\n");
    apagarLeds();
    //por dixys
    // para probar, luego quitar. Esto hace que cada ticker se envie un dato analogico via BLE

    //  valores refernenciales para los estados de los sensores.
    
    #if A0_USO == ON
        value_A0 = ANALOG_A0.read();
    #else
        value_A0 = 0;
    #endif
    
    #if A1_USO == ON
        value_A1 = ANALOG_A1.read();
    #else
        value_A1 = 0;
    #endif
    
    #if A2_USO == ON
        value_A2 = ANALOG_A2.read();
    #else
        value_A2 = 0;
    #endif
    
    #if A3_USO == ON
        value_A3 = ANALOG_A3.read();
    #else
        value_A3 = 0;
    #endif
    
    #if A4_USO == ON
        value_A4 = ANALOG_A4.read();
    #else
        value_A4 = 0;
    #endif
    
    #if A5_USO == ON
        value_A5 = ANALOG_A5.read();
    #else
        value_A5 = 0;
    #endif

    while(1) {
        ble.waitForEvent();
    }
}