/* Simple embedded web server on LPC1768 with the WROOM2 WiFi module. */
// Copyright Toshihiro Matsui, IISEC, March, 2019
// Any form of use of this program is fully granted to anyone.
//

#include "mbed.h"

#define MAXRECVBUF 2048
#define WEBFILE "webfile.htm"
#define debug 0

// KBline extends Serial to allow asynchronous input of command from PC
// and to enable line editing during the input
class KBline : public Serial   // Keyboard line
{ char line[100];

  public:
    int index, receiving;
  private:
    void serialirq() 
      { char ch;
        if (this->readable()) {
          ch=this->getc();
          switch(ch) { //simple line editing
              case '\n':
              case '\r':  //LF and CR terminate reading
                line[index++]=ch; line[index++]=0;
                this->printf("\r\n");
                receiving=3;
                break;
              case 21:  /* control-U cancels entire line*/
                this->printf("\n\r"); index=0; break;
              case 0x08:
              case 0x7f: /*BS and DEL delete the last character*/
                if (index>0) { index--;
                   this->putc(0x08); this->putc(0x20); this->putc(0x08);}
                break;
              default:
                receiving=2; line[index++]=ch; this->putc(ch); break; 
              } } }
  public:
    KBline(PinName tx, PinName rx) : Serial(tx, rx)
    { index=0; receiving=0; }; //constructor
    void start_getline() {
       index=0; receiving=1;
       this->attach(this, &KBline::serialirq); }
    char *getline() {
       while (receiving!=3) wait(0.1);
       return(line); }    } ;

/********************************************************************/
class Wroom  : public Serial
 {  char sendbuf[256];
    char recvbuf[MAXRECVBUF];  /* 1024 */
    Timeout tmout;
    int recvindex;
    int receiving;
 
    public:
     Wroom(PinName tx, PinName rx) : Serial(tx, rx)
       { receiving=0;} 

    private: 
     void recvtimeout() {
         tmout.detach();
         recvbuf[recvindex++]=0; /*NULL at the end*/
         receiving=3; } /* receiving finished*/
     void recvirq() {
        while (this->readable()) {
            recvbuf[recvindex++]=this->getc();
            if (recvindex>=MAXRECVBUF) recvindex=0; }
        // if receiving==0, inputs should be ignored and discarded.
        receiving=2;
        tmout.detach();
        tmout.attach(this, &Wroom::recvtimeout, 1.0);         }  

    public:
     void start_receiving() {
         recvindex=0; receiving=1;   }
     int  receive_finished() { return(receiving==3);}
     char *getreply() {
         /* tmout.attach(this, &Wroom::recvtimeout, 7.0);*/
         while (receiving!=3) { wait(0.1); }
         // receiving=0;
         return(recvbuf); }  
    char *getmessage() { //get http request
         this->start_receiving();
         return(this->getreply()); }    
    char *atcom(char *com) {
         this->start_receiving();
         this->printf("%s\r\n", com); //pc->printf("atcom: %s\n", com);
         return(this->getreply());}
    char *atcom(char *com, int mode) {
         sprintf(sendbuf, "%s=%d", com, mode);
         return(this->atcom(sendbuf));  }
    char *atcom(char *com, int param1, int param2) {
         sprintf(sendbuf, "%s=%d,%d", com, param1, param2);
         // pc->printf("atcom2 -->%s\r\n", sendbuf);
         return(this->atcom(sendbuf)); }
    void configure(KBline *p) {  //set up WROOM2
         KBline *pc=p;
         this->baud(115200); wait(1.0);
         recvindex=0; this->attach(this, &Wroom::recvirq);
         pc->printf("AT-->%s\n", this->atcom("AT"));
         pc->printf("ATE0-->%s\n", this->atcom("ATE0"));  //turn off echo
         pc->printf("CWMODE-->%s\n", this->atcom("AT+CWMODE", 1)); //1=station, 2=AP, 3=both
         pc->printf("CIPMUX-->%s\n", this->atcom("AT+CIPMUX", 1)); //0=single, 1=multi
         pc->printf("CIPSTO-->%s\n", this->atcom("AT+CIPSTO", 120)); //timeout
         pc->printf("CWLAP-->%s\n", this->atcom("AT+CWLAP")); //AP list
         // connection to AP giving SSID and PASS is not programmed.
         }     };

/***************************************************************/
DigitalOut myled(LED4), led1(LED1), led2(LED2), led3(LED3);
AnalogIn adc(p20);
KBline pc(USBTX, USBRX);
Wroom wroom(p28,p27);
LocalFileSystem local("local");
char webbuf[2000];

void strsubst(char *current, char *pat, char *subst, char *news)
 //substitute pat in current with subst and generate in news
{ char *s;
  int i;
  s=strstr(current, pat);
  i=s-current;
  if (s) {
    strncpy(news, current, i);
    strncpy(news+i, subst, strlen(subst));
    strncpy(news+i+strlen(subst), current+i+strlen(pat), strlen(current)-i-strlen(pat));
    news[strlen(current)-strlen(pat)+strlen(subst)]=NULL;
    }
  }

// read a file from the FAT fs in the LPC1768 USB memory 
char *readfile(char *fname, char *buf)
{ int rdfd, filelen;
  char fullfname[128];
  sprintf(fullfname, "/local/%s", fname);
  rdfd=open(fullfname,O_RDONLY);
  if (rdfd<0) {
    pc.printf("%s not found\r\n", fullfname);
    return(NULL); }
  else { 
    filelen=read(rdfd, buf, 2000);
    if (0<filelen && filelen<2000)  buf[filelen]=0;
    pc.printf("%d bytes read from %s\r\n", filelen, fullfname); 
    return(buf);}
  }
  
char *makepage()
{ char altwebbuf[2000], adval[16];
  readfile(WEBFILE, webbuf);
  if (led1) {
      strsubst(webbuf, "$LED1ON$", "checked", altwebbuf);
      strsubst(altwebbuf, "$LED1OFF$", " ", webbuf); }
  else {
      strsubst(webbuf, "$LED1OFF$", "checked", altwebbuf);
      strsubst(altwebbuf, "$LED1ON$", " ", webbuf); }
  if (led2) {
      strsubst(webbuf, "$LED2ON$", "checked", altwebbuf);
      strsubst(altwebbuf, "$LED2OFF$", " ", webbuf); }
  else {
      strsubst(webbuf, "$LED2OFF$", "checked", altwebbuf);
      strsubst(altwebbuf, "$LED2ON$", " ", webbuf); }
  sprintf(adval, "%6.3f", adc.read());
  strsubst(webbuf, "$ADVAL$", adval, altwebbuf);
  strcpy(webbuf, altwebbuf);
  return(webbuf);   }

void change_led(char *request)
{ 
  if (strstr(request, "led1=1")) led1=1;
  if (strstr(request, "led1=0")) led1=0;
  if (strstr(request, "led2=1")) led2=1;
  if (strstr(request, "led2=0")) led2=0;
   }

int main() {
char *line, *page;

  pc.baud(9600);
  pc.printf("\r\nWROOM AT commander.\r\n");  
  wroom.configure(&pc);
  pc.printf("\r\nWROOM is configured.\r\n");  
  pc.printf(">");
  pc.start_getline();
  wroom.start_receiving();
  while(1) {
     myled=1; //prompt
     if (pc.receiving==3) { //kb input completed
       line=pc.getline();  /*pc.printf(line);*/ 
       myled=0; 
       if (line[0] == 'r') {  //receive http request
         line=wroom.getmessage(); 
         change_led(line); }
       else if (*line == 's') { //send http message
         page=makepage();
         wroom.atcom("AT+CIPSEND",0,strlen(page));
         pc.printf(wroom.getreply());
         wroom.atcom(page); //send html document
         pc.printf(wroom.getreply());
         wroom.atcom("AT+CIPCLOSE", 0);
         line=wroom.getreply();     }
       else { //AT command
         if(debug) pc.printf(line);
         wroom.atcom(line); line=wroom.getreply(); } 
       pc.printf(line); 
       wroom.start_receiving();  //wroom.receiving=1;
       pc.printf(">"); pc.start_getline();  } 
     if (wroom.receive_finished()) {
         pc.printf(wroom.getreply());
         /*  */
         wroom.start_receiving(); } 
     wait(0.1); led3 = !led3;
     }
  } ;
  
