#include "mbed.h"
#include "spec_3sp_co.h"

extern Serial pc;


SPEC_3SP_CO::SPEC_3SP_CO(PinName sda, PinName scl, uint8_t addr, float sensitivity):i2c(sda, scl)
{
    _i2caddr = addr;
    i2c.frequency(100000);
    
    _sensitivity = sensitivity;
    
    if (ADS1x1x_init(&my_adc,ADS1115,ADS1x1x_I2C_ADDRESS_ADDR_TO_GND,MUX_SINGLE_0,PGA_4096)==0)
    {
        pc.printf("Could not initialise ADC structure.");
    }
    
}


// Writes 16 bits to the specified destination register.
void SPEC_3SP_CO::ADS1x1x_write_register(uint8_t i2c_address, uint8_t reg, uint16_t value)
{
    // send instruction to device
    char send_data[3];
    send_data[0] = reg;
    send_data[1] = ((uint8_t)(value>>8));
    send_data[2] = ((uint8_t)(value&0xff));
    i2c.write(_i2caddr, send_data, sizeof(send_data));
}

// Read 16 bits from the specified destination register.
uint16_t SPEC_3SP_CO::ADS1x1x_read_register(uint8_t i2c_address, uint8_t reg)
{
    char send_data[] = { reg };
    char recv_data[2];

    i2c.write(_i2caddr, send_data, sizeof(send_data));
    i2c.read (_i2caddr, recv_data, sizeof(recv_data));
    
    uint16_t result;
    result = recv_data[0] << 8;
    result |= recv_data[1];
    
    //pc.printf ("%04X\r\n", result);
    return result;
}

uint8_t SPEC_3SP_CO::init()
{
    return this->ADS1x1x_init(&my_adc, ADS1013, ADS1x1x_I2C_ADDRESS_ADDR_TO_VCC, MUX_DIFF_0_1, PGA_4096);
}

void SPEC_3SP_CO::start_conversion()
{
    this->ADS1x1x_start_conversion(&my_adc);
}

float SPEC_3SP_CO::read()
{
    int16_t result = this->ADS1x1x_read(&my_adc);
    float adc_volts = (((float)result/4096.0)*2.048);
    float co_ppm = ((adc_volts / 220000.0) * 1000000000.0) / _sensitivity;
    return co_ppm;
}
    

uint8_t SPEC_3SP_CO::ADS1x1x_init(ADS1x1x_config_t *p_config, ADS1x1x_chip_t chip, uint8_t i2c_address, ADS1x1x_mux_t input, ADS1x1x_pga_t gain)
{
  uint8_t result = 0;

  if (p_config!=0)
  {
    // Set generic parameters.
    p_config->chip = chip;
    p_config->i2c_address = i2c_address;

    // Set configuration bits for default operation.
    p_config->config = 0;
    ADS1x1x_set_os(p_config,OS_SINGLE);
    ADS1x1x_set_multiplexer(p_config,input);
    ADS1x1x_set_pga(p_config,gain);
    ADS1x1x_set_mode(p_config,MODE_SINGLE_SHOT);
    if (p_config->chip==ADS1013 || p_config->chip==ADS1014 || p_config->chip==ADS1015)
    {
      ADS1x1x_set_data_rate(p_config,DATA_RATE_ADS101x_1600);
    }
    else
    {
      ADS1x1x_set_data_rate(p_config,DATA_RATE_ADS111x_128);
    }
    ADS1x1x_set_comparator_mode(p_config,COMPARATOR_MODE_TRADITIONAL);
    ADS1x1x_set_comparator_polarity(p_config,COMPARATOR_POLARITY_ACTIVE_LO);
    ADS1x1x_set_comparator_latching(p_config,COMPARATOR_NON_LATCHING);
    ADS1x1x_set_comparator_queue(p_config,COMPARATOR_QUEUE_NONE);
    
    result = 1;
  }

  return result;
}

void SPEC_3SP_CO::ADS1x1x_start_conversion(ADS1x1x_config_t *p_config)
{
  // Write configuration to the ADC.
  ADS1x1x_write_register(p_config->i2c_address,ADS1x1x_REG_POINTER_CONFIG,p_config->config);
}


/**************************************************************************/
/*!
    @brief  Read the ADC conversion result.
            The user must provide a valid pointer to a
            correctly filled ADS1x1x_config_t structure.
*/
/**************************************************************************/
int16_t SPEC_3SP_CO::ADS1x1x_read(ADS1x1x_config_t *p_config)
{
  // Read the conversion result.
  int16_t result = (int16_t)ADS1x1x_read_register(p_config->i2c_address,ADS1x1x_REG_POINTER_CONVERSION);
  // Adjust for ADC resolution if needed.
  if (p_config->chip==ADS1013 || p_config->chip==ADS1014 || p_config->chip==ADS1015)
  {
    result >>= 4;
  }
  return result;
}


void SPEC_3SP_CO::ADS1x1x_set_threshold_lo(ADS1x1x_config_t *p_config, uint16_t value)
{
  if (p_config->chip==ADS1013 || p_config->chip==ADS1014 || p_config->chip==ADS1015)
  {
    value <<= 4;
  }
  ADS1x1x_write_register(p_config->i2c_address,ADS1x1x_REG_POINTER_LO_THRESH,value);
}


void SPEC_3SP_CO::ADS1x1x_set_threshold_hi(ADS1x1x_config_t *p_config, uint16_t value)
{
  if (p_config->chip==ADS1013 || p_config->chip==ADS1014 || p_config->chip==ADS1015)
  {
    value <<= 4;
  }
  ADS1x1x_write_register(p_config->i2c_address,ADS1x1x_REG_POINTER_HI_THRESH,value);
}


void SPEC_3SP_CO::ADS1x1x_set_config_bitfield(ADS1x1x_config_t *p_config, uint16_t value, uint16_t mask)
{
  p_config->config &= ~mask;
  p_config->config |= (value & mask);
}


void SPEC_3SP_CO::ADS1x1x_set_os(ADS1x1x_config_t *p_config, ADS1x1x_os_t value)
{
  ADS1x1x_set_config_bitfield(p_config,(uint16_t)value,ADS1x1x_REG_CONFIG_OS_MASK);
}


void SPEC_3SP_CO::ADS1x1x_set_multiplexer(ADS1x1x_config_t *p_config, ADS1x1x_mux_t value)
{
  ADS1x1x_set_config_bitfield(p_config,(uint16_t)value,ADS1x1x_REG_CONFIG_MULTIPLEXER_MASK);
}


void SPEC_3SP_CO::ADS1x1x_set_pga(ADS1x1x_config_t *p_config, ADS1x1x_pga_t value)
{
  ADS1x1x_set_config_bitfield(p_config,(uint16_t)value,ADS1x1x_REG_CONFIG_PGA_MASK);
}


void SPEC_3SP_CO::ADS1x1x_set_mode(ADS1x1x_config_t *p_config, ADS1x1x_mode_t value)
{
  ADS1x1x_set_config_bitfield(p_config,(uint16_t)value,ADS1x1x_REG_CONFIG_MODE_MASK);
}


void SPEC_3SP_CO::ADS1x1x_set_data_rate(ADS1x1x_config_t *p_config, ADS1x1x_data_rate_t value)
{
  ADS1x1x_set_config_bitfield(p_config,(uint16_t)value,ADS1x1x_REG_CONFIG_DATA_RATE_MASK);
}


void SPEC_3SP_CO::ADS1x1x_set_comparator_mode(ADS1x1x_config_t *p_config, ADS1x1x_comparator_mode_t value)
{
  ADS1x1x_set_config_bitfield(p_config,(uint16_t)value,ADS1x1x_REG_CONFIG_COMPARATOR_MODE_MASK);
}


void SPEC_3SP_CO::ADS1x1x_set_comparator_polarity(ADS1x1x_config_t *p_config, ADS1x1x_comparator_polarity_t value)
{
  ADS1x1x_set_config_bitfield(p_config,(uint16_t)value,ADS1x1x_REG_CONFIG_COMPARATOR_POLARITY_MASK);
}


void SPEC_3SP_CO::ADS1x1x_set_comparator_latching(ADS1x1x_config_t *p_config, ADS1x1x_comparator_latching_t value)
{
  ADS1x1x_set_config_bitfield(p_config,(uint16_t)value,ADS1x1x_REG_CONFIG_COMPARATOR_LATCHING_MASK);
}


void SPEC_3SP_CO::ADS1x1x_set_comparator_queue(ADS1x1x_config_t *p_config, ADS1x1x_comparator_queue_t value)
{
  ADS1x1x_set_config_bitfield(p_config,(uint16_t)value,ADS1x1x_REG_CONFIG_COMPARATOR_QUEUE_MASK);
}