#include "mbed.h"
#include "mDot.h"
#include "ChannelPlans.h"

//#define TEST_HPM              // PM2.5 / PM10
//#define TEST_MICS_VZ_89TE     // voc
//#define TEST_SPEC_3SP_CO        // CO
#define TEST_K30                // CO2 sensor

#ifdef TEST_HPM
    #include "hpm.h"
#endif

#ifdef TEST_MICS_VZ_89TE
    #include "mics_vz_89te.h"
#endif

#ifdef TEST_SPEC_3SP_CO
    #include "spec_3sp_co.h"
#endif

#ifdef TEST_K30
    #include "K30_I2C.h"
#endif


mDot* dot = NULL;
Serial      pc          (USBTX, USBRX);      // xDot USB interface
Serial      device      (UART_TX, UART_RX);  // xDot UART to sensors

#ifdef TEST_HPM
    HPM hpm = HPM::HPM(&device, &pc);
    long int PM25;
    long int PM10;
#endif

#ifdef TEST_MICS_VZ_89TE
    MICS_VZ_89TE sensor_voc = MICS_VZ_89TE::MICS_VZ_89TE(I2C_SDA, I2C_SCL, 0x70<<1);
    int co2_fromVOC = 0;
    int voc = 0;
#endif

#ifdef TEST_SPEC_3SP_CO
    SPEC_3SP_CO sensor_co = SPEC_3SP_CO::SPEC_3SP_CO(I2C_SDA, I2C_SCL, ADS1x1x_I2C_ADDRESS_ADDR_TO_VCC<<1, 3.29);
    int co = 0;
#endif

#ifdef TEST_K30
    K30_I2C k30(I2C_SDA, I2C_SCL, 0x68<<1);
    int co2 = 0;
#endif

DigitalOut  blueLED     (GPIO0);
DigitalOut  orangeLED   (GPIO1);
DigitalOut  enable_5V   (GPIO3);
DigitalOut  enable_voc  (UART_RTS);
DigitalOut  enable_co   (SPI_SCK);

bool b_response;

void turnOn5v(void)
{
    pc.printf("Turning on 5v rail...");
    enable_5V = true;
    wait(1);
    pc.printf("done\r\n");    
}
    
void turnOffv(void)
{
    pc.printf("Turning off 5v rail...");
    enable_5V = false;
    wait(1);
    pc.printf("done\r\n");    
}

int main ()
{
    // Initialising
    blueLED = true;
    orangeLED = false;
                
    pc.baud(57600);        // start debug port
    pc.printf("************************\r\n");
    pc.printf(" HARDWARE TEST FIRMWARE\r\n");
    pc.printf("************************\r\n\r\n");


#ifdef TEST_HPM
    turnOn5v();
    do {
        b_response = hpm.init();
        if (!b_response)
        {
            pc.printf("Initialisation of PM sensor failed\r\n");
            wait(1.0);
        }
    } while (!b_response);
    pc.printf("Initialisation of PM sensor success\r\n");
        
#endif

#ifdef TEST_MICS_VZ_89TE    
    pc.printf("Enabling MICS_VZ_89TE...");
    enable_voc = true;
    wait(1);
    pc.printf("done\r\n");
    
    sensor_voc.getVersion();
#endif
 
#ifdef TEST_SPEC_3SP_CO
    pc.printf("Enabling SPEC_3SP_CO...");
    enable_co = true;
    wait(1);
    pc.printf("done\r\n");

    sensor_co.init();
#endif

#ifdef TEST_K30
    turnOn5v();
#endif
  
    orangeLED = blueLED = false;
    
    while(1)
    {
#ifdef TEST_HPM
        b_response = hpm.read_measurement(PM25, PM10); 
        if (b_response)
        {
            pc.printf ("PM2.5 = %d ug/m3,  PM10 = %d ug/m3\r", PM25, PM10);
        }
        else
        {
            pc.printf("Read measurement status is %d\r\n", b_response);
        }
#endif
        
#ifdef TEST_MICS_VZ_89TE
        char status = sensor_voc.readSensor(co2_fromVOC, voc);
        pc.printf ("MICS_VZ_89TE:  CO2=%d ppm, VOC%d ppb  (status=%02x)\r\n", co2_fromVOC, voc, status);
#endif

#ifdef TEST_SPEC_3SP_CO
        sensor_co.start_conversion();
        // wait 7.8ms
        wait(0.01);
        
        float co_ppm = sensor_co.read();
        pc.printf ("SPEC_3SP_CO:  CO=%.2fppm\r\n", co_ppm);
#endif

#ifdef TEST_K30
        int rc_check = k30.readCO2(co2);
        if (rc_check)
            pc.printf("K30 CO2:%d ppm\r\n",co2);
        else
            pc.printf("Failed to read CO2 sensor\r\n");
#endif

        wait (1);
        blueLED = !blueLED;
        pc.printf ("\r\n");
    }   
}