
/******************************************************************************
* Includes
*******************************************************************************/

#include "mbed.h"
#include "mlx90614.h"
#include "Hexi_KW40Z.h"
#include "Hexi_OLED_SSD1351.h"
#include "OLED_types.h"
#include "OpenSans_Font.h"
#include "string.h"
#include "irimages_c.h"

 

DigitalOut blueLed(LED3,1);
DigitalOut haptic(PTB9);

I2C i2c(PTD9,PTD8);         //sda,scl (start i2c interace)

Serial pc(USBTX,USBRX);     //serial usb config for debug

MLX90614 IR_thermometer(&i2c); // instantiate IR Thermometer


/* prototypes */
void Sys_Init(void);
void readTemp(void);
void StartHaptic(void);
void StopHaptic(void const *n);
void txTask(void);
void animateDisplay(void);
void displayTemp(void);

/* Define timer for haptic feedback */
RtosTimer hapticTimer(StopHaptic, osTimerOnce);

/* Instantiate the Hexi KW40Z Driver (UART TX, UART RX) */ 
KW40Z kw40z_device(PTE24, PTE25);


/* Instantiate the SSD1351 OLED Driver */ 
SSD1351 oled(PTB22,PTB21,PTC13,PTB20,PTE6, PTD15); /* (MOSI,SCLK,POWER,CS,RST,DC) */

/*Create a Thread to handle sending BLE Sensor Data */ 
Thread txThread;

/*Create a Thread to handle reading the sensor and display*/ 
Thread dispThread;

 /* Text Buffer */ 
char text[20];

/* Temperature images */
const uint8_t *blue1;       // Pointer for the image to be displayed
const uint8_t *orange2;     // Pointer for the image to be displayed
const uint8_t *red3;        // Pointer for the image to be displayed

float tempC; //temperature in degrees C
float tempF; //temperature in degrees F


/****************************Call Back Functions*******************************/

void ButtonRight(void)
{
    StartHaptic();
    kw40z_device.ToggleAdvertisementMode();
}

void ButtonLeft(void)
{
    StartHaptic();
    kw40z_device.ToggleAdvertisementMode();
}

void PassKey(void)
{
    StartHaptic();
  
    /* Display Bond Pass Key in a 95px by 18px textbox at x=0,y=40 */
    sprintf(text,"%d", kw40z_device.GetPassKey());
    oled.TextBox((uint8_t *)text,0,72,95,18);
}

/***********************End of Call Back Functions*****************************/

/********************************Main******************************************/

int main()
{    
    Sys_Init(); // system initial values
    
    while (true) 
    {
        blueLed = !kw40z_device.GetAdvertisementMode(); /*Indicate BLE Advertisment Mode*/   
        Thread::wait(50);
        
    }
}

/******************************End of Main*************************************/


/* txTask() transmits the sensor data */
void txTask(void){
   
   while (true) 
   {
        
        /*Notify Hexiwear App that it is running Sensor Tag mode*/
        kw40z_device.SendSetApplicationMode(GUI_CURRENT_APP_SENSOR_TAG);
      
        kw40z_device.SendTemperature(tempC*100); //send IR Click Temperature
      
        Thread::wait(1000);                 
    }
}


void StartHaptic(void)  {
    hapticTimer.start(50);
    haptic = 1;
}

void StopHaptic(void const *n) {
    haptic = 0;
    hapticTimer.stop();
}

void readTemp(void) {
    if (IR_thermometer.getTemp(&tempC)) {
    //gets temperature from sensor via I2C bus
        
        tempF = tempC * 9.0 / 5 + 32; // convert C to F
        
        //print temperature on PC
        printf("Temperature is %4.2F degrees C\r\n",tempC);
        printf("Temperature is %4.2F degrees F\r\n",tempF);
        }
}

void Sys_Init(void){
    
    blue1 = blue1s_bmp;      // Pointer for the image to be displayed
    orange2 = orange2s_bmp;  // Pointer for the image to be displayed
    red3 = red3s_bmp;        // Pointer for the image to be displayed
    
    /* Fills the screen with solid black */         
    oled.FillScreen(COLOR_BLACK);
    
     /* Turn on the backlight of the OLED Display */
//    oled.DimScreenON();
       
    oled.DrawImage(blue1,0,0);          // Fill 96px by 96px Screen with 96px by 96px NXP Image starting at x=0,y=0
    wait(0.5);
    oled.DrawImage(orange2,0,0);        // Fill 96px by 96px Screen with 96px by 96px NXP Image starting at x=0,y=0
    wait(0.5);
    oled.DrawImage(red3,0,0);           // Fill 96px by 96px Screen with 96px by 96px NXP Image starting at x=0,y=0
    wait(0.5);  
    
    /* Get OLED Class Default Text Properties */
    oled_text_properties_t textProperties = {0};
    oled.GetTextProperties(&textProperties);    
    
    /* Register callbacks to application functions */
    kw40z_device.attach_buttonLeft(&ButtonLeft);
    kw40z_device.attach_buttonRight(&ButtonRight);
    kw40z_device.attach_passkey(&PassKey);

    /* Change font color to white */ 
    textProperties.font = OpenSans_12x18_Regular;
    textProperties.fontColor   = COLOR_WHITE;
    textProperties.alignParam = OLED_TEXT_ALIGN_LEFT;
    oled.SetTextProperties(&textProperties);
    
    strcpy((char *) text,"C");
    /* Display Label at x=60,y=72 */ 
    oled.Label((uint8_t *)text,60,72);
        
    txThread.start(txTask); /*Start transmitting Sensor Tag Data */
    dispThread.start(animateDisplay); /* start reading the temperature */
}

 

void animateDisplay(void){
            
    while(true){
                
        readTemp();
        
        if (tempC > 30) // for temps over 30 display red
             oled.DrawImage(red3,0,0);
        else if (tempC < 30 || tempC > 10) // for temps between 10 and 30 display orange
             oled.DrawImage(orange2,0,0);
        else if (tempC < 10)  // for temps under 10 display blue       
             oled.DrawImage(blue1,0,0);
        displayTemp(); // show temp on display
        
        Thread::wait(500);
    }    
}            

void displayTemp(void){
 
     /* format the text */       
     sprintf(text,"%.1f", tempC);
    /* Display Label at x=22,y=80 */ 
     oled.TextBox((uint8_t *)text,22,72,30,18);
}                         