

/******************************************************************************
* Includes
*******************************************************************************/

#include "mbed.h"
#include "Hexi_OLED_SSD1351.h"
#include "Hexi_KW40Z.h"
#include "images.h"
#include "new_images.h"
#include "string.h"
#include "stdbool.h"

/******************************************************************************
* Input and Output Pinout
*******************************************************************************/

AnalogIn ain(PTB2);
SSD1351 oled(PTB22,PTB21,PTC13,PTB20,PTE6, PTD15); // SSD1351 OLED Driver (MOSI,SCLK,POWER,CS,RST,DC)
KW40Z kw40z_device(PTE24, PTE25); // Instantiate the Hexi KW40Z Driver (UART TX, UART RX)
DigitalOut blueLed(LED3,1);
Serial pc(USBTX, USBRX);

Thread txThread;

/******************************************************************************
* prototypes
*******************************************************************************/
void InitModules();
void ReadSensor();
void CalculatePPM();
void DisplayAirQValue(uint16_t value);


/******************************************************************************
* Module Variable Definitions
*******************************************************************************/
                
uint16_t ppm;

uint16_t value = 0;
uint16_t value_old = 0;
uint16_t adc_rd;
    
//const uint8_t *image1;          // old cover image
const uint8_t *green_arrow;     // Pointer for the image to be displayed
const uint8_t *yellow_arrow;    // Pointer for the image to be displayed
const uint8_t *orange_arrow;    // Pointer for the image to be displayed
const uint8_t *red_arrow;       // Pointer for the image to be displayed
const uint8_t *purple_arrow;    // Pointer for the image to be displayed

char text[20];                  // Text Buffer for dynamic value displayed

bool test = false; // used to put the program in text mode to check images    

/**************************************************************************************************
* Call Back Functions
**************************************************************************************************/

void ButtonRight(void)
{
    kw40z_device.ToggleAdvertisementMode();
}

void ButtonLeft(void)
{
    kw40z_device.ToggleAdvertisementMode();
}

void PassKey(void)
{
    /* Display Bond Pass Key in a 95px by 18px textbox at x=0,y=40 */
    sprintf(text,"%d", kw40z_device.GetPassKey());
    oled.TextBox((uint8_t *)text,0,72,95,18);
}



/**************************************************************************************************
* Function Main()
* -------------------------------------------------------------------------------------------------
* Overview:
* Input: None
* Output: None
**************************************************************************************************/

int main() 
{

    InitModules();
   
    while (true) 
    {
       
        ReadSensor();
        pc.printf("Sensor Value 0x%04X\n\r",ain.read_u16());

        blueLed = !kw40z_device.GetAdvertisementMode(); /*Indicate BLE Advertisment Mode*/
       
  
        CalculatePPM();
        DisplayAirQValue(ppm);
        
        Thread::wait(500); 

    }
}

/**************************************************************************************************
* Function TxTask()
**************************************************************************************************/

void txTask(void){
   
   while (true) 
   {
        
        /*Notify Hexiwear App that it is running Sensor Tag mode*/
        kw40z_device.SendSetApplicationMode(GUI_CURRENT_APP_SENSOR_TAG);
      
        kw40z_device.SendPressure(10*ppm); //send ppm Air Quality Click value
      
        Thread::wait(1000);                 
    }
}

/**************************************************************************************************
* Function InitModules()
* -------------------------------------------------------------------------------------------------
* Overview:
* Input: None
* Output: None
**************************************************************************************************/

void InitModules()
{
    //image1  = AirQuality;         // Setting pointer location of the 96 by 96 pixel bitmap
    //oled.DrawImage(image1,0,0);     // Fill 96px by 96px Screen with 96px by 96px NXP Image starting at x=0,y=0  
    
    green_arrow = Green_arrow_bmp;      // Setting pointer location of the 96 by 96 pixel bitmap Arrow Point to Green To begin
    yellow_arrow = Yellow_arrow_bmp;    // Setting pointer location of the 96 by 96 pixel bitmap
    orange_arrow = Orange_arrow_bmp;    // Setting pointer location of the 96 by 96 pixel bitmap 
    red_arrow = Red_arrow_bmp;          // Setting pointer location of the 96 by 96 pixel bitmap 
    purple_arrow = Purple_arrow_bmp;    // Setting pointer location of the 96 by 96 pixel bitmap 
    
    oled.DrawImage(green_arrow,0,0);     // Fill 96px by 96px Screen with 96px by 96px NXP Image starting at x=0,y=0 
    
    oled_text_properties_t textProperties = {0};    // Get OLED Class Default Text Properties
    oled.GetTextProperties(&textProperties); 
    
    /* Register callbacks to application functions */
    kw40z_device.attach_buttonLeft(&ButtonLeft);
    kw40z_device.attach_buttonRight(&ButtonRight);
    kw40z_device.attach_passkey(&PassKey);
      
    //PPM text now in the Bitmap
   
    //textProperties.fontColor = COLOR_BLUE;          // Set text properties to white and right aligned for the dynamic text
    //textProperties.alignParam = OLED_TEXT_ALIGN_LEFT;
    //oled.SetTextProperties(&textProperties);  
      
    //strcpy((char *) text,"ppm:");       // Display Legends
   // oled.Label((uint8_t *)text,15,75);      
    
    textProperties.fontColor = COLOR_WHITE;          // Set text properties to white and right aligned for the dynamic text
    textProperties.alignParam = OLED_TEXT_ALIGN_RIGHT;
    oled.SetTextProperties(&textProperties);  

    sprintf(text,"%i",value);                       // Format the value
    oled.TextBox((uint8_t *)text,55,76,20,15);   //Increase textbox for more digits

    txThread.start(txTask); /*Start transmitting Sensor Tag Data */      

}


/**************************************************************************************************
* Function ReadSensor()
* -------------------------------------------------------------------------------------------------
* Overview: Read sensor.
* Input: None
* Output: None
**************************************************************************************************/

void ReadSensor()
{
    adc_rd = ain.read_u16();
    wait_ms(10);
}


/**************************************************************************************************
* Function CalculatePPM()
* -------------------------------------------------------------------------------------------------
* Overview: Calculation of PPM.
* Input: None
* Output: None
**************************************************************************************************/
    
void CalculatePPM()
{
    const double Rl      = 5000.0;               // Rl (Ohm) - Load resistance
    const double Vadc_33 = 0.0000503548;         // ADC step 3,3V/65535 0.00503mV (16bit ADC)
    double Vrl;                                  // Output voltage
    double Rs;                                   // Rs (Ohm) - Sensor resistance
    double ratio;                                // Rs/Rl ratio
    double lgPPM;

    Vrl = (double)adc_rd * Vadc_33;            // For 3.3V Vcc use Vadc_33
    Rs = Rl * (3.3 - Vrl)/Vrl;                 // Calculate sensor resistance
    ratio = Rs/Rl;                             // Calculate ratio
    lgPPM = (log10(ratio) * -0.8) + 0.9;       // Calculate ppm
    ppm = 6* pow(10,lgPPM);                       // Calculate ppm
}



/**************************************************************************************************
* Function DisplayAirQValue()
* -------------------------------------------------------------------------------------------------
* Overview:
* Input: None
* Output: None
**************************************************************************************************/

void DisplayAirQValue(uint16_t value )
{
    if (value_old != value)
    {

        if( value > 200) 
            oled.DrawImage(purple_arrow,0,0);     // Fill 96px by 96px Screen with 96px by 96px NXP Image starting at x=0,y=0
        else if (value < 200 && value > 150) 
            oled.DrawImage(red_arrow,0,0);
        else if (value < 150 && value > 100) 
            oled.DrawImage(orange_arrow,0,0);
        else if (value < 100 && value > 50) 
            oled.DrawImage(yellow_arrow,0,0); 
        else  
            oled.DrawImage(green_arrow,0,0);
        
        oled_text_properties_t textProperties = {0};    // Get OLED Class Default Text Properties
        oled.GetTextProperties(&textProperties); 

        // clear the previous value
        textProperties.fontColor = COLOR_WHITE;          // Set text properties to white and right aligned for the dynamic text
        textProperties.alignParam = OLED_TEXT_ALIGN_RIGHT;
        oled.SetTextProperties(&textProperties);  

        sprintf(text,"%i",value_old);                       // Format the value
        oled.TextBox((uint8_t *)text,55,76,20,15);   //Increase textbox for more digits

        // display the new value
        textProperties.fontColor = COLOR_WHITE;          // Set text properties to white and right aligned for the dynamic text
        textProperties.alignParam = OLED_TEXT_ALIGN_RIGHT;
        oled.SetTextProperties(&textProperties);  

        sprintf(text,"%i",value);                       // Format the value
        oled.TextBox((uint8_t *)text,55,76,20,15);   //Increase textbox for more digits
    }
    value_old = value;
}