#ifndef __GPSUBX_UART_HPP__
#define __GPSUBX_UART_HPP__

#include "mbed.h"
#include "Vector3.hpp"

#define POSECEF_LEN 20
#define POSLLH_LEN 28
#define STATUS_LEN 16
#define TIMEUTC_LEN 20
#define VELECEF_LEN 20
#define VELNED_LEN 36

#define UBX_SYNC 0x62 << 8 | 0xb5

#define RECEIVE_SIZE 1024
#define SENTENCE_SIZE 64

//extern Serial twelite;

//// 0x01 0x01
//union POSECEF
//{
//    char byte_data[POSECEF_LEN+8];
//    struct
//    {
//        unsigned short sync;
//        char m_class;
//        char m_id;
//        unsigned short pay_len;
//        unsigned int iTOW;
//        int ecefX;
//        int ecefY;
//        int ecefZ;
//        unsigned int pAcc;
//        char check_a;
//        char check_b;
//    } data;
//};

// 0x01 0x02
union POSLLH
{
    char byte_data[POSLLH_LEN];
    struct
    {
        unsigned int iTOW;
        int lon;
        int lat;
        int height;
        int hMSL;
        unsigned int hAcc;
        unsigned int vAcc;
    } data;
};

// 0x01 0x03
union STATUS
{
    char byte_data[STATUS_LEN];
    struct
    {
        unsigned int iTOW;
        unsigned char gpsFix;
        unsigned char flags;
        unsigned char fixStat;
        unsigned char flags2;
        unsigned int ttff;
        unsigned int msss;
    } data;
};

// 0x01 0x21
union TIMEUTC
{
    char byte_data[TIMEUTC_LEN];
    struct
    {
        unsigned int iTOW;
        unsigned int tAcc;
        int nano;
        unsigned short year;
        unsigned char month;
        unsigned char day;
        unsigned char hour;
        unsigned char min;
        unsigned char sec;
        unsigned char valid;
    } data;
};

//// 0x01 0x11
//union VELECEF
//{
//    char byte_data[VELECEF_LEN];
//    struct
//    {
//        unsigned int iTOW;
//        int ecefVX;
//        int ecefVY;
//        int ecefVZ;
//        unsigned int sAcc;
//    } data;  
//};

// 0x01 0x12
union VELNED
{
    char byte_data[VELNED_LEN];
    struct
    {
        unsigned int iTOW;
        int velN;
        int velE;
        int velD;
        unsigned int speed;
        unsigned int gSpeed;
        signed int heading;
        unsigned int sAcc;
        unsigned int cAcc;
    } data;
};
    

/**GPS/GNSS library for UART (UBX protocol)
* RX pin is required.
@code
#include "mbed.h"
#include "GPSUBX_UART.cpp"

Serial pc(USBTX, USBRX, 115200);

GPSUBX_UART gps(PD_1, PD_0); // mbed Nucleo 767

void Display()
{
    pc.printf("POSLLH: %d, %f, %f, %f\r\n", gps.iTOW_POSLLH, gps.Longitude, gps.Latitude, gps.Height);
    pc.printf("VELNED: %d, %f, %f, %f\r\n", gps.iTOW_VELNED, gps.VelocityNED.x, gps.VelocityNED.y, gps.VelocityNED.z);
    pc.printf("TIMEUTC: %4d/%2d/%2d %2d:%2d %2d\r\n", gps.Year, gps.Month, gps.Day, gps.Hours, gps.Minutes, gps.Seconds);
}

int main()
{
    gps.Attach();
    int timer = 0;
    while (1)
    {
        gps.Update();
        wait(0.1);
        timer++;
        if (timer >= 5)
        {
            Display();
            timer = 0;
        }
    }
}
@endcode
*/
class GPSUBX_UART
{
private:

    RawSerial serial;
    char receive_buffer[RECEIVE_SIZE];
    char sentence_buffer[SENTENCE_SIZE];
    static const float Radius = 6378136.6f;
    volatile int receive_index;
    
    void Receive();
    void Decode(char buffer[], int mc, int mi);
    
public:

    int dummy; // Compiler bug? If "dummy" does not exist, the value of "Year" (under "public:") may include bugs.
    /**Year*/
    volatile int Year;
    /**Month*/
    volatile int Month;
    /**Day*/
    volatile int Day;
    /**Hours*/
    volatile int Hours;
    /**Minutes*/
    volatile int Minutes;
    /**Seconds*/
    volatile int Seconds;
    /**Longitude (deg)*/
    volatile float Longitude;
    /**Latitude (deg)*/
    volatile float Latitude;
    /**Height [m]*/
    volatile float Height;
    /**iTOW of POSLLH [ms]*/
    volatile float Hacc;
    /**Horizontal acculacy of POSLLH [mm]*/
    volatile int iTOW_POSLLH;
    /**Velocity NED [m/s]*/
    volatile Vector3 VelocityNED;
    /**Speed [m/s]*/
    volatile float Speed;
    /**Ground speed [m/s]*/
    volatile float GroundSpeed;
    /**Heading angle (deg) */
    volatile float Heading;
    /**iTOW of VELNED [ms]*/
    volatile int iTOW_VELNED;
    
    /**iTOW of STATUS [ms]*/
    volatile int iTOW_STATUS;
    /**gpsFix Status*/
    volatile char gpsFix;
    
    
    /**Time zone*/
    volatile int TimeZone;
        
    Vector3 UniversalZeroPosition;
    Vector3 UniversalZeroUnitN;
    Vector3 UniversalZeroUnitE;
    Vector3 UniversalZeroUnitD;
    Vector3 UniversalPosition;
    /**Position NED [m]*/
    Vector3 PositionNED;
    
    /**Constructor
    *@param tx UART TX pin
    *@param rx UART RX pin
    *@param baud UART baud rate (default: 38400)
    *@param timezone Time zone (default: +9)
    */
    GPSUBX_UART(PinName tx, PinName rx, int baud = 38400, int timezone = +9);
    /**call Update() regularly in the main loop
    */
    void Update();
    /**call Attach() once before measuring position
    */
    void Attach();
    
    Vector3 ToUniversalUnit();
    Vector3 ToUniversal();
    void CalculateUnit();
    /**call Calculate() before calculating PositionNED
    */
    void Calculate(float& posN,float& posE,float& posD,float& velN,float& velE,float& velD);
    Vector3 Calculate(Vector3 position);
    
    static void Checksum(char payload[], int n, char* ck_a, char* ck_b);
};
#endif