//////////////////////////////////////////////////////////////////////////////////////
//                                                                                  //
//      File        : main.cpp                                                      //
//      Version     : 0.1                                                           //
//      Initial     : 25 may 2015                                                   //
//      Author      : Dany Brugman                                                  //
//      Comment     : Function to control steppermotor using CANbus                 //
//                                                                                  //
//                                                                                  //
//      Changelog   :                                                               //
//      Date:           Name:       Comment:                                        //
//      25/05/2015      DNB         First version                                   //
//      01/06/2015      DNB         Stepper control on interrupt                    //
//                                                                                  //
//////////////////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////////////////
// includes                                                                         //
//////////////////////////////////////////////////////////////////////////////////////
#include "mbed.h"
#include "PinDetect.h"
#include "Stepper.h"

//////////////////////////////////////////////////////////////////////////////////////
// device ID                                                                        //
// CAN module foil control port side define FOIL_PORT                               //
// CAN module foil control starboard side define FOIL_STARB                         //
//////////////////////////////////////////////////////////////////////////////////////
#define FOIL_PORT
//////////////////////////////////////////////////////////////////////////////////////
// defines                                                                          //
////////////////////////////////////////////////////////////////////////////////////// 
#ifdef FOIL_PORT
    #define DEVICE_ID   101
    #define DEVICE_RUN  103    
    #define ACK         111
    #define REQ_STATUS  2001
#endif  
#ifdef FOIL_STARB
    #define DEVICE_ID   102
    #define DEVICE_RUN  104  
    #define ACK         112
    #define REQ_STATUS  2002
#endif   

#define CCW                     0
#define CW                      1

//////////////////////////////////////////////////////////////////////////////////////
// interrupts                                                                       //
//////////////////////////////////////////////////////////////////////////////////////
Ticker  ticker;
Ticker  tLed1On;
Ticker  tled2on;

Timeout tLed1Off;
Timeout tled2off;
Timeout tStepTimer;
Timeout tPulseTimer;

//////////////////////////////////////////////////////////////////////////////////////
// port declaration                                                                 //
//////////////////////////////////////////////////////////////////////////////////////
DigitalOut Led1(LED1);
DigitalOut Led2(LED2);
DigitalOut Led4(LED4);

CAN CANBus(p30, p29);

PinDetect  SW_END(p7);

DigitalOut dir(p11);        //breadbord dir(p11);
DigitalOut step(p12);       //breadbord step(p12);
DigitalOut sleep(p8);       //breadbord sleep(p13); pcb sleep(p8); 

Serial debug(USBTX, USBRX);

//////////////////////////////////////////////////////////////////////////////////////
// global variables                                                                 //
//////////////////////////////////////////////////////////////////////////////////////
char counter = 0;
char cMessage, cStatus;

Stepper sStepCount;

int i;

bool bEndReached = false;

//////////////////////////////////////////////////////////////////////////////////////
// functions declarations                                                           //
//////////////////////////////////////////////////////////////////////////////////////

void tPulse_cb (void);
void tVarPulse_cb (void);

//////////////////////////////////////////////////////////////////////////////////////
// functions                                                                        //
//////////////////////////////////////////////////////////////////////////////////////
void vSW_ENDOn_cb(void)
{
    if(dir == CCW)
    { 
        sStepCount.vSetStepCount(0);
        bEndReached = true;
    }
}

void vSW_ENDOff_cb(void)
{
    bEndReached = false;
}

 
void send() 
{
    if(CANBus.write(CANMessage(1226, &counter, 1))) 
    {
        counter++;
    } 
}

//--- working function ---
/*void pulse(void)  
{
    step = 1;
    wait_us(2);
    step = 0;
    wait_us(2);    
}*/

//--- working function ---
/*void circle(void)     
{
    sleep = 1;
    for(int i=0;i<200;i++)
    {
        pulse();  
        wait_us(900);
    }
    sleep = 0;
}*/

//--- working function ---
/*void vNumberCircles(int number)       
{
    static int counter;
    if (number != 0)      // only execut if non-zero
    {
    for(counter = 0; counter < number; counter++)
        circle();
    }
}*/

//////////////////////////////////////////////////////////////////////////////////////
// functions  good working order stand alone                                        //
////////////////////////////////////////////////////////////////////////////////////// 
/*void tPulse_cb (void)
{
    if(sStepCount.iGetStepCount() !=0) //if stepcounter != 0
    {
        sleep = 1;
        if (step == 0) step =1;
        else step = 0;
        sStepCount.vDecreaseStepCount();    // decrease counter
        // reschedule new event
        tStepTimer.detach();
        tStepTimer.attach_us(tPulse_cb, 450);
    }
    else
    {   // cancel tStepTimer event, step goal is reached
        sleep = 0;
        tStepTimer.detach();  
    }             
}*/

//////////////////////////////////////////////////////////////////////////////////////
// call back 1us pulse for stepper driver                                           //
////////////////////////////////////////////////////////////////////////////////////// 
void tPulse_cb (void)
{
    step = 1;        
    if (sStepCount.bGetRampUp())   sStepCount.vDecreaseOffTimeCount();     // decrease ramp-up timer
    if (sStepCount.bGetRampDown()) sStepCount.vIncreaseOffTimeCount();     // increase ramp-down timer
    // reschedule new event
    tStepTimer.detach();
    tPulseTimer.attach_us(tVarPulse_cb, 2);     // puls for 200us
}

//////////////////////////////////////////////////////////////////////////////////////
// call back variable off time for stepper driver                                   //
////////////////////////////////////////////////////////////////////////////////////// 
void tVarPulse_cb (void)
{
    if(sStepCount.iGetStepCount() !=0) //if stepcounter != 0
    {
        sleep = 1;
        if (step == 0) step =1;
        else step = 0;
        sStepCount.vDecreaseStepCount();    // decrease counter
        // reschedule new event
        tPulseTimer.detach();
        tStepTimer.attach_us(tPulse_cb, sStepCount.iGetOffTimeCount());     // set variable offtime
    }
    else
    {   // cancel tStepTimer event, step goal is reached
        sleep = 0;
        tPulseTimer.detach();  
    }             
}

//////////////////////////////////////////////////////////////////////////////////////
// set number of steps to step (stepper driver)                                     //
//////////////////////////////////////////////////////////////////////////////////////
void vNumberSteps(int number)
{
    sStepCount.vSetStepCount(number*100);
    sStepCount.vResetOffTimeCount();
    tStepTimer.attach_us(tPulse_cb, sStepCount.iGetOffTimeCount());     // set variable offtime
    //tStepTimer.attach_us(tPulse_cb, 450); 
}
 
void tLed1Off_cb(void) 
{
    Led1 = 0;
}
 
void tLed1On_cb(void) 
{
    Led1 = 1;
    tLed1Off.detach();
    tLed1Off.attach(&tLed1Off_cb, 0.05);
}

void tled2off_cb(void) 
{
    Led2 = 0;
}
 
void tled2on_cb(void) 
{
    Led2 = 1;
    tled2off.detach();
    tled2off.attach(&tled2off_cb, 0.05);
}

//////////////////////////////////////////////////////////////////////////////////////
// end switch handling                                                              //
//////////////////////////////////////////////////////////////////////////////////////
void switchReleased(void){
    vSW_ENDOn_cb();
    //cMessage = 0x00;
    //CANBus.write(CANMessage(100, &cMessage, 1));
    //debug.printf("Switch released: %d \r\n", cMessage);
}

void switchPressed(void){
    vSW_ENDOff_cb();
    //cMessage = 0x00;
    //CANBus.write(CANMessage(100, &cMessage, 1));
    //debug.printf("Switch released: %d \r\n", cMessage);
}

//////////////////////////////////////////////////////////////////////////////////////
// main loop                                                                        //
//////////////////////////////////////////////////////////////////////////////////////
int main() {
    debug.printf("\033[2J");
    debug.printf("\033[9;10fDevice ID:%i\r\n", DEVICE_ID);
    
//    Led1 = 1;
//    Led2 = 1;
    // schedule led events
//    tLed1Off.attach(&tLed1On_cb, 0.05);
//    tLed1On.attach(&tLed1On_cb, 2);
//    tled2off.attach(&tled2on_cb, 0.05);
//    tled2on.attach(&tled2on_cb, 1);
 //////////////////////////////////////////////////////////////////////////////////////
// switch interrupts                                                                //
//////////////////////////////////////////////////////////////////////////////////////    
    // End switch 
    SW_END.mode(PullDown);
    SW_END.attach_asserted(&switchPressed);
    SW_END.attach_deasserted(&switchReleased);
    SW_END.setSampleFrequency();                   // Defaults to 20ms.   
    
    sleep = 0;                                  // stepper controller to sleep
    
    CANBus.frequency(250000);
    CANMessage msg;
    //char Value;
    
    while(1) 
    { 
        if(CANBus.read(msg))
        {
            switch(msg.id)
            {   // Send system status if MainModule requests
                case REQ_STATUS:
                            cStatus = 0xFF;                                     // OK
                            if(msg.data[0] == 00) CANBus.write(CANMessage(DEVICE_ID, &cStatus, 1));
                            debug.printf("\033[9;10fDevice send ID:%i\r\n", DEVICE_ID);
                            wait_us(100);
                            break;
                // process stepper data direction/steps            
                case DEVICE_RUN:
                            if(msg.data[0] != 00) 
                            {
                                tLed1On.attach(&tLed1On_cb, 0.5); 
                                sleep = 1;   
                                dir = (msg.data[1]);                            // set direction
                                if (!((dir == CCW)&&(bEndReached)))
                                    vNumberSteps((uint32_t)msg.data[0]);            // set steps and kick stepper driver
                            }
                            cStatus = ACK;
                            CANBus.write(CANMessage(DEVICE_ID, &cStatus, 1));   // reply message received and executed
                            tLed1On.attach(&tLed1On_cb, 2);                     // indicate stepper active
                            break;
                
                // just dump received message (debug)
                default:    
                            debug.printf("Message id: %d\r\n", msg.id);
                            debug.printf("Message length: %d\r\n", msg.len);
                            for (i=0; i < msg.len; i++)
                            {
                                debug.printf("Message %d received: %d\r\n", i, msg.data[i]);
                            }
                            break;
            }
        }
        wait_ms(500);
    }
}

//////////////////////////////////////////////////////////////////////////////////////
// EOF                                                                              //
//////////////////////////////////////////////////////////////////////////////////////
