/*

 STM32F746 GroveStreams Stream Feed via Ethernet

 This GroveStreams example is designed for the STM32F746.
 A full "how to" guide for this sketh can be found at https://www.grovestreams.com/developers/getting_started_stm32F746.html
 This example:
  * Demonstrates uploading data to GroveStreams while downloading commands.
  * Demonstrates the GroveStreams API: https://www.grovestreams.com/developers/api.html
  * Passing in a custom name for a new component
  * Passing in a component template ID to be used for new component definitions
  * Downloading a couple of commands during the sample upload and implements them.
  * Demonstrates an accurate way to poll and send samples periodically
  * Demonstrates send retries and Internet Connection Reset logic to ensure the
    STM32 stays connected and can regain connectivity after a network outage.
  * Printing verbose trace statements to the LCD (optional)
 The STM32 uses DHCP and DNS for a simpler network setup.

 License:
  Copyright (C) 2017 GroveStreams LLC.
  Licensed under the Apache License, Version 2.0 (the "License");
  you may not use this file except in compliance with the License.
  You may obtain a copy of the License at: http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  See the License for the specific language governing permissions and
  limitations under the License.
*/

#if !FEATURE_LWIP
#error [NOT_SUPPORTED] LWIP not supported for this target
#endif

#include "mbed.h"
#include "LcdDiscoF746NgTracer.h"
#include "GroveStreams.h"
#include "MbedJSONValue.h"

// GroveStreams Settings
const char gsApiKey[] = "YOUR_SECRET_API_KEY_HERE";   //Change This!!!
const char gsCompName[] = "STM32F746+Discovery";  //Optionally change. Set this to give your component a name when it initially registers. Encode special chars such as spaces.

const char gsCompTmplId[] = "";  //Optional. Tells GS what template to use when the feed initially arrives and a new component needs to be created.

//GroveStreams Stream IDs. Stream IDs tell GroveStreams which component streams the values will be assigned to.
//Don't change these unless you edit your GroveStreams component definition and change the stream IDs to match these.
const char gsStreamId1[] = "voltage";
const char gsStreamId2[] = "temperature";

// Other Settings
int updateFrequency = 20;    // Update frequency in seconds. Change this to change your sample frequency.

AnalogIn adc_temp(ADC_TEMP);
AnalogIn adc_vref(ADC_VREF);
DigitalOut myled(LED1);

int main()
{
    //Create Lcd class for verbose tracing
    LcdDiscoF746NgTracer lcd;
    
    //lastSuccessfulUploadTime is used for upload frequency.
    time_t lastSuccessfulUploadTime = 0;  

    lcd.printf("Starting...");

    GroveStreams groveStreams(gsApiKey, &lcd);

    const char* myMac = groveStreams.getMACAddress();

    while (true) {
        // Update sensor data to GroveStreams
        time_t seconds = time(NULL);

        if(seconds - lastSuccessfulUploadTime > updateFrequency) {
            lcd.clear();
            
            lcd.printf("Getting Samples...");
            
            //Assemble the samples into URL parameters which are seperated with the "&" character
            // Example: &s1=6.2&s2=78.231
            int temperature = adc_temp.read() * 100.0f;
            int voltage = adc_vref.read() * 100.0f;
            char samples[64] = {0};
            sprintf(samples, "&%s=%d&%s=%d", gsStreamId1, voltage, gsStreamId2, temperature);
            
            //Append on command requests (request stream values)
            //This will indicate to GroveStreams to return the last value
            // of each request stream during the sample upload
            strcat(samples, "&rsid=freq&rsid=led");
            
            char resultBuffer[700]= {0};

            //Sending Samples (and returning current command stream values)
            time_t connectAttemptTime = time(NULL);
            int sendResult = groveStreams.send(myMac, samples, gsCompName, gsCompTmplId, resultBuffer, sizeof resultBuffer);

            if (sendResult == 0) {
                lcd.printf("Send Successful");
                lastSuccessfulUploadTime = connectAttemptTime;
                
                //Handle command streams
                if (strlen(resultBuffer) > 0 && resultBuffer[0] == '{') {
                    MbedJSONValue mbedJson;
                    parse(mbedJson, resultBuffer);
                    
                    if (mbedJson.hasMember("freq") && mbedJson["freq"].get<int>() >= 10) {
                        //Change the update frequency
                        updateFrequency = mbedJson["freq"].get<int>();
                        lcd.printf("updateFrequency: %d", updateFrequency);
                    }
                    if (mbedJson.hasMember("led")) {
                        //Change LED
                        myled = mbedJson["led"].get<bool>() ? 1 : 0;
                        lcd.printf("LED: %s", mbedJson["led"].get<bool>() ? "On" : "Off");
                    }
                }
            } 
        }
    }
}

