/*
 STM32F746 GroveStreams Stream Feed via Ethernet

 This GroveStreams embed code is designed for an STM32 with Ethernet.
 It has been tested on the STM32F46NG Discovery board.
 A full "how to" guide for this sketh can be found at https://www.grovestreams.com/developers/getting_started_stm32F746_simple.html
 This example updates two stream feeds, temperature and voltage, via the
 GroveStreams API: https://www.grovestreams.com/developers/api.html#1

 License:
  Copyright 2017 GroveStreams LLC.
  Licensed under the Apache License, Version 2.0 (the "License");
  you may not use this file except in compliance with the License.
  You may obtain a copy of the License at: http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  See the License for the specific language governing permissions and
  limitations under the License.

 GroveStreams Setup:

 * Sign Up for Free User Account - https://www.grovestreams.com
 * Create a GroveStreams organization
 * Enter your GroveStreams secret api key under "GroveStreams Settings" in this code
 *    (The api key can be retrieved from within a GroveStreams organization: click the Api Keys toolbar button,
 *     select your Api Key, and click View Secret Key. Paste the Secret Key below)

 STM32 Requirements:

 * STM32 with Ethernet capabilities
 * mbed IDE and compiler: https://developer.mbed.org

 Network Requirements:

 * Ethernet port on Router
 * DHCP enabled on Router
 * Unique MAC Address and IP Address for STM32

*/

#if !FEATURE_LWIP
#error [NOT_SUPPORTED] LWIP not supported for this target
#endif

#include "mbed.h"
#include "EthernetInterface.h"
#include "NetworkAPI/tcp/socket.hpp"

// GroveStreams Settings
const char gsApiKey[] = "YOUR_SECRET_API_KEY_HERE";   //Change This!!!
const char gsCompName[] = "STM32F746+Discovery";  //Optionally change. Set this to give your component a name when it initially registers. Encode special chars such as spaces.

//GroveStreams Stream IDs. Stream IDs tell GroveStreams which component streams the values will be assigned to.
//Don't change these unless you edit your GroveStreams component definition and change the stream IDs to match these.
const char gsStreamId1[] = "voltage";
const char gsStreamId2[] = "temperature";

EthernetInterface eth;


AnalogIn adc_temp(ADC_TEMP);
AnalogIn adc_vref(ADC_VREF);


int main()
{
    eth.init();
    eth.connect();
 
    //myIpAddress is used for X-Forwarded-For
    //myMax is used for the component ID
    const char* myIpAddress = eth.getIPAddress();
    const char* myMac = eth.getMACAddress();

    network::tcp::Socket socket;
    while (true) {
      
        //Assemble Samples
        int temperature = adc_temp.read() * 100;
        int voltage = adc_vref.read() * 100;
        char samples[50] = {0};
        sprintf(samples, "&%s=%d&%s=%d", gsStreamId1, voltage, gsStreamId2, temperature);
    
        //You may need to increase the size of sbuffer if any other char array sizes have increased
        char sbuffer[512] = {0};
        char rbuffer[700];
        
        //Assemble the HTTP PUT string (All on the URL GroveStreams Feed PUT API)
        sprintf(sbuffer, "PUT /api/feed?compId=%s&compName=%s&api_key=%s%s HTTP/1.1\r\nHost:\r\nConnection:close\r\nX-Forwarded-For:%s\r\n\r\n",
                myMac, gsCompName, gsApiKey, samples, myIpAddress);
        
        //Send the Samples into your GroveStreams organization streams
        socket.open();
    
        socket.connect("grovestreams.com", 80);
    
        socket.write(sbuffer, strlen(sbuffer));
        
        socket.read(rbuffer, sizeof rbuffer);
     
        socket.close();
        
        wait(20);
    }
    
    eth.disconnect();
}

