#include "mbed.h"
#include "SDFileSystem.h"
#include "ST7565_LCD.h"
#include "QEI.h"

#define BAT_GAIN    6.80
#define BAT_OFFS    0.0

PwmOut      BEEP        (D2);      // Buzzer/speaker (PWM output)
PwmOut      BKL         (D3);      // LCD backlight control (PMW output)
DigitalOut  KAL         (D0);      // Keep-Alive/turn-off
DigitalOut  BTC         (D2);      // Aux BT module control
DigitalIn   Button      (D4);      // Pushbutton (digital input)
AnalogIn    BATT        (A0);      // Battery monitor
AnalogIn    ALS         (A1);      // Ambient Light sensor
 
SDFileSystem sd(PB_15, PB_14, PB_13, PB_1, "sd"); // MOSI, MISO, SCK, CS

// Quadrature encoder
QEI Wheel(D5, D6, NC, 16);

// Tickers
Ticker Sec_Beat;            // Timer ticker
Ticker Display_Refresh;     // Display refresh ticker

//Serial ports
Serial PC(USBTX, USBRX);    // Virtual COM via USB
//Serial BT_module(D1, D0);    // BlueTooth module 


extern unsigned int buffer[128*64/8];   // RAM buffer used by LCD
time_t seconds;     // timestamp
char Text[40]="";   // Text string used by LCD 
float Vbatt, ALLevel;   // battery votage and ambient light level

// ------------------- Prototypes ----------------------- 
void Timer_tick(void);
void Update_Display(void);
void Set_Time(void);
void Read_Voltages(void);
void PowerOff(void);

int main() 
{

    KAL = 1;      // ensure self-sustained power
    //Button.mode(PullUp);      // enable pushbutton pull-up
    BKL.period_ms(3);         // set LCD backlight PWM
    BKL.write(1.0);  
    BEEP.period_us(2300);      // set initial buzzer period and duty-cycle
    BEEP.write(0.2);
    Wheel.reset();        // clear encoder
    LCD_reset();   

    // splash screen with date and time
    sprintf(Text,__DATE__);
    LCD_drawstring(buffer, 60, 5, Text);
    sprintf(Text,__TIME__);
    LCD_drawstring(buffer, 78, 6, Text);  
    LCD_write_buffer(buffer);


    // enable LCD refresh ticker        
    Display_Refresh.attach(&Update_Display, 0.3);

    if(Button)   // if pushbutton is pressed
        Set_Time();  // set RTC time and date  

    wait(2);
    Clear_buffer(buffer);
    BKL.write(0.5);
    
    //PowerOff();   // Power-off test


    // enable sec-beat ticker  
    Sec_Beat.attach(&Timer_tick, 1);
      


    printf("Hello World!\n");    
    mkdir("/sd/system", 0777);
    FILE *fp = fopen("/sd/system/sdlog.txt", "w");
    if(fp == NULL) {
        error("Could not open file for write\n");
    }
    fprintf(fp, "Hello fun SD Card World!");
    fclose(fp); 
    printf("Goodbye World!\n");
    
    while(1) 
    {
       // dance 
    }    
    
    
}


//===========================================================================

// ------------- Called every second ----------------------

void Timer_tick() 
{ 
    seconds = time(NULL); 
    strftime(Text, 50, "%d-%b-%Y  %H:%M:%S", localtime(&seconds));
    LCD_drawstring(buffer, 0, 0, Text);

    // read voltages 
    Read_Voltages();
    
    // write values to buffer
    sprintf(Text,"VBATT= %4.2f", Vbatt);
    LCD_drawstring(buffer, 0, 2, Text);    
    sprintf(Text,"ALL= %4.2f", ALLevel);
    LCD_drawstring(buffer, 0, 3, Text); 
                              
    // Send data to COM port
    //PC.printf("Tset= %3.0f, Tpit= %3.0f, Tmeat= %3.0f, PWM= %3.0f\n",Tset, Tpit, Tmeat, FanSpeed); 

    //LCD_write_buffer(buffer);   // LCD update
                         
    return;
}


//---------------------------------------------------------------------------
void Update_Display(void)
{
    LCD_write_buffer(buffer);   // LCD update 
    return;  
}

//---------------------------------------------------------------------------
void Set_Time(void)
{
    uint8_t Year=0, Month=0, Day=0, Hours=0, Mins=0, Secs=0;
    time_t seconds;
    struct tm t;

    sprintf(Text,"TIME & DATE SETTING");
    LCD_drawstring(buffer, 0, 0, Text); 
    
    // Set year
    while(Button);
    wait_ms(50);  

    while(!Button)
    { 
        if(int(Wheel.getPulses())<0)
            Wheel.reset();
        Year = (uint8_t)(Wheel.getPulses());
        
        if(Year>99) 
            Wheel.reset();
             
        sprintf(Text, "Year: %2d", Year);
        LCD_drawstring(buffer, 0, 2, Text);
   
    }

    // Set month
    while(Button);
    wait_ms(50);  
    Wheel.reset();
    while(!Button)
    { 
        if(int(Wheel.getPulses())<0)
            Wheel.reset();
        Month = (uint8_t)(Wheel.getPulses()/2);
        
        if(Month>11) 
            Wheel.reset();
            
        sprintf(Text, "Month: %2d", Month+1);
        LCD_drawstring(buffer, 0, 3, Text);
   
    }


    // Set day
    while(Button);
    wait_ms(50);  
    Wheel.reset();
    while(!Button)
    { 
        if(int(Wheel.getPulses())<0)
            Wheel.reset();
        Day = (uint8_t)(Wheel.getPulses()/2);
        
        if(Day>30) 
            Wheel.reset();
            
        sprintf(Text, "Day: %2d", Day+1);
        LCD_drawstring(buffer, 0, 4, Text);
   
    }

    // Set hours
    while(Button);
    wait_ms(50);  
    Wheel.reset();
    while(!Button)
    { 
        if(int(Wheel.getPulses())<0)
            Wheel.reset();
        Hours = (uint8_t)(Wheel.getPulses()/2);
        
        if(Hours>22) 
            Wheel.reset();
            
        sprintf(Text, "Hours: %2d", Hours+1);
        LCD_drawstring(buffer, 0, 5, Text);
   
    }
    Hours++;

    // Set minutes
    while(Button);
    wait_ms(50);  
    Wheel.reset();
    while(!Button)
    { 
        if(int(Wheel.getPulses())<0)
            Wheel.reset();
        Mins = (uint8_t)(Wheel.getPulses()/2);
        
        if(Mins>59) 
            Wheel.reset();
            
        sprintf(Text, "Minutes: %2d", Mins);
        LCD_drawstring(buffer, 0, 6, Text);
   
    }

    t.tm_year = Year + 100;
    t.tm_mon  = Month;
    t.tm_mday = Day + 1;
    t.tm_hour = Hours;
    t.tm_min  = Mins;
    t.tm_sec  = Secs;

    seconds = mktime(&t);
    set_time(seconds);
    
    return;
}    


//---------------------------------------------------------------------------
void Read_Voltages(void)
{

    double ADC_value;
    uint8_t smooth = 10; // Number of samples to smooth
    uint8_t i;
    
    // Read battery voltage
    
    ADC_value = BATT.read();    // cleanup
    wait_ms(50);
    ADC_value = 0;    
    for(i=0;i<smooth;i++) 
        ADC_value += BATT.read();       

    ADC_value = ADC_value/smooth;
    Vbatt = (float)(ADC_value*BAT_GAIN)+BAT_OFFS;
    
    
    // Read Ambient Light Level
    
    ADC_value = ALS.read();    // cleanup
    wait_ms(50);
    ADC_value = 0;    
    for(i=0;i<smooth;i++) 
        ADC_value += ALS.read();       

    ADC_value = ADC_value/smooth;
    ALLevel = (float)(ADC_value);      
   
return;
}


//--------------------------------------------------------------------------- 
void PowerOff(void) 
{
    BKL.write(1);
    Clear_buffer(buffer);
    sprintf(Text,"POWERING OFF");
    LCD_drawstring(buffer, 20, 3, Text); 
    LCD_write_buffer(buffer);
    wait(2);
    
    KAL = 0;
}