#include "mbed.h"
#include "HTTPClient.h"

#include "hcsr04.h"
#include "GroveTemp.h"
#include "GPS.h"

#include "Watchdog.h"

#include "SHA1.h"
#include "HMAC.h"

#define TRIG_PIN A2
#define ECHO_PIN A1
#define MOISTURE_PIN A3
#define HTTP_ENDPOINT "http://104.236.187.29/sensor"
#define SENSOR_NAME "Throop"

#define CELLULAR_NETWORK 1

#ifdef CELLULAR_NETWORK
#include "MDM.h"
#define SIMPIN      NULL
#define APN         "Broadband"
#define USERNAME    NULL
#define PASSWORD    NULL 
#else
#include "EthernetInterface.h"
#endif

int main() 
{   
    Serial console(USBTX, USBRX); // tx, rx
    char macAddress[6];
    mbed_mac_address(macAddress);
    console.printf("MAC address is %02X%02X%02X%02X%02X%02X\r\n", macAddress[0], macAddress[1], macAddress[2], macAddress[3], macAddress[4], macAddress[5]);
    
    Watchdog wdt;
    
    GPSSerial gps;

#ifdef CELLULAR_NETWORK
    MDMSerial mdm;
    //mdm.setDebug(4);
    if (!mdm.connect(SIMPIN, APN,USERNAME,PASSWORD))
        return -1;
#else
    EthernetInterface ethernet;
    ethernet.init(); // DHCP
    if (ethernet.connect() != 0)
        return -1;
#endif
    HTTPClient http;
    
    bool first = true;
    
    int gpsRet;
    char gpsBuf[512];
    double lat = 0, lon = 0;
    
    HCSR04 distS(TRIG_PIN, ECHO_PIN);
    double distV;
    char str[512];
    
    GroveTempSensor tempS;
    double tempV;
    
    AnalogIn moistureS(MOISTURE_PIN);
    double moistureV;
    
    wdt.kick(30.0);
    
    while (true) {
        // Distance sensor
        distS.start();
        wait_ms(500); 
        distV = distS.get_dist_cm();
        console.printf("Distance reading: %0.2f cm\r\n", distV);
        
        // Temperature sensor
        tempV = tempS.getTemp();
        console.printf("Temperature reading: %0.2f C\r\n", tempV);
        
        // Moisture sensor
        moistureV = 1.0 - moistureS;
        console.printf("Moisture reading: %0.1f%%\r\n", moistureV * 100);
        
        // GPS
        console.printf("Checking for GPS messages\r\n");
        while ((gpsRet = gps.getMessage(gpsBuf, sizeof(gpsBuf))) > 0)
        {
            int len = LENGTH(gpsRet);
            char ch;
            //console.printf("NMEA: %.*s\r\n", len-2, gpsBuf); 
            if ((PROTOCOL(gpsRet) == GPSParser::NMEA) && (len > 6)
             && strncmp(gpsBuf, "$G", 2) == 0
             && strncmp(gpsBuf + 3, "GLL", 3) == 0
             && gps.getNmeaAngle(1,gpsBuf,len,lat)
             && gps.getNmeaAngle(3,gpsBuf,len,lon)
             && gps.getNmeaItem(6,gpsBuf,len,ch)
             && ch == 'A')
            {
                console.printf("GPS Location: %.5f %.5f\r\n", lat, lon);
                break;
            }
        }
        wdt.kick();
        
        // Combine readings in JSON
        char json[255];
        snprintf(json, 255,
            "{\"distance\":%0.1f,\"temperature\":%0.1f,\"moisture\":%0.2f,\"latitude\":%0.5f,\"longitude\":%0.5f,\"first\":%d}",
            distV, tempV, moistureV, lat, lon, first);
           
        // Compute a MAC (message authentication code) of the value,
        // using our MAC (media access control) address as a key.
        // Yay acronyms.
        SHA1 hasher;
        HMAC hmacer(&hasher, (uint8_t *)macAddress, sizeof(macAddress));
        hmacer.update((uint8_t *)json, strlen(json));
        uint8_t hmac[20];
        hmacer.finalize(hmac);
        char hmacHex[41], *hmacHexPtr = hmacHex;
        for (int i = 0; i < 20; i++)
            hmacHexPtr += sprintf(hmacHexPtr, "%02X", hmac[i]);
        
        //POST data
        HTTPMap map;
        HTTPText inText(str, 512);
        map.put("name", SENSOR_NAME);
        map.put("value", json);
        map.put("mac", hmacHex);
        console.printf("\r\nTrying to POST data to server...\r\n");
        int ret = http.post(HTTP_ENDPOINT, map, &inText);
        if (!ret)
        {
          console.printf("Executed POST successfully - read %d characters\r\n", strlen(str));
          console.printf("Result: %s\r\n", str);
        }
        else
        {
          console.printf("Error - ret = %d - HTTP return code = %d\r\n", ret, http.getHTTPResponseCode());
        }
        
        first = false;
        wdt.kick();
        wait_ms(10000);
        wdt.kick();
    }
}

