/*
 * Copyright (c) 2017-2019 Future Electronics
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#ifndef TEMP_SENSOR_H_
#define TEMP_SENSOR_H_

#include <stdint.h>
#include "Sensor.h"
#include "Hs3001Driver.h"

/** Tempereature Sensor based on HS3001 chip
 */
class TempSensor : public Sensor<int16_t> {
public:
    /** Create and initialize sensor interface.
     *
     * @param i2c I2C bus to use
     * @param address I2C address to use to select sensor on a bus
     */
    TempSensor(I2C &i2c, uint8_t address) : _driver(i2c, address) {}

    /** Schedule measurement process.
     */
    virtual void start(EventQueue& ev_queue)
    {
        // start new measurement cycle
        _driver.start_conversion();
        // set up updater() function to be called every second (1000ms)
        ev_queue.call_every(1000, callback(this, &TempSensor::updater));
    }

protected:
    void updater()
    {
        uint16_t humidity;
        int16_t temperature;

        // read last measured temperature value and update
        // characteristic value if new value is available
        if (_driver.read(humidity, temperature) == Hs3001Driver::STATUS_OK) {
            update_value(temperature);
        }

        // start new measurement cycle
        _driver.start_conversion();
    }

    Hs3001Driver _driver;
};

#endif // TEMP_SENSOR_H_
