#include "main.h"
#include "GSM_Wrapper.h"
#include "WakeUp.h"
#include "pH_Sensor.h"
#include "storage.h"
#include "GPS_Wrapper.h"
#include "Temp_Sensor.h"
#include <string>

#define PC_BAUD 115200  // High baud rate for PC serial connection
#define GSM_INITIAL_DELAY 20 // Delay to allow GSM to initally turn on

// Global sensor objects
pH_Sensor pH;       // pH Sensor object - initialized in setup()
GPS_Sensor GPS;     // GPS sensor object - initialized in setup()
GSM_Sensor GSM;     // GSM sensor object - initialized in setup()
Temp_Sensor Temp;   // Temperature sensor object - initialized in setup()
Storage storage;    // Storage sensor object - initialized in setup()
Serial pcSerial(USBTX, USBRX);  // PC serial communication -- initialized in setup(). Used for debugging

// Function that sets up the PC serial connection. Used for printing debug statements
void setupPCSerial() {
    wait(2);
    pcSerial.baud(PC_BAUD);
    pcSerial.printf("\n\n PC Serial connection established at 115200 baud.\n");
    wait(2);
}

/* Function: setup
 * ---------------
 * Sets up all the sensors and PC serial.
 */
void setup() {
    setupPCSerial();
    GPS.setup();
    pH.setup();
    // Delay to allow GSM to turn on initially
    wait(GMS_INITIAL_DELAY);
    // Turn the GSM back off
    GSM.changePowerState();
} 

/* Function: enterSleep
 * Enters deep sleep. Turns off all necessary
 * sensors, and resets the clock after waking
 * up (and turns the GPS on so that it can start getting
 * a fix).
 */
void enterSleep(int sec)
{
    GPS.turnOff();
    wait(2);
    WakeUp::set(sec);
    deepsleep();
    WakeUp::resetADC();
    GPS.turnOn();
    wait(2);
}

/* Function: read
 * --------------
 * Calls the read function for each sensor, and adds it
 * to the reading buffer. Writes it to the perminant storage.
 */
void read(struct reading& lastReadingBuffer)
{
    printf("~~~~~[pH]~~~~~\n");
    lastReadingBuffer.pH = pH.read();
    printf("~~~~~[Temperature]~~~~~\n");
    lastReadingBuffer.temperature = Temp.read();
    printf("~~~~~[GPS]~~~~~\n");
    GPS.read(lastReadingBuffer);
    storage.write((uint8_t *) &lastReadingBuffer, sizeof(struct reading));
    GSM.changePowerState();
}

/* Function: send
 * --------------
 * Sends the data over GSM by reading from the
 * nonvolitile storage. Only if it succeeds does it
 * change the toSend bool to false (thus attempting to send
 * the next time if it fails).
 */
void send(int& nreadings, bool& toSend)
{
    struct reading data[nreadings];
    storage.read((uint8_t*) data);
    size_t bufsize = SIZE_OF_ENTRY*nreadings+1;
    char buffer[bufsize];
    for(int i = 0; i < nreadings; i++) {
        int n = sprintf(buffer+SIZE_OF_ENTRY*i, "%8f %8f %10f %10f %d %d %d %d %d\t", data[i].temperature, data[i].pH, data[i].latitude, data[i].longitude, data[i].day,
                        data[i].month, data[i].year, data[i].hour, data[i].minutes);
    }
    if(GSM.send((uint8_t*) buffer, bufsize)) {
        toSend = false;
        nreadings = 0;
        storage.reset();
    }
}

/* Function: main
 * --------------
 * Never returns. Calls setup, initializes nreadings and toSend,
 * then repeatedly reads the sensors and sends the data (if 
 * the number of readings is correct) before entering sleep.
 */
int main()
{
    setup();
    int nreadings = 0;
    bool toSend = false;
    while (true) {
        struct reading lastReadingBuffer;
        read(lastReadingBuffer);
        nreadings++;
        if(nreadings == N_READINGS_PER_SEND) toSend = true;
        if(toSend) send(nreadings, toSend);
        GSM.changePowerState();
        wait(2);
        enterSleep(N_SECONDS_SLEEP);
    }
}