/*
MIT License

Copyright (c) 2017 Fast Sensing Inc.

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in all
copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
SOFTWARE.
*/

#ifndef FASTSENSING_H
#define FASTSENSING_H

#include "mbed.h"
#include "EthernetInterface.h"

/** Fastsensing Data Update class.
 *  You can update data to your Fastsensing account used this class. This Liberary is tested on mbedOS 5.3.
 *
 * Example:
 * @code
 * #include "mbed.h"
 * #include "Fastsensing.h"
 * 
 * Fastsensing fast;
 * AnalogIn illu(p20);
 * DigitalOut myled(LED1);
 * 
 * int main()
 * {
 *  float lx = illu * 3.3 / 3 * 1000;
 *  printf("lx : %f\n", lx);
 *  for(int i = 0; i < 5; i++) {
 *     fast.updateData("UIYCOQAATQAO5AF7", channel[0], lx,1 );
 *     wait(5.0);
 *  }
 *    while(1) {
 *      myled = 1;
 *      wait(0.5);
 *      myled = 0;
 *      wait(0.5);
 *  }
 * }
 * @endcode
 */
class Fastsensing {
    private:
        EthernetInterface eth;  //create eth instance from Ethernet
        
    public:
        /**
        * Only one data update.
        * @param *deviceId It is issued your dashboard on fastsening. Device ID length is 16 characters.
        * @param *channelId It is issued your dashboard on fastsensing. Channel ID length is 8characters.
        * @param data It is the data you want to update.
        * @param display 1 is display your sent data and recieve data. 0 is not.
        * @return
        *   0 on success, -1 on deviceId length error, -2 on channelId length error
        */
        int updateData(char *deviceId, char *channelId, double data, int display);
        /**
        * Only one data update.
        * @param *deviceId It is issued your dashboard on fastsening. Device ID length is 16 characters.
        * @param *channelId[3] These are issued your dashboard on fastsensing. Channel ID length is 8characters.
        * @param data[3] These are the data you want to update.
        * @param display 1 is display your sent data and recieve data. 0 is not.
        * @return
        *   0 on success, -1 on deviceId length error, -2 on channelId length error
        */
        int updateDataAll(char *deviceId, char *channelId[3], float data[3], int display);

};

#endif   