//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
//                                                                     
//                                          
//                           IUT DE BREST GEII                                            
//
//                                                                     
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
//      
//                       
//   target:            Carte Nucleo L073RZ et SX1272 mbed shield (LoRa)
//   Compiler:          Mbed 
//   Date:              12/2019
//
//   Authors:           Mohammed AL GHAMDI, Gwénolé MOAl, Gurvan PAUL, Thomas UGUEN
//
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
//
//   Purpose:           Ce programme permet de recevoir la valeur de l'écart (longitude , latitude, 
//                      altitude) calculé auparavant via la balise fixe puis de l'utiliser afin de 
//                      corriger la position de la balise mobile.
//          
//
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

#include "mbed.h"
#include "sx1276-hal.h"
#include "SerialGPS.h"
#include "device.h"


// Définissez cet indicateur sur «1» pour afficher les messages de débogage sur la console

#define DEBUG_MESSAGE   1

// Réglez cette variable sur «1» pour utiliser la modulation LoRa ou sur «0» pour utiliser la modulation FSK

#define USE_MODEM_LORA  1
#define USE_MODEM_FSK   !USE_MODEM_LORA

// Principaux réglages des données des cartes et du shield (LoRa) 

#define RF_FREQUENCY                                    868000000               // Hz
#define TX_OUTPUT_POWER                                 14                      // 14 dBm

#if USE_MODEM_LORA == 1                                                         // Modem LoRa

#define LORA_BANDWIDTH                              0                          // [0: 125 kHz,
                                                                                //  1: 250 kHz,
                                                                                //  2: 500 kHz,
                                                                                //  3: Reserver]
#define LORA_SPREADING_FACTOR                       7                           // [SF7..SF12]
#define LORA_CODINGRATE                             1                           // [1: 4/5,
                                                                                //  2: 4/6,
                                                                                //  3: 4/7,
                                                                                //  4: 4/8]
#define LORA_PREAMBLE_LENGTH                        8                           // Idem Tx/Rx
#define LORA_SYMBOL_TIMEOUT                         5                           // Symbole
#define LORA_FIX_LENGTH_PAYLOAD_ON                  false
#define LORA_FHSS_ENABLED                           false
#define LORA_NB_SYMB_HOP                            4
#define LORA_IQ_INVERSION_ON                        false
#define LORA_CRC_ENABLED                            true

#elif USE_MODEM_FSK == 1                                                        // Modem FSK

#define FSK_FDEV                                    25000                       // Hz
#define FSK_DATARATE                                19200                       // bps
#define FSK_BANDWIDTH                               50000                       // Hz
#define FSK_AFC_BANDWIDTH                           83333                       // Hz
#define FSK_PREAMBLE_LENGTH                         5                           // Idem Tx/Rx
#define FSK_FIX_LENGTH_PAYLOAD_ON                   false
#define FSK_CRC_ENABLED                             true

#else
#error "Veuillez définir un modem LoRa ou FSK dans les options du compilateur."
#endif


#define RX_TIMEOUT_VALUE                                3500                    // en ms
#define BUFFER_SIZE                                     32                      // Définition de la taille du Buffer

#if( defined ( TARGET_KL25Z ) || defined ( TARGET_LPC11U6X ) )
DigitalOut led(LED2);
#else
DigitalOut led(LED1);
#endif
DigitalIn button(USER_BUTTON);

// Déclarations globales des variables

typedef enum {
    LOWPOWER =0,
    IDLE,
    RX,
    RX_TIMEOUT,
    RX_ERROR,
} AppStates_t;

// Déclaration du Buffer et taille

volatile AppStates_t State = IDLE;
uint16_t BufferSize = BUFFER_SIZE;
char Buffer[BUFFER_SIZE];
char i,j,v;
float ecartlong;
float ecartlat;
float ecartalt;
float longitude;
float latitude;
float altitude;



// Initialisation RssiValue et SnrValue à "0" (variables non utilisées)

int16_t RssiValue = 0.0;
int8_t SnrValue = 0.0;

// Initialisation Radio
 
static RadioEvents_t RadioEvents;
SX1276MB1xAS Radio( NULL );

// Fonctions du Programme                
                    
                     
void OnRxDone( uint8_t *payload, uint16_t size, int16_t rssi, int8_t snr)       // Message dit "reçu"
{
    Radio.Sleep( );
    BufferSize = size;
    memcpy( Buffer, payload, BufferSize );                                      // Cette fonction permet de copier un bloc de mémoire spécifié par le paramètre payload, et dont la taille est spécifiée via le paramètre BufferSize
    RssiValue = rssi;
    SnrValue = snr;
    State = RX;
    debug_if( DEBUG_MESSAGE, "> OnRxDone\n\r" );                                // Message reçu sur un moniteur série
}

void OnRxTimeout( void )                                                        // Aucune reception de message, buffer vide
{
    Radio.Sleep( );
    Buffer[ BufferSize ] = 0;                                                   // Buffer vide 
    State = RX_TIMEOUT;
    debug_if( DEBUG_MESSAGE, "> OnRxTimeout\n\r" );                             // Message reçu sur un moniteur série
}

void OnRxError( void )                                                          // Erreur de reception de message
{
    Radio.Sleep( );
    State = RX_ERROR;
    debug_if( DEBUG_MESSAGE, "> OnRxError\n\r" );                               // Message reçu sur un moniteur série
}

// Déclaration des liaisons série (Hyperterminal et GPS)

Serial pc(USBTX, USBRX, 9600);
SerialGPS gps( A4, A5, 4800);

// Programme principal

int main() {
    
    pc.format(8,Serial::None,1);
                
    debug( "\n\r\n\r TEST reception module LoRA \n\r" );                        // Message d'information initialisation du programme
    debug( "> Carte Nucleo-L073RZ <\r\n" );

    
    RadioEvents.RxDone = OnRxDone;                                              // Initialisation de la conduite Radio
    RadioEvents.RxError = OnRxError;
    RadioEvents.RxTimeout = OnRxTimeout;
    Radio.Init( &RadioEvents );

    while( Radio.Read( REG_VERSION ) == 0x00  ) {                               // Vérification de la connection avec la carte
        debug( "La radio n'a pas pu etre detectee !\n\r", NULL );
        wait( 1 );
    }
    Radio.SetChannel( RF_FREQUENCY );

#if USE_MODEM_LORA == 1                                                         // Définition des réglages du Module LoRa, si le module est utilisé

    debug_if( LORA_FHSS_ENABLED, "\n\n\r             > LORA FHSS Mode < \n\n\r");
    debug_if( !LORA_FHSS_ENABLED, "\n\n\r             > LORA Mode < \n\n\r");

    Radio.SetTxConfig( MODEM_LORA, TX_OUTPUT_POWER, 0, LORA_BANDWIDTH,
                       LORA_SPREADING_FACTOR, LORA_CODINGRATE,
                       LORA_PREAMBLE_LENGTH, LORA_FIX_LENGTH_PAYLOAD_ON,
                       LORA_CRC_ENABLED, LORA_FHSS_ENABLED, LORA_NB_SYMB_HOP,
                       LORA_IQ_INVERSION_ON, 2000000 );

    Radio.SetRxConfig( MODEM_LORA, LORA_BANDWIDTH, LORA_SPREADING_FACTOR,
                       LORA_CODINGRATE, 0, LORA_PREAMBLE_LENGTH,
                       LORA_SYMBOL_TIMEOUT, LORA_FIX_LENGTH_PAYLOAD_ON, 0,
                       LORA_CRC_ENABLED, LORA_FHSS_ENABLED, LORA_NB_SYMB_HOP,
                       LORA_IQ_INVERSION_ON, true );

#elif USE_MODEM_FSK == 1                                                        // Définition des réglages du Module FSK, si le module est utilisé

    debug("\n\n\r              > FSK Mode < \n\n\r");
    Radio.SetTxConfig( MODEM_FSK, TX_OUTPUT_POWER, FSK_FDEV, 0,
                       FSK_DATARATE, 0,
                       FSK_PREAMBLE_LENGTH, FSK_FIX_LENGTH_PAYLOAD_ON,
                       FSK_CRC_ENABLED, 0, 0, 0, 2000000 );

    Radio.SetRxConfig( MODEM_FSK, FSK_BANDWIDTH, FSK_DATARATE,
                       0, FSK_AFC_BANDWIDTH, FSK_PREAMBLE_LENGTH,
                       0, FSK_FIX_LENGTH_PAYLOAD_ON, 0, FSK_CRC_ENABLED,
                       0, 0, false, true );

#else

#error "Veuillez définir un modem dans les options du compilateur."             // Message reçu si le LoRa ou FSK n'est pas utilisé

#endif

    debug_if( DEBUG_MESSAGE, "Debut de la boucle\r\n" );                        // Début de la communication entre les deux cartes

    led = 0;

    Radio.Rx( RX_TIMEOUT_VALUE );     

    while( 1 ) {          
    
                        
        switch( State ) {
        
             case RX:                                                           // Réception du message du Buffer
                if( BufferSize > 0 ) {                                          // Buffer rempli                   
                 if (gps.sample()) {
                pc.printf("Trame recue :");
                pc.printf("sats %d, long %f, lat %f, alt %f, geoid %f, time %f\n\r", gps.sats, gps.longitude, gps.latitude, gps.alt, gps.geoid, gps.time);                                     
                }                                                        
                    led = !led;
                    //debug( "Reception...\r\n" ); 
                    //printf( "Donnee recu : %s\r\n", Buffer );  // On affiche le contenu du Buffer                                
                    Radio.Rx( RX_TIMEOUT_VALUE );                                 
                    State = LOWPOWER;
                    wait_ms(1000);               
                                                                 
                    if(Buffer[i]=='1'){                                         // On compare la valeur du Buffer afin d'identifier si il s'agit de la longitude/latitude/altitude
                    wait(3);                                                    // Si Buffer=1 alors il s'agit de la longitude                       
                    sscanf(Buffer,"%f",&ecartlong);                             // On associe la valeur du Buffer à une variable ecart
                    longitude=((gps.longitude*1000000)+ecartlong)/1000000;      // On corrige la longitude avec cette écart           
                    pc.printf(" Longitude corrigee %f\n\n\r", longitude);       // longitude corrigée                   
                    }
                    
                    if(Buffer[i]=='2'){                                         // On compare la valeur du Buffer afin d'identifier si il s'agit de la longitude/latitude/altitude
                    wait(3);                                                    // Si Buffer=2 alors il s'agit de la latitude         
                    sscanf(Buffer,"%f",&ecartlat);                              // On associe la valeur du Buffer à une variable ecart      
                    latitude=((gps.latitude*1000000)+ecartlat)/1000000;         // On corrige l'altitude avec cette écart           
                    pc.printf("Latitude corrigee %f\n\n\r",latitude);           // latitude corrigée                   
                    }
                    
                    if(Buffer[i]=='3'){                                         // On compare la valeur du Buffer afin d'identifier si il s'agit de la longitude/latitude/altitude
                    wait(3);                                                    // Si Buffer=3 alors il s'agit de l'altitude                  
                    sscanf(Buffer,"%f",&ecartalt);                              // On associe la valeur du Buffer à une variable ecart
                    altitude=((gps.alt*1000000)+ecartalt)/1000000;              // On corrige la latitude avec cette écart            
                    pc.printf("Altitude corrigee %f\n\n\r", altitude);          // altitude corrigée                                        
                    }
                                                  
                    }
                break;
                
            case RX_TIMEOUT:                                                    // Réception null, pas de connection
                Radio.Rx( RX_TIMEOUT_VALUE );
                State = LOWPOWER;
                break;
                
            case RX_ERROR:                                                      // Réception contenant une ou plusieurs erreur(s) 
                debug("Donnee recu avec erreur(s)\r\n"); 
                wait_ms( 2000 );
                State = LOWPOWER;
                break;
                
            case LOWPOWER:
                break;
                
            default:
                State = LOWPOWER;
                break;
        }               
                      
            
            
    }
}

        
    
