/* Display library for Adafruit 358 TFT display written for the FRDM-K64F
 * 
 * Copyright (c) 2014 Brian Mazzeo
 * Released under the MIT License: http://mbed.org/license/mit
 *
 * This 1.8" display has 128x160 color pixels
 */

#include "mbed.h"
#include "Adafruit_358.h"


//Constructor
Adafruit_358::Adafruit_358(PinName MOSI, PinName MISO, PinName SCLK, PinName CS, PinName RESET, PinName DC, const char *name)
    :  GraphicsDisplay(name), _spi(MOSI, MISO, SCLK), _cs(CS), _reset(RESET), _dc(DC) 
{   
    char_x = 0;
    char_y = 0;
    screen_reset();
}

int Adafruit_358::width()
{
    return 128;    
}

int Adafruit_358::height()
{
    return 160;    
}


void Adafruit_358::wr_cmd(unsigned char cmd)
{
    _dc = 0;
    _cs = 0;
    _spi.write(cmd);
    _dc = 1;
};
    

void Adafruit_358::screen_reset()
{
    _spi.format(8,3);
    _spi.frequency(10000000);
    _cs = 1;
    _dc = 1;
    
    _reset = 1;                // Reset the display
    wait_us(500);
    _reset = 0;
    wait_us(500);
    _reset = 1;
    
    wait_ms(5);
    
    wr_cmd(ST7735_SWRESET);           // SW reset
    wait_ms(150);
    
    wr_cmd(ST7735_SLPOUT);           // Sleep off
    wait_ms(255);
       
    wr_cmd(ST7735_FRMCTR1);         // Frame rate control
    _spi.write(0x01);                   
    _spi.write(0x2C);                   
    _spi.write(0x2D);                 
    
    wr_cmd(ST7735_FRMCTR2);
    _spi.write(0x01);
    _spi.write(0x2C);
    _spi.write(0x2D);
    
    wr_cmd(ST7735_FRMCTR3);
    _spi.write(0x01);
    _spi.write(0x2C);
    _spi.write(0x2D);
    _spi.write(0x01);
    _spi.write(0x2C);
    _spi.write(0x2D);

    wr_cmd(ST7735_INVCTR);
    _spi.write(0x07);
    
    wr_cmd(ST7735_PWCTR1);
    _spi.write(0xA2);
    _spi.write(0x02);
    _spi.write(0x84);
    
    wr_cmd(ST7735_PWCTR2);
    _spi.write(0xC5);
    
    wr_cmd(ST7735_PWCTR3);
    _spi.write(0x0A);
    _spi.write(0x00);
    
    wr_cmd(ST7735_PWCTR4);
    _spi.write(0x8A);
    _spi.write(0x2A);
    
    wr_cmd(ST7735_PWCTR5);
    _spi.write(0x8A);
    _spi.write(0xEE);
    
    wr_cmd(ST7735_VMCTR1);
    _spi.write(0x0E);
    
    wr_cmd(ST7735_INVOFF);
    
    wr_cmd(ST7735_MADCTL);          // Memory access control (directions)
    _spi.write(0xC8);                   // row addr/col adddr, bottom to top refresh
    
    wr_cmd(ST7735_COLMOD);          // Color mode
    _spi.write(0x05);                   // 16 bit color
    
    wr_cmd(ST7735_CASET);           // Column address set
    _spi.write(0x00);               
    _spi.write(0x02);
    _spi.write(0x00);
    _spi.write(0x7F+0x02);
    
    wr_cmd(ST7735_RASET);           // Row address set
    _spi.write(0x00);
    _spi.write(0x01);
    _spi.write(0x00);
    _spi.write(0x9F+0x01);
    
    wr_cmd(ST7735_NORON);           // Normal display on    
    wait_ms(10);
    
    wr_cmd(ST7735_DISPON);          // Main screen turn on
    wait_ms(100);
}

void Adafruit_358::pixel(int x, int y, int color)
{
    wr_cmd(0x2A);
    _spi.write(x >> 8);
    _spi.write(x);
    _cs = 1;
    wr_cmd(0x2B);
    _spi.write(y >> 8);
    _spi.write(y);
    _cs = 1;
    wr_cmd(0x2C);
    _spi.write(color >> 8);
    _spi.write(color & 0xff);
    _cs = 1;    
}

void Adafruit_358::WindowMax (void)
{
    window (0, 0, width(),  height());
}

void Adafruit_358::cls() {
    fillrect(0, 0, width(), height(), _background);
}


void Adafruit_358::set_font(unsigned char* f)
{
    font = f;
}

void Adafruit_358::fillrect(int x0, int y0, int x1, int y1, uint16_t color)
{
    for (int x_coord = x0; x_coord <= x1; x_coord++) {
        for (int y_coord = y0; y_coord <= y1; y_coord++) {
            pixel(x_coord, y_coord, color);
            }
        }
}    