/**
 * @author Hugues Angelis
 *
 * @section LICENSE
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 *
 * @section DESCRIPTION
 *
 * Vellerman VMA306 Ultrasonic Range Finder Library
 *
 * @Note :
 *  The purpose of this library is to adapt SRF05 library to allow full control of the sensor : 
 *      - allow cycle control (time between 2 pulses)
 *      - delayed trigerring (start periodical mesurement with a delay) 
 *  because in case of multiple US sensor use, pulse from a sensor can be detected by another like its own echo and thus interfere with mesurement.
 *
 *  Cycle time must be greater or equal to 50ms.
 *  After 30ms the sensor times out so 30ms output pulse means no detection
 *
 *  You may divide the echo pulse width by 58µs/cm to calculate the almost exact distance to the obstacle in centimeters.
 *
 *  To allow non blocking read you may use the global variable VMA306Flag witch is set to one each time a new mesure is ready. This variable must be cleared by user
 */

#ifndef _VMA306_H
#define _VMA306_H

/**
 * Includes : Mbed Library
 */
#include "mbed.h"

/**
 * VMA306 Ultrasonic Range Finder
 * similar to SRF05 : https://www.robot-electronics.co.uk/htm/srf05tech.htm
 * Poor documentation can be found here (FR) : https://www.gotronic.fr/art-capteur-a-ultrasons-vma306-26096.htm
 */
class VMA306 {
public:

    /** Constructor : Create a VMA306 object, connected to the specified pins
     *
     * @param trigger (PinName) : DigitalOut to trigger the VMA306
     * @param echo (PinName) : InterruptIn to measure the return pulse width
     */
    VMA306(PinName trigG, PinName echoG, PinName trigB, PinName echoB, PinName trigD, PinName echoD);
    
    /** Gives indication of a new mesurement
     *
     *  @return 1 if new data is ready, else 0
     */
    int isUSGReady ();
    
    /** A non-blocking function that will return the last measurement
     *
     * @returns floating point representation of distance in cm
     */
    float readUSG();
    /** Gives indication of a new mesurement
     *
     *  @return 1 if new data is ready, else 0
     */
    int isUSBReady ();
    
    /** A non-blocking function that will return the last measurement
     *
     * @returns floating point representation of distance in cm
     */
    float readUSB();
    /** Gives indication of a new mesurement
     *
     *  @return 1 if new data is ready, else 0
     */
    int isUSDReady ();
    
    /** A non-blocking function that will return the last measurement
     *
     * @returns floating point representation of distance in cm
     */
    float readUSD();

private :
    DigitalOut      _trig1;
    InterruptIn     _echo1;
    DigitalOut      _trig2;
    InterruptIn     _echo2;
    DigitalOut      _trig3;
    InterruptIn     _echo3;
    Timer           _timer;
    Ticker          _ticker;
    void            _rise1 (void);
    void            _fall1 (void);
    void            _rise2 (void);
    void            _fall2 (void);
    void            _rise3 (void);
    void            _fall3 (void);
    void            _startRanging (void);
    float           _dist1, _dist2, _dist3;
    int             _VMA306Flag1, _VMA306Flag2, _VMA306Flag3;
    int             _startValue1, _startValue2, _startValue3;
};

#endif
