/**
 * @brief       HX711.h
 * @details     24-Bit Analog-to-Digital Converter (ADC) for Weigh Scales.
 *              Function file.
 *
 *
 * @return      NA
 *
 * @author      Manuel Caballero
 * @date        11/September/2017
 * @version     11/September/2017    The ORIGIN
 * @pre         NaN.
 * @warning     NaN
 * @pre         This code belongs to Nimbus Centre ( http://www.nimbus.cit.ie ).
 */

#include "HX711.h"


HX711::HX711 ( PinName PD_SCK, PinName DOUT )
    : _PD_SCK               ( PD_SCK )
    , _DOUT                 ( DOUT )
{

}


HX711::~HX711()
{
}



/**
 * @brief       HX711_Reset   ( void )
 *
 * @details     It performs an internal reset.
 *
 * @param[in]    NaN.
 *
 * @param[out]   NaN.
 *
 *
 * @return       Status of HX711_Reset.
 *
 *
 * @author      Manuel Caballero
 * @date        11/September/2017
 * @version     11/September/2017   The ORIGIN
 * @pre         When PD_SCK pin changes from low to high and stays at high for
 *              longer than 60μs, HX711 enters power down mode.
 *
 *              When PD_SCK returns to low, chip will reset and enter normal
 *              operation mode.
 * @warning     NaN.
 */
HX711::HX711_status_t  HX711::HX711_Reset   ( void )
{
    _PD_SCK  =  HX711_PIN_HIGH;
    wait_us ( 120 );                                                            // Datasheet p5. At least 60us ( Security Factor: 2*60us = 120us )
    _PD_SCK  =  HX711_PIN_LOW;



    if ( _DOUT == HX711_PIN_HIGH )
        return   HX711_SUCCESS;
    else
        return   HX711_FAILURE;
}



/**
 * @brief       HX711_PowerDown   ( void )
 *
 * @details     It puts the device in power-down mode.
 *
 * @param[in]    NaN.
 *
 * @param[out]   NaN.
 *
 *
 * @return       Status of HX711_PowerDown.
 *
 *
 * @author      Manuel Caballero
 * @date        11/September/2017
 * @version     11/September/2017   The ORIGIN
 * @pre         When PD_SCK pin changes from low to high and stays at high for
 *              longer than 60μs, HX711 enters power down mode.
 * @warning     NaN.
 */
HX711::HX711_status_t  HX711::HX711_PowerDown   ( void )
{
    _PD_SCK  =  HX711_PIN_HIGH;
    wait_us ( 120 );                                                            // Datasheet p5. At least 60us ( Security Factor: 2*60us = 120us )



    if ( _DOUT == HX711_PIN_HIGH )
        return   HX711_SUCCESS;
    else
        return   HX711_FAILURE;
}



/**
 * @brief       HX711_SetChannelAndGain   ( HX711_channel_gain_t myChannel_Gain )
 *
 * @details     It sets both the channel and the gain for the next measurement.
 *
 * @param[in]    myChannel_Gain:    Channel and Gain to perform the new measurement.
 *
 * @param[out]   NaN.
 *
 *
 * @return       Status of HX711_SetChannelAndGain.
 *
 *
 * @author      Manuel Caballero
 * @date        11/September/2017
 * @version     11/September/2017   The ORIGIN
 * @pre         NaN.
 * @warning     NaN.
 */
HX711::HX711_status_t  HX711::HX711_SetChannelAndGain    ( HX711_channel_gain_t myChannel_Gain )
{
    uint32_t myPulses   =    0;
    uint32_t i          =    0;                                                 // Counter and timeout variable

    // Select the gain/channel
    switch ( myChannel_Gain ) {
        default:
        case CHANNEL_A_GAIN_128:
            _HX711_CHANNEL_GAIN  =   CHANNEL_A_GAIN_128;                        // Update the gain parameter
            myPulses             =   25;
            break;

        case CHANNEL_B_GAIN_32:
            _HX711_CHANNEL_GAIN  =   CHANNEL_B_GAIN_32;                         // Update the gain parameter
            myPulses             =   26;
            break;

        case CHANNEL_A_GAIN_64:
            _HX711_CHANNEL_GAIN  =   CHANNEL_A_GAIN_64;                         // Update the gain parameter
            myPulses             =   27;
            break;
    }


    // Wait until the device is ready or timeout
    i        =   23232323;
    _PD_SCK  =  HX711_PIN_LOW;
    while ( ( _DOUT == HX711_PIN_HIGH ) && ( --i ) );

    // Check if something is wrong with the device because of the timeout
    if ( i < 1 )
        return   HX711_FAILURE;

    // Change the gain for the NEXT mesurement ( previous data will be ignored )
    do {
        wait_us ( 1 );                                                          // Datasheet p5. T3 and T4 ( Min. 0.2us | Typ. 1us )
        _PD_SCK  =  HX711_PIN_HIGH;
        wait_us ( 1 );                                                          // Datasheet p5. T3 and T4 ( Min. 0.2us | Typ. 1us )
        _PD_SCK  =  HX711_PIN_LOW;

        myPulses--;
    } while ( myPulses > 0 );




    if ( _DOUT == HX711_PIN_HIGH )
        return   HX711_SUCCESS;
    else
        return   HX711_FAILURE;
}



/**
 * @brief       HX711_GetChannelAndGain   ( void )
 *
 * @details     It gets both the channel and the gain for the current measurement.
 *
 * @param[in]    NaN.
 *
 * @param[out]   NaN.
 *
 *
 * @return       Channel and Gain.
 *
 *
 * @author      Manuel Caballero
 * @date        12/September/2017
 * @version     12/September/2017   The ORIGIN
 * @pre         NaN.
 * @warning     NaN.
 */
HX711::HX711_channel_gain_t  HX711::HX711_GetChannelAndGain ( void )
{
    return   _HX711_CHANNEL_GAIN;
}



/**
 * @brief       HX711_ReadRawData   ( HX711_channel_gain_t myChannel_Gain, Vector_count_t*, uint32_t )
 *
 * @details     It reads the raw data from the device according to the channel
 *              and its gain.
 *
 * @param[in]    myChannel_Gain:    Channel and Gain to perform the new read.
 * @param[in]    myAverage:         How many measurement we have to get and deliver the average.
 *
 * @param[out]   myNewRawData:      The new value from the device.
 *
 *
 * @return       Status of HX711_ReadRawData.
 *
 *
 * @author      Manuel Caballero
 * @date        11/September/2017
 * @version     12/September/2017   Gain mode was fixed, now it gets the value
 *                                  a given gain/channel. A timeout was added to
 *                                  avoid the microcontroller gets stuck.
 *              11/September/2017   The ORIGIN
 * @pre         NaN.
 * @warning     NaN.
 */
HX711::HX711_status_t  HX711::HX711_ReadRawData    ( HX711_channel_gain_t myChannel_Gain, Vector_count_t* myNewRawData, uint32_t myAverage )
{
    uint32_t i           =   0;                                                 // Counter and timeout variable
    uint32_t ii          =   0;                                                 // Counter variable
    uint32_t myAuxData   =   0;
    uint32_t myPulses    =   0;



    myNewRawData->myRawValue    =   0;                                          // Reset variable at the beginning

    // Check the gain if it is different, update it ( previous data will be ignored! )
    if ( myChannel_Gain != _HX711_CHANNEL_GAIN )
        HX711_SetChannelAndGain ( myChannel_Gain );


    // Start collecting the new measurement as many as myAverage
    for ( ii = 0; ii < myAverage; ii++ ) {
        // Reset the value
        myAuxData    =   0;

        // Wait until the device is ready or timeout
        i        =   23232323;
        _PD_SCK  =  HX711_PIN_LOW;
        while ( ( _DOUT == HX711_PIN_HIGH ) && ( --i ) );

        // Check if something is wrong with the device because of the timeout
        if ( i < 1 )
            return   HX711_FAILURE;


        // Read the data
        for ( i = 0; i < 24; i++ ) {
            wait_us ( 1 );                                                      // Datasheet p5. T3 and T4 ( Min. 0.2us | Typ. 1us )
            _PD_SCK  =  HX711_PIN_HIGH;
            wait_us ( 1 );                                                      // Datasheet p5. T3 and T4 ( Min. 0.2us | Typ. 1us )
            myAuxData    <<=     1;
            _PD_SCK  =  HX711_PIN_LOW;

            // High or Low bit
            if ( _DOUT == HX711_PIN_HIGH )
                myAuxData++;
        }

        // Last bit to release the bus
        wait_us ( 1 );                                                          // Datasheet p5. T3 and T4 ( Min. 0.2us | Typ. 1us )
        _PD_SCK  =  HX711_PIN_HIGH;
        wait_us ( 1 );                                                          // Datasheet p5. T3 and T4 ( Min. 0.2us | Typ. 1us )
        _PD_SCK  =  HX711_PIN_LOW;


        // Depending on the Gain we have to generate more CLK pulses
        switch ( _HX711_CHANNEL_GAIN ) {
            default:
            case CHANNEL_A_GAIN_128:
                myPulses             =   25;
                break;

            case CHANNEL_B_GAIN_32:
                myPulses             =   26;
                break;

            case CHANNEL_A_GAIN_64:
                myPulses             =   27;
                break;
        }

        // Generate those extra pulses for the next measurement
        for ( i = 25; i < myPulses; i++ ) {
            wait_us ( 1 );                                                      // Datasheet p5. T3 and T4 ( Min. 0.2us | Typ. 1us )
            _PD_SCK  =  HX711_PIN_HIGH;
            wait_us ( 1 );                                                      // Datasheet p5. T3 and T4 ( Min. 0.2us | Typ. 1us )
            _PD_SCK  =  HX711_PIN_LOW;
        }

        // Update data to get the average
        myAuxData                  ^=    0x800000;
        myNewRawData->myRawValue   +=    myAuxData;
    }

    myNewRawData->myRawValue /=  (float)myAverage;



    if ( _DOUT == HX711_PIN_HIGH )
        return   HX711_SUCCESS;
    else
        return   HX711_FAILURE;
}