/*
/ _____)             _              | |
( (____  _____ ____ _| |_ _____  ____| |__
\____ \| ___ |    (_   _) ___ |/ ___)  _ \
_____) ) ____| | | || |_| ____( (___| | | |
(______/|_____)_|_|_| \__)_____)\____)_| |_|
    ( C ) Semtech

Description: Port of Semtech SX1276 C++ library to support SX1272 chipset performed by Espotel

License: Revised BSD License, see LICENSE.TXT file include in the project

Maintainers: www.espotel.com

*/
#include "sx1272-hal.h"

#define TX_STATE 0

const RadioRegisters_t SX1272BRD::RadioRegsInit[] =
{                                                 
    { MODEM_FSK , REG_LNA                , 0x23 },
    { MODEM_FSK , REG_RXCONFIG           , 0x1E },
    { MODEM_FSK , REG_RSSICONFIG         , 0xD2 },
    { MODEM_FSK , REG_PREAMBLEDETECT     , 0xAA },
    { MODEM_FSK , REG_OSC                , 0x07 },
    { MODEM_FSK , REG_SYNCCONFIG         , 0x12 },
    { MODEM_FSK , REG_SYNCVALUE1         , 0xC1 },
    { MODEM_FSK , REG_SYNCVALUE2         , 0x94 },
    { MODEM_FSK , REG_SYNCVALUE3         , 0xC1 },
    { MODEM_FSK , REG_PACKETCONFIG1      , 0x98 },
    { MODEM_FSK , REG_FIFOTHRESH         , 0x8F },
    { MODEM_FSK , REG_IMAGECAL           , 0x02 },
    { MODEM_FSK , REG_DIOMAPPING1        , 0x00 },
    { MODEM_FSK , REG_DIOMAPPING2        , 0x30 },
    { MODEM_LORA, REG_LR_PAYLOADMAXLENGTH, 0x40 },  
};

SX1272BRD::SX1272BRD( void ( *txDone )( ), void ( *txTimeout ) ( ), void ( *rxDone ) ( uint8_t *payload, uint16_t size, int16_t rssi, int8_t snr ),
                            void ( *rxTimeout ) ( ), void ( *rxError ) ( ), void ( *fhssChangeChannel ) ( uint8_t channelIndex ), void ( *cadDone ) ( bool ChannelActivityDetected ),
                            PinName mosi, PinName miso, PinName sclk, PinName nss, PinName reset,
                            PinName dio0, PinName dio1, PinName dio2, PinName dio3, PinName dio4, PinName dio5,
                            PinName antSwitch )
                            : SX1272( txDone, txTimeout, rxDone, rxTimeout, rxError, fhssChangeChannel, cadDone, mosi, miso, sclk, nss, reset, dio0, dio1, dio2, dio3, dio4, dio5),
                            antSwitch( antSwitch )
{
    Reset( );
    
    RxChainCalibration( );
    
    IoInit( );
    
    SetOpMode( RF_OPMODE_SLEEP );
    
    IoIrqInit( dioIrq );
    
    RadioRegistersInit( );

    SetModem( MODEM_FSK );

    this->settings.State = IDLE ;
}


SX1272BRD::~SX1272BRD ( )
{
    IoIrqDeInit( );
};

//-------------------------------------------------------------------------
//                      Board relative functions
//-------------------------------------------------------------------------
void SX1272BRD::IoInit( void )
{
    AntSwInit( );
    SpiInit( );
}

void SX1272BRD::RadioRegistersInit( ){
    uint8_t i = 0;
    for( i = 0; i < sizeof( RadioRegsInit ) / sizeof( RadioRegisters_t ); i++ )
    {
        SetModem( RadioRegsInit[i].Modem );
        Write( RadioRegsInit[i].Addr, RadioRegsInit[i].Value );
    }    
}

void SX1272BRD::SpiInit( void )
{
    nss = 1;    
    spi.format( 8,0 );   
    uint32_t frequencyToSet = 8000000;
    #if( defined ( TARGET_NUCLEO_L152RE ) ||  defined ( TARGET_LPC11U6X ) ||  defined ( TARGET_ELMO_F411RE ) )
        spi.frequency( frequencyToSet );
    #elif( defined ( TARGET_KL25Z ) ) //busclock frequency is halved -> double the spi frequency to compensate
        spi.frequency( frequencyToSet * 2 );
    #else
        #warning "Check the board's SPI frequency"
    #endif
    wait(0.1); 
}

void SX1272BRD::IoIrqInit( DioIrqHandler *irqHandlers )
{
    #if( defined ( TARGET_NUCLEO_L152RE ) ||  defined ( TARGET_LPC11U6X ) )
        dio0.mode(PullDown);
        dio1.mode(PullDown);   
        dio2.mode(PullDown);
        dio3.mode(PullDown); 
        dio4.mode(PullDown); 
    #endif
    dio0.rise( this, static_cast< TriggerSX1272BRD > ( irqHandlers[0] ) );
    dio1.rise( this, static_cast< TriggerSX1272BRD > ( irqHandlers[1] ) );
    dio2.rise( this, static_cast< TriggerSX1272BRD > ( irqHandlers[2] ) );
    dio3.rise( this, static_cast< TriggerSX1272BRD > ( irqHandlers[3] ) );
    dio4.rise( this, static_cast< TriggerSX1272BRD > ( irqHandlers[4] ) );
}

void SX1272BRD::IoIrqDeInit( )
{
    dio0.disable_irq();
    dio1.disable_irq();
    dio2.disable_irq();
    dio3.disable_irq();
    dio4.disable_irq();
}

void SX1272BRD::IoDeInit( void )
{
    //nothing
}

uint8_t SX1272BRD::GetPaSelect( uint32_t channel )
{
	return RF_PACONFIG_PASELECT_RFO;
}

void SX1272BRD::SetAntSwLowPower( bool status )
{
    if( isRadioActive != status )
    {
        isRadioActive = status;
    
        if( status == false )
        {
            AntSwInit( );
        }
        else
        {
            AntSwDeInit( );
        }
    }
}


void SX1272BRD::AntSwInit( void )
{
    antSwitch.output();
    antSwitch = !TX_STATE;
}

void SX1272BRD::AntSwDeInit( void )
{
    antSwitch = !TX_STATE;
}

void SX1272BRD::SetAntSw( uint8_t rxTx )
{
//    if( this->rxTx == rxTx )
//    {
//        //no need to go further
//        return;
//    }

//    this->rxTx = rxTx;

    if( rxTx != 0 )
    {
        antSwitch = TX_STATE;
    }
    else
    {
        antSwitch = !TX_STATE;
    }
}

bool SX1272BRD::CheckRfFrequency( uint32_t frequency )
{
    //TODO: Implement check, currently all frequencies are supported
    return true;
}


void SX1272BRD::Reset( void )
{
    reset.output();
    reset = 1;
    wait_ms( 1 );
    reset.input();
    wait_ms( 6 );
}
    
void SX1272BRD::Write( uint8_t addr, uint8_t data )
{
    Write( addr, &data, 1 );
}

uint8_t SX1272BRD::Read( uint8_t addr )
{
    uint8_t data;
    Read( addr, &data, 1 );
    return data;
}

void SX1272BRD::Write( uint8_t addr, uint8_t *buffer, uint8_t size )
{
    uint8_t i;

    nss = 0;
    spi.write( addr | 0x80 );
    for( i = 0; i < size; i++ )
    {
        spi.write( buffer[i] );
    }
    nss = 1;
}

void SX1272BRD::Read( uint8_t addr, uint8_t *buffer, uint8_t size )
{
    uint8_t i;

    nss = 0;
    spi.write( addr & 0x7F );
    for( i = 0; i < size; i++ )
    {
        buffer[i] = spi.write( 0 );
    }
    nss = 1;
}

void SX1272BRD::WriteFifo( uint8_t *buffer, uint8_t size )
{
    Write( 0, buffer, size );
}

void SX1272BRD::ReadFifo( uint8_t *buffer, uint8_t size )
{
    Read( 0, buffer, size );
}
