#include "Terminal.h"
#include "Cmd.h"

Terminal::Terminal(Settings* settings, Serial *serial, InterruptIn* pinStart)
    : settings_(settings), serial_(serial), pinStart_(pinStart), buttonStart_(false)
{
    cmdFactory_ = new CmdFactory(settings_, serial_);

    if (NULL != pinStart_)
    {
        pinStart_->rise(this, &Terminal::buttonPressed);
        pinStart_->enable_irq();
    }
}
 
Terminal::~Terminal()
{
    if (NULL != pinStart_)
    {
        pinStart_->disable_irq();
    }
    delete cmdFactory_;
}

void Terminal::buttonPressed()
{
    buttonStart_ = true;
}


void Terminal::executeCmd(string rxBuffer)
{
        list<string> tokens = tokenize(rxBuffer);

        const bool single = (tokens.front() == "s");
        if (single)
        {
            tokens.pop_front();
        }

        Cmd* cmd = NULL;

        if (!tokens.empty())
        {
            cmd = cmdFactory_->createCmd(tokens.front());
            tokens.pop_front();
        }

        if(NULL != cmd)
        {
            //serial_->printf("\n odpowiedz %s\r\n", cmd->getResponse().c_str());

            cmd->execute(tokens);

            uint32_t uiPrompt = 0;

            buttonStart_ = false;
            while((!(cmd->process())) && !buttonStart_ && !single && (!(serial_->readable())))
            {
                serial_->printf("%c\r", "/|\\-"[uiPrompt++ & 3]);
            }

            delete cmd;
            pinStart_->enable_irq();
            serial_->printf(".\r");
        }
}

void Terminal::start(void)
{
    DigitalOut led1(LED1);
    string rxBuffer;
    
    while(1)
    {
        if(serial_->readable()) 
        {
            char c = serial_->getc();
            serial_->printf("%c", c);

            if (c==127) // backspace sends del ?
            {
                rxBuffer.resize(rxBuffer.size() > 0 ? (rxBuffer.size()-1) : 0);
            } else
            {
                rxBuffer += c;
            }
        }
        
        if(rxBuffer.find("\r") != std::string::npos)
        {
            serial_->printf("\r\n");
            // omit string that contains only '\r' char
            if(1 != rxBuffer.length())
            {
                // remove  carriage return char
                rxBuffer = rxBuffer.substr(0, rxBuffer.length() - 1);
                executeCmd(rxBuffer);
            }
            rxBuffer.clear();
        }

        if (buttonStart_)
        {
            executeCmd("pp");
        }

        led1 = !led1;
        //wait(0.5);
    }
}

list<string> Terminal::tokenize(const string& str, const string& delimiters)
{
    list<string> tokens;

    // Skip delimiters at beginning.
    string::size_type startPos = str.find_first_not_of(delimiters, 0);
    //serial_->printf("\n lastpos = %i\r\n", lastPos);
    // Find first "delimiter".
    string::size_type pos  = str.find_first_of(delimiters, startPos);
    //serial_->printf("\n pos = %i\r\n", pos);

    while (string::npos != pos || string::npos != startPos)
    {
        // Found a token, add it to the vector.
        tokens.push_back(str.substr(startPos, pos - startPos));
        // Skip delimiters.  Note the "not_of"
        startPos = str.find_first_not_of(delimiters, pos);
        // Find next "non-delimiter"
        pos = str.find_first_of(delimiters, startPos);
    }
    return tokens;
}
