/******************************************************************************
 *
 * File:                main.cpp
 * Desciption:          Ground Zero
 *
 *****************************************************************************/
#include "mbed.h"
#include "rtos.h"
#include <stdio.h>
#include "mDot.h"
#include "global.h"
#include "ConfigurationHandler.h"
#include "AnalyticsLogger.h"
#include "ModbusMaster.h"
#include "BLEDataHandler.h"
#include "LoRaInit.h"
#include "ControlTask.h"
#include "OutputTask.h"
#include "CloudDataHandler.h"
#include "rtc.h"

// main thread identifier (for signaling)
osThreadId mainThreadId = NULL;

// bootup sequence signals
int sig_output_continue = 0x1;
int sig_config_continue = 0x2;

// data handler to configuration hanlder mailbox
Mail<Message_t, 16> MailBox;
Mail<ModbusMasterReq_t, 2> ModbusMasterMailBox;
Mail<OutputControlMsg_t, 16> OutputMasterMailBox;
Mail<AnalyticsLoggerReq_t, 16> AnalyticsLoggerMailBox;
Mail<BLEHandlerReq_t, 1> BLEHandlerMailBox;

// local function prototypes
static void banner(void);

// for object (singleton) access outside of main()
mDot *GLOBAL_mdot;

Thread *GLOBAL_analyticsLogger_thread = NULL;
Thread *GLOBAL_modbusMaster_thread = NULL;
Thread *GLOBAL_BLE_thread = NULL;
Thread *GLOBAL_CDH_thread = NULL;
Thread *GLOBAL_configHandler_thread = NULL;
Thread *GLOBAL_controlTask_thread = NULL;
Thread *GLOBAL_outputTask_thread = NULL;

// store modbus register information
std::map<std::string,ModbusRegister> ModbusRegisterMap;
std::map<std::string,SimulateInput> SimulateInputMap;

I2C i2c_instance(I2C_SDA, I2C_SCL);
I2C* i2c;

/*****************************************************************************
 * Function:            timestamp_boot_record()
 * Description:         timestamp the boot record with current time
 *
 * @param               none
 * @return              none
 *****************************************************************************/
static void timestamp_boot_record(void)
{
    char time_string[80];
    time_t curr_sec;
    struct tm *ts;

    curr_sec = time(0);
    ts = localtime(&curr_sec);
    strftime(time_string, sizeof(time_string), "%Y-%m-%d %H:%M:%S", ts);

    // see if we can open the boot record
    mDot::mdot_file file = GLOBAL_mdot->openUserFile("boot.time", mDot::FM_RDWR);
    if ( file.fd < 0 ) {
        file = GLOBAL_mdot->openUserFile("boot.time", mDot::FM_CREAT|mDot::FM_RDWR);
        GLOBAL_mdot->writeUserFile(file, time_string, sizeof(time_string));
    } else {
        GLOBAL_mdot->seekUserFile(file, 0, SEEK_CUR);
        GLOBAL_mdot->writeUserFile(file, time_string, sizeof(time_string));
    }
    GLOBAL_mdot->closeUserFile(file);
    return;
}

/*****************************************************************************
 * Function:            banner()
 * Description:         Display the application boot banner
 *
 * @param               none
 * @return              none
 *****************************************************************************/
static void banner( void )
{
    struct tm       *ts;
    time_t curr_sec;
    char time_string[80];

    curr_sec = time(0);
    ts = localtime(&curr_sec);
    strftime(time_string, sizeof(time_string), "%Y-%m-%d %H:%M:%S", ts);

    printf("\n\n\r\nWelcome to Project: ICE v0.0.69\n");
    printf("\rThe Intelligent Connected Experience\n");
    printf("\rCopyright 2016 Nalco Water, an Ecolab Company\n");

    printf("\r _________ _______  _______ \n");
    printf("\r \\__   __/(  ____ \\(  ____ \\ \n");
    printf("\r    ) (   | (    \\/| (    \\/ \n");
    printf("\r    | |   | |      | (__     \n");
    printf("\r    | |   | |      |  __)   \n");
    printf("\r    | |   | |      | (      \n");
    printf("\r ___) (___| (____/\\| (____/\\ \n");
    printf("\r \\_______/(_______/(_______/  \n");

    printf("\r\nCurrent time is: %s\r\n", time_string);

    printf("\r\n\r\n");
    printf("\rMultiTech mDot library version: %s\n\r\n\r\n", GLOBAL_mdot->getId().c_str());

}

/*****************************************************************************
 * Function:            banner()
 * Description:         Display the application boot banner
 *
 * @param               none
 * @return              none
 *****************************************************************************/
int main( void )
{
    mDot *dot;
    struct tm rtc_time;
    time_t curr_sec;
    int year=0;

    // singleton object instantiation
    GLOBAL_mdot = dot = mDot::getInstance();
    mDotRadioInit( dot );

    i2c = &i2c_instance;

    rtc_init();

    rtc_get_time(&year, &rtc_time.tm_mon, &rtc_time.tm_mday, &rtc_time.tm_hour, &rtc_time.tm_min, &rtc_time.tm_sec);
    rtc_time.tm_mon = rtc_time.tm_mon - 1;
    rtc_time.tm_year = year - 1900;
    curr_sec =  mktime( &rtc_time );

    set_time(curr_sec);
    timestamp_boot_record();

    // for signaling from the configuration handler
    mainThreadId = osThreadGetId();

    banner();

    Thread modbusMaster_thread(ModbusMaster, NULL, osPriorityHigh, MODBUS_MASTER_STACK_SIZE, NULL);
    osSignalWait(sig_output_continue, osWaitForever);

    // start the output task
    Thread outputTask_thread(OutputTask,  NULL, osPriorityNormal, OUTPUT_TASK_STACK_SIZE, NULL);
    osSignalWait(sig_output_continue, osWaitForever);

    // start the configuration handler
    Thread configHandler_thread(ConfigurationHandler, NULL, osPriorityNormal, CONFIG_HANDLER_STACK_SIZE,  NULL);
    osSignalWait(sig_config_continue, osWaitForever);

    // we're clear to start running the controls
    Thread controlTask_thread(ControlTask, NULL, osPriorityNormal, CONTROL_TASK_STACK_SIZE,  NULL);
    logInfo("\r%s:%d: continuing to initialize...\n", __func__, __LINE__);

    Thread BLE_thread(BLEDataHandler, NULL, osPriorityNormal, BLE_DATA_HANDLER_STACK_SIZE, NULL);
    printf("\r%s:%d: continuing to initialize...\n", __func__, __LINE__);

    Thread CDH_thread(CloudDataHandler, NULL, osPriorityNormal, CLOUD_DATA_HANDLER_STACK_SIZE, NULL);
    logInfo("\r%s:%d: continuing to initialize...\n", __func__, __LINE__);
    
    Thread analyticsLoggerThread(AnalyticsLogger, NULL, osPriorityHigh, ANALYTICS_LOGGER_STACK_SIZE, NULL);
    logInfo("\r%s:%d: continuing to initialize...\n", __func__, __LINE__);


    // assign globals
    GLOBAL_analyticsLogger_thread = &analyticsLoggerThread;
    GLOBAL_modbusMaster_thread = &modbusMaster_thread;
    GLOBAL_BLE_thread = & BLE_thread;
    GLOBAL_CDH_thread = & CDH_thread;
    GLOBAL_configHandler_thread = &configHandler_thread;
    GLOBAL_controlTask_thread = &controlTask_thread;
    GLOBAL_outputTask_thread = &outputTask_thread;

    Thread::wait(1000);
    printf("\r\n");

    // start the command shell
    ntshell_execute(&ntshell, func_read, func_write, func_cb_ntshell);
}