/******************************************************************************
 *
 * File:                OutputTask.cpp
 * Desciption:          source for the ICE Output task
 *
 *****************************************************************************/
#include "OutputTask.h"
#include "global.h"
#include "cJSON.h"
#include "ModbusMasterApi.h"
#include "LoggerApi.h"
#include "Control.h"
#include <vector>
#include <string>
#include <algorithm>

// local functions
static int  createOutput(const char *controlFile);
static void loadPersistentOutputs(void);
static void writeOutputs(const std::string, const std::string);
static int  enableOutputReqHandler(OutputControlMsg_t *msg);
static int  disableOutputReqHandler(OutputControlMsg_t *msg);
static int  unregisterControl(const char *id, unsigned int pri, const char *output);

// The Output Map
//
// this is the main data structure used to distinguish which control has
// priority of an output. the layout is as-follows:
// outputMap["o_rly1"]-> Control<"ManControl_rly1",       100, ON > <<- highest pri control manipulates relay
//                       Control<"SetpointControl_rly1",  800, OFF> <<- lowest pri (queued up)
//
// outputMap["o_rly2"]-> Control<"SetpointControl_rly2",  800, ON >
//
// outputMap["o_rly3"]-> Control<"TimerControl_rly3",     700, ON>
//
//
// The Control Vector (per relay) is always sorted by priority, whereas
// the highest priority control is at the front (v.front()) of the
// list.

typedef std::map<std::string, vector<Control> > StringOutputVector_t;
StringOutputVector_t outputMap;

// operator for sorting the outputs vectors per output
bool operator<(const Control &control1, const Control &control2)
{
    return control1.getPriority() < control2.getPriority();
}

/*****************************************************************************
 * Function:            OutputTask
 * Description:         Main entry point for the Output Task
 *
 * @param               args -> not used
 * @return              none
 *****************************************************************************/
void OutputTask(void const *args)
{
    int rc;
    UNUSED(args);

    printf("\r%s has started...\n", __func__);

    loadPersistentOutputs();
    osSignalSet(mainThreadId, sig_output_continue);

    while (true) {
        // wait for an event
        osEvent evt = OutputMasterMailBox.get();
        if (evt.status == osEventMail) {

            OutputControlMsg_t *msg = (OutputControlMsg_t*) evt.value.p;

            switch ( msg->action ) {
                case ACTION_NEW:
                    // read the file and and create an output entry
                    rc = createOutput(msg->controlFile);
                    if ( rc != 0 ) {
                        logError("%s: failed to create output %s\n",
                                 __func__, msg->controlFile);
                    }
                    break;
                case ACTION_CONTROL_ON:
                    logInfo("%s is requesting ON control of %s", msg->id, msg->output_tag);
                    rc = enableOutputReqHandler(msg);
                    if ( rc != 0 ) {
                        logError("%s: failed to enabled output for %s",
                                 __func__, msg->id);
                    }
                    break;
                case ACTION_CONTROL_OFF:
                    logInfo("%s is requesting OFF control of %s", msg->id, msg->output_tag);
                    rc = disableOutputReqHandler(msg);
                    if ( rc != 0 ) {
                        printf("%s: failed to disabled output for %s",
                               __func__, msg->id);
                    }
                    break;
                case ACTION_CONTROL_UNREGISTER:
                    logInfo("%s is requesting its deletion from %s", msg->id, msg->output_tag);
                    rc = unregisterControl(msg->id, msg->priority, msg->output_tag);
                    if ( rc != 0 ) {
                        printf("%s: failed to unregister control %s",
                               __func__, msg->id);
                    }
                    break;
                default:
                    break;
            }

            // free the message
            OutputMasterMailBox.free(msg);

            // refresh the outputs
            writeOutputs(msg->id, msg->output_tag);
        }
    }
}

/*****************************************************************************
 * Function:            recordEvent
 * Description:         send an event to the logger
 *
 * @param               output  -> the output channel
 * @param               control -> the control, this can be null
 * @return              none
 *****************************************************************************/
void recordEvent(std::string output, const Control* control)
{

    EventReasonStruct_t ev;
    ModbusValue input_value;
    ModbusValue output_value;
    memset(&ev, 0, sizeof(ev));

    // if there's no control, that means the only control that was on
    // the relay stack has been destroyed, so we'll send a NO CONTROL
    // code.
    if ( !control ) {
        ev.eventReason = EVENT_REASON_NO_CONTROL;
        ModbusMasterReadRegister(output, &output_value);
        strncpy(ev.outputTag, output.c_str(), sizeof(ev.outputTag));
        ev.outputValue = output_value.value;

        printf("\rEVENT RECORD\n");
        printf("\rev.eventReason = %d\n",    ev.eventReason);
        printf("\rev.outputTag   = %s\n",    ev.outputTag);
        printf("\rev.outputValue = %.02f\n", ev.outputValue);

        EventLoggerApi(ev);
        return;
    }

    switch ( control->getControlType() ) {
        case CONTROL_SETPOINT:
        case CONTROL_COMPOSITE:
            ev.eventReason = EVENT_REASON_AUTO;
            strncpy(ev.inputTag, control->getInput().c_str(), sizeof(ev.inputTag));
            strncpy(ev.outputTag, output.c_str(), sizeof(ev.outputTag));
            ModbusMasterReadRegister(control->getInput(), &input_value);
            ModbusMasterReadRegister(output, &output_value);
            ev.inputValue = input_value.value;
            ev.outputValue = output_value.value;
            printf("\rEVENT RECORD\n");
            printf("\rev.eventReason = %d\n", ev.eventReason);
            printf("\rev.outputTag   = %s\n", ev.outputTag);
            printf("\rev.outputValue = %.02f\n", ev.outputValue);
            EventLoggerApi(ev);
            break;
            
        case CONTROL_MANUAL:
            ev.eventReason = EVENT_REASON_MANUAL;
            strncpy(ev.outputTag, output.c_str(), sizeof(ev.outputTag));
            ModbusMasterReadRegister(output, &output_value);
            ev.outputValue = output_value.value;
            printf("\rEVENT RECORD\n");
            printf("\rev.eventReason = %d\n", ev.eventReason);
            printf("\rev.outputTag = %s\n", ev.outputTag);
            printf("\rev.outputValue = %.02f\n", ev.outputValue);
            EventLoggerApi(ev);
            break;
            
        case CONTROL_TIMER:
            ev.eventReason = EVENT_REASON_TIMER;
            strncpy(ev.outputTag, output.c_str(), sizeof(ev.outputTag));
            ModbusMasterReadRegister(output, &output_value);
            ev.outputValue = output_value.value;
            printf("\rEVENT RECORD\n");
            printf("\rev.eventReason = %d\n", ev.eventReason);
            printf("\rev.outputTag = %s\n", ev.outputTag);
            printf("\rev.outputValue = %.02f\n", ev.outputValue);
            EventLoggerApi(ev);
            break;
            
        case CONTROL_FAILSAFE:
            ev.eventReason = EVENT_REASON_FAILSAFE;
            strncpy(ev.outputTag, output.c_str(), sizeof(ev.outputTag));
            ModbusMasterReadRegister(output, &output_value);
            ev.outputValue = output_value.value;
            printf("\rEVENT RECORD\n");
            printf("\rrev.eventReason = %d\n", ev.eventReason);
            printf("\rev.outputTag = %s\n", ev.outputTag);
            printf("\rev.outputValue = %.02f\n", ev.outputValue);
            EventLoggerApi(ev); 
            break;
        default:
            break;
    }
}

/*****************************************************************************
 * Function:            writeOutputs
 * Description:         send a message to the modbus master of who's in control
 *
 *
 * @param               id          -> control identifier
 * @param               output_tag  -> the output to write
 * @return              none
 *****************************************************************************/
static void writeOutputs(const std::string id, const std::string output_tag)
{
    UNUSED(id);
    
    if ( output_tag.empty() ) return;

    StringOutputVector_t::iterator pos;

    // find the output 
    pos = outputMap.find(output_tag);
    if ( pos != outputMap.end() ) {
        // we found the ouput, but nothing's controlling it... 
        if ( pos->second.empty() ) {
            ModbusMasterWriteRegister(pos->first, RELAY_STATUS_NOT_CONTROLLED);
            recordEvent(pos->first, NULL);
        } else { 
            ModbusMasterWriteRegister(pos->first, pos->second.begin()->getMappedState());
            recordEvent(pos->first, pos->second.begin());
        }
    } else {
        logError("%s failed to find the selected output %s",
                 __func__, output_tag.c_str());
    }
}

/*****************************************************************************
 * Function:            createOutput
 * Description:
 *
 * @param               outputFile -> name of output file
 * @return              none
 *****************************************************************************/
static int createOutput(const char *outputFile)
{
    char dataBuf[MAX_FILE_SIZE];
    int status = GLOBAL_mdot->readUserFile(outputFile, (void *)dataBuf, sizeof(dataBuf));
    if ( status != true ) {
        logError("%s failed to read %s", __func__, outputFile);
        return -1;
    }
    
    cJSON * root = cJSON_Parse(dataBuf);
    std::string id = cJSON_GetObjectItem(root,"id")->valuestring;
    cJSON_Delete(root);
    
    vector<Control> v;
    outputMap[id] = v;

    return 0;
}

/*****************************************************************************
 * Function:            enableOutputReqHandler
 * Description:         handle a request to enable an output
 *
 * @param               msg -> the message request
 * @return              -1 on error
 *****************************************************************************/
static int enableOutputReqHandler(OutputControlMsg_t *msg)
{
    // attempt to find the output in the map
    StringOutputVector_t::iterator pos;

    pos = outputMap.find(msg->output_tag);
    if ( pos == outputMap.end() ) {
        printf("%s: failed to find the designated output %s\n",
               __func__, msg->output_tag);
        return -1;
    }

    if ( pos->second.empty() ) {
        // this is a new request
        std::string cid(msg->id);
        std::string input(msg->input_tag);
        Control c(cid, msg->controlType, input, msg->priority, CONTROL_ON);
        pos->second.push_back(c);
    } else {
        // find this control in the list
        vector<Control>::iterator v;
        for ( v = pos->second.begin(); v != pos->second.end(); ++v ) {
            if ( strcmp(v->getId().c_str(), msg->id) == 0 )  {
                v->setState(CONTROL_ON);
                break;
            }
        }
        if ( v == pos->second.end() ) {
            // this is a new request, so add it and sort the vector
            std::string cid(msg->id);
            std::string input(msg->input_tag);
            Control c(cid, msg->controlType, input, msg->priority, CONTROL_ON);
            pos->second.push_back(c);
            std::sort(pos->second.begin(), pos->second.end());
            printf("\r\r\nDEBUG: vector size = %d\n", pos->second.size()); 
        }
    }

    return 0;
}

/*****************************************************************************
 * Function:            disableOutputReqHandler
 * Description:         handle a request to disable an output
 *
 * @param               msg -> the message request
 * @return              none
 *****************************************************************************/
static int disableOutputReqHandler(OutputControlMsg_t *msg)
{
    // attempt to find the output in the map
    StringOutputVector_t::iterator pos;

    pos = outputMap.find(msg->output_tag);
    if ( pos == outputMap.end() ) {
        printf("%s: failed to find the designated output %s\n",
               __func__, msg->output_tag);
        return -1;
    }

    // if the control list is empty, push this control on the list
    if ( pos->second.empty() ) {
        std::string cid(msg->id);
        std::string input(msg->input_tag);
        Control c(cid, msg->controlType, input, msg->priority, CONTROL_OFF);
        pos->second.push_back(c);
    } else {
        // find this control in the list
        vector<Control>::iterator v;
        for ( v = pos->second.begin(); v != pos->second.end(); ++v ) {
            if ( strcmp(v->getId().c_str(), msg->id) == 0 )  {
                v->setState(CONTROL_OFF);
                break;
            }
        }

        if ( v == pos->second.end() ) {
            // this is a new request, so add it and sort the vector
            std::string cid(msg->id);
            std::string input(msg->input_tag);
            Control c(cid, msg->controlType, input, msg->priority, CONTROL_OFF);
            pos->second.push_back(c);
            std::sort(pos->second.begin(), pos->second.end());
        }
    }

    return 0;
}

/*****************************************************************************
 * Function:            unregisterControl
 * Description:
 *
 * @param               id     -> control identifier
 * @param               pri    -> priority
 * @param               output -> output (e.g. "o_rly05")
 *
 * @return              0 on success; -1 on error
 *****************************************************************************/
static int unregisterControl(const char *id, unsigned int pri, const char *output)
{
    // attempt to find the output in the map
    StringOutputVector_t::iterator pos;
    bool found = false;

    pos = outputMap.find(output);
    if ( pos == outputMap.end() ) {
        printf("%s: failed to find the designated output %s\n",
               __func__, output);
        return -1;
    }

    // find the control in the list
    vector<Control>::iterator v;
    for ( v = pos->second.begin(); v != pos->second.end(); ++v) {
        if ( strcmp(v->getId().c_str(), id) == 0 )  {
            // delete this entry
            pos->second.erase(v);
            found = true;
            break;
        }
    }

    if ( !found ) {
        logError("%s: failed to find control %s in list", __func__, id);
        return -1;
    }
    return 0;
}

/*****************************************************************************
 * Function:            loadPersistentOutputs
 * Description:         build the output map
 *
 * @param               none
 * @return              none
 *****************************************************************************/
static void loadPersistentOutputs(void)
{
    bool status;
    cJSON *root;

    printf("\rLoading persistent outputs:\n");

    std::vector<mDot::mdot_file> file_list = GLOBAL_mdot->listUserFiles();

    for (std::vector<mDot::mdot_file>::iterator i = file_list.begin(); i != file_list.end(); ++i) {
        // load in all of the output files 
        if( strncmp( i->name, OUTPUT_STR, strlen(OUTPUT_STR)) == 0 ) {
            char scratchBuf[MAX_FILE_SIZE];

            status = GLOBAL_mdot->readUserFile(i->name, scratchBuf, MAX_FILE_SIZE);
            if( status != true ) {
                logError("%s: failed to read %s", __func__, i->name);
            } else {
                logInfo("%s: successfully read %s", __func__, i->name);
            }
    
            root = cJSON_Parse(scratchBuf);
            //parse( json_value, scratchBuf );
            std::string id = cJSON_GetObjectItem(root,"id")->valuestring;
            printf("\r   output %s loaded\n", i->name);
            cJSON_Delete(root);

            // emplace the empty control vector into the output map
            vector<Control> v;
            outputMap[id] = v;
        }
    }
}

/*****************************************************************************
 * Function:            DisplayOutputs
 * Description:         Display a list of outputs and their controls
 *
 * @param               none
 * @return              none
 *****************************************************************************/
void DisplayOutputs(void)
{
    StringOutputVector_t::iterator pos;
    
    for ( pos = outputMap.begin(); pos != outputMap.end(); ++pos ) {
        if ( pos->second.empty() ) {
            printf("\r  [%s]-> [no controls] \n", pos->first.c_str());
        } else {
            printf("\r  [%s]-> ", pos->first.c_str());
            vector<Control>::iterator i;
            for ( i = pos->second.begin(); i != pos->second.end(); ++i ) {
                i->display();
            }
            printf("\n");
        }
    }
    printf("\r\n");
}
