#ifndef _MOD_H_
#define _MOD_H_

// -------------------------------------------------------------------------------------------------
// globals

#define MOD_DATA 4
#define MOD_IREG 60

extern unsigned short int mod_data[MOD_DATA];
extern unsigned short int mod_ireg[MOD_IREG];

extern int mod_verbose;

// -------------------------------------------------------------------------------------------------
// modbus standard communication structures: (from slave perspective)
//
// MOD_FUNC_GET_COIL (NBYT = total bytes of data transmitted)
// MOD_FUNC_GET_DISC
//
//      byte:           0           1           2           3           4           5           6           7
//      received:   ADDR        FUNC        SREG-H  SREG-L  NREG-H  NREG-L  RCRC-L  RCRC-H
//      transmit:   ADDR        FUNC        NBYT        DATA-H  DATA-L  TCRC-L  TCRC-H  NULL
//
// MOD_FUNC_GET_IREG (NBYT = total bytes of data transmitted)
// MOD_FUNC_GET_HREG
//
//      byte:           0           1           2           3           4           5           6           7
//      received:   ADDR        FUNC        SREG-H  SREG-L  NREG-H  NREG-L  RCRC-L  RCRC-H
//      transmit:   ADDR        FUNC        NBYT        DATA-H  DATA-L  TCRC-L  TCRC-H  NULL
//
// MOD_FUNC_SET_COIL
//
//      byte:           0           1           2           3           4           5           6           7           8
//      received:   ADDR        FUNC        SREG-H  SREG-L  DATA-H  DATA-L  RCRC-L  RCRC-H
//      transmit:   ADDR        FUNC        SREG-H  SREG-L  DATA-H  DATA-L  TCRC-L  TCRC-H  NULL
//
// MOD_FUNC_SET_COILS
//
//      byte:           0           1           2           3           4           5           6           7           8           9           10
//      received:   ADDR        FUNC        SREG-H  SREG-L  NREG-H  NREG-L  NBYT        DATA-H  DATA-L  RCRC-L  RCRC-H
//      transmit:   ADDR        FUNC        SREG-H  SREG-L  NREG-H  NREG-L  TCRC-L  TCRC-H  NULL
//
// MOD_FUNC_SET_HREG
//
//      byte:           0           1           2           3           4           5           6           7           8
//      received:   ADDR        FUNC        SREG-H  SREG-L  DATA-H  DATA-L  RCRC-L  RCRC-H
//      transmit:   ADDR        FUNC        SREG-H  SREG-L  DATA-H  DATA-L  TCRC-L  TCRC-H  NULL
//
// MOD_FUNC_SET_HREGS
//
//      byte:           0           1           2           3           4           5           6           7           8           9           10
//      received:   ADDR        FUNC        SREG-H  SREG-L  NREG-H  NREG-L  NBYT        DATA-H  DATA-L  RCRC-L  RCRC-H
//      transmit:   ADDR        FUNC        SREG-H  SREG-L  NREG-H  NREG-L  TCRC-L  TCRC-H  NULL
//
// response after error (EFUNC = FUNC + 0x80)
//
//      byte:           0           1           2           3           4           5
//      transmit:   ADDR        EFUNC       ERROR       TCRC-L  TCRC-H  NULL
//
// -----------------------------------------------------------------------------
// modbus definitions

enum // function codes
{
    MOD_FUNC_00,
    MOD_FUNC_GET_COIL,      // 01: read coil status
    MOD_FUNC_GET_DISC,          // 02: read discrete inputs
    MOD_FUNC_GET_HREG,      // 03: read holding registers
    MOD_FUNC_GET_IREG,      // 04: read input registers
    MOD_FUNC_SET_COIL,      // 05: force single coil
    MOD_FUNC_SET_HREG,      // 06: preset single register
    MOD_FUNC_07,
    MOD_FUNC_08,
    MOD_FUNC_09,
    MOD_FUNC_10,
    MOD_FUNC_11,
    MOD_FUNC_12,
    MOD_FUNC_13,
    MOD_FUNC_14,
    MOD_FUNC_SET_COILS,     // 15: force multiple coils
    MOD_FUNC_SET_HREGS,     // 16: preset multiple registers
    MOD_FUNC_NUM
};

enum // error codes ( * = used by Nubio )
{
    MOD_ERROR_NONE,
    MOD_ERROR_ILL_FUNC,     // 01: * slave received illegal function code
    MOD_ERROR_ILL_ADDR,     // 02: * slave received illegal register address
    MOD_ERROR_ILL_DATA,     // 03: * slave received illegal register value
    MOD_ERROR_SLAVE_FAIL,   // 04: * slave device-specific failure
    MOD_ERROR_SLAVE_ACK,    // 05: * slave ACK
    MOD_ERROR_SLAVE_BUSY,   // 06: slave is busy
    MOD_ERROR_SLAVE_NACK,   // 07: * slave NACK (Nubio returns this for CRC error)
    MOD_ERROR_SLAVE_PARITY, // 08: slave memory parity error
    MOD_ERROR_RX_TIMEOUT,   // 09: master receive timeout
    MOD_ERROR_TX_TIMEOUT,   // 10: master transmit timeout
    MOD_ERROR_CRC,          // 11: master CRC error
    MOD_ERROR_NUM
};

// -------------------------------------------------------------------------------------------------
// function prototypes

void mod_init(void);
void mod_clear(void);

void mod_process(void);

void mod_ireg_dump(void);

unsigned short int mod_calc_crc(unsigned char *buffer, int length);

int mod_read(int inp_addr, int inp_func, int inp_sreg, int inp_nreg, unsigned char *rd_buf);
int mod_write(int inp_addr, int inp_func, int inp_sreg, int inp_nreg, unsigned char *xmt_buf);

// -------------------------------------------------------------------------------------------------

#endif // _MOD_H_
