/******************************************************************************
 *
 * File:                TimerControl.cpp
 * Desciption:          ICE Timer Control Class implementation
 *
 *****************************************************************************/
#include "TimerControl.h"
#include "mDot.h"
#include "cJSON.h"
#include "global.h"
#include <string>
#include <iostream>
#include <iomanip>

extern mDot *GLOBAL_mdot;

//
// method:      load
// description: load the pertinents from the control file
//
// @param       _controlFile
// @return      true if loaded; false otherwise
//
bool TimerControl::load(string _controlFile)
{
    // try to open the control file
    mDot::mdot_file file = GLOBAL_mdot->openUserFile(_controlFile.c_str(), mDot::FM_RDONLY);
    if ( file.fd < 0 ) {
        return false;
    }

    // read the data into a buffer
    char dataBuf[350];

    int bytes_read = GLOBAL_mdot->readUserFile(file, (void *)dataBuf, sizeof(dataBuf));
    if ( bytes_read != sizeof(dataBuf) ) {
        logError("%s: failed to read %d bytes from %s", __func__, sizeof(dataBuf), controlFile.c_str());
        return false;
    }

    // close the file
    GLOBAL_mdot->closeUserFile(file);

    cJSON * root = cJSON_Parse(dataBuf);
    
    id              = cJSON_GetObjectItem(root,"id")->valuestring;
    output          = cJSON_GetObjectItem(root, "output")->valuestring;
    priority        = atoi(cJSON_GetObjectItem(root, "priority")->valuestring);
    startTime       = atoi(cJSON_GetObjectItem(root, "starttime")->valuestring);
    duration        = atoi(cJSON_GetObjectItem(root, "duration")->valuestring);
    
    cJSON_Delete(root);
    
    return true;
}

//
// method:      start
// description: initialize the control
//
// @param       none
// @return      none
//
void TimerControl::start(void)
{
    currentState = STATE_OFF;
}

//
// method:      timerStart
// description: examine the timestamp to determine if the timer control
//              should be running
//
// @param       none
// @return      true if timer should be running; false otherwise
//
bool TimerControl::timerStart(void)
{
    unsigned long currentTime = time(NULL);

    if ( currentTime < startTime )
        return false;

    if ( currentTime >= startTime && currentTime <= (startTime + duration) ) {
        return true;
    }
    return false;
}
//
// method:      timerStop
// description: determines if a running timer should has reached its duration
//
// @param       none
// @return      true if the timer has expired; false otherwise
//
bool TimerControl::timerStop(void)
{
    if ( time(NULL) >= startTime + duration ) {
        return true;
    }
    return false;
}

//
// method:      update
// description: run the simplified state machine
//
// @param       none
// @return      none
//
TimerError_t TimerControl::update(void)
{
    TimerError_t rc = TIMER_CONTROL_OK;

    switch ( this->currentState ) {
        case STATE_OFF:
            if ( this->timerStart() ) {
                currentState = STATE_RUNNING;
                this->startFeed();
            }
            break;
        case STATE_RUNNING:
            if ( this->timerStop() ) {
                currentState = STATE_OFF;
                this->stopFeed();
                this->unregisterControl();
                rc = TIMER_CONTROL_DESTROY;
            }
            break;
        case STATE_DISABLED:
            // not implelmented
        default:
            break;
    }
    return rc;

}

//
// method:      startFeed
// description: signal the output thread to start a feed
//
// @param       none
// @return      none
void TimerControl::startFeed(void)
{
    logInfo("%s: %s attempting to start feed on relay %s\n",
            __func__, controlFile.c_str(), output.c_str());

    OutputControlMsg_t *output_mail = OutputMasterMailBox.alloc();
    memset(output_mail, 0, sizeof(OutputControlMsg_t));

    output_mail->action      = ACTION_CONTROL_ON;
    output_mail->controlType = CONTROL_TIMER;
    output_mail->priority    = priority;

    strncpy(output_mail->output_tag, this->output.c_str(), sizeof(output_mail->output_tag)-1);
    strncpy(output_mail->id, this->id.c_str(), sizeof(output_mail->id)-1);
    OutputMasterMailBox.put(output_mail);
}

//
// method:      stopFeed
// description: signal the output thread to stop a feed
//
// @param       none
// @return      none
void TimerControl::stopFeed(void)
{
    logInfo("%s: %s attempting to start feed on relay %s\n",
            __func__, controlFile.c_str(), output.c_str());

    OutputControlMsg_t *output_mail = OutputMasterMailBox.alloc();
    memset(output_mail, 0, sizeof(OutputControlMsg_t));

    output_mail->action      = ACTION_CONTROL_OFF;
    output_mail->controlType = CONTROL_TIMER;
    output_mail->priority    = priority;

    strncpy(output_mail->output_tag, this->output.c_str(), sizeof(output_mail->output_tag)-1);
    strncpy(output_mail->id, this->id.c_str(), sizeof(output_mail->id)-1);
    OutputMasterMailBox.put(output_mail);
}

//
// Method:              unregisterControl
// Description:         send OFF indication to Output Master for this control's
//                      relay
//
// @param               none
// @return              none
//
void TimerControl::unregisterControl(void)
{
    logInfo("%s: %s attempting to unregister %s\n",
            __func__, controlFile.c_str());

    OutputControlMsg_t *output_mail = OutputMasterMailBox.alloc();
    memset(output_mail, 0, sizeof(OutputControlMsg_t));

    output_mail->action       = ACTION_CONTROL_UNREGISTER;
    output_mail->controlType  = CONTROL_TIMER;
    output_mail->priority     = priority;
    strncpy(output_mail->output_tag, this->output.c_str(), sizeof(output_mail->output_tag)-1);
    strncpy(output_mail->id, this->id.c_str(), sizeof(output_mail->id)-1);

    OutputMasterMailBox.put(output_mail);
}

//
// methid:      display
// description: display the elements of this timer control object
//
// @param       none
// @return      none
//
void TimerControl::display(void)
{
    string mapper[] = { "OFF",
                        "RUNNING",
                        "DISABLED"
                      };

    printf("\r\n");
    cout << left << setw(10) << setfill(' ') << "timer:";
    cout << left << setw(40) << setfill(' ') << controlFile;
    cout << left << setw(20) << setfill(' ') << id;
    cout << left << setw(20) << setfill(' ') << output;
    cout << "pri:" << left << setw(12) << setfill(' ') << priority;
    cout << "start:" << left << setw(12) << setfill(' ') << startTime;
    cout << "duration: " << left << setw(12) << setfill(' ') << duration;
    cout << "end: " << left << setw(12) << setfill(' ') << startTime + duration;
    cout << "remaining: " << left << setw(12) << setfill(' ') << startTime + duration - time(NULL);
    cout << left << setw(12) << setfill(' ') << mapper[currentState]; 
    cout.flush(); 
}