/******************************************************************************
*
* File:                SetpointControl.cpp
* Desciption:          ICE Setpoint Control class implementation
*
*****************************************************************************/
#include "SetpointControl.h"
#include "mDot.h"
#include "cJSON.h"
#include "ModbusMasterApi.h"
#include "global.h"
#include <string>
#include <iostream>
#include <iomanip>

extern mDot *GLOBAL_mdot;

//
// method:          load
// description:     open the configuration file and assign data to the
//                  setpoint control object
//
// @param           controlFile -> name of the control file
// @return          true if data is assigned; false on error
//
bool SetpointControl::load(string _controlFile)
{
    controlFile = _controlFile;

    // open and read from the control file
    mDot::mdot_file file = GLOBAL_mdot->openUserFile(controlFile.c_str(), mDot::FM_RDONLY);
    if ( file.fd < 0 ) {
        logError("%s: failed to open %s\n", __func__, controlFile.c_str());
        return false;
    }

    // read the data into a buffer
    char dataBuf[MAX_FILE_SIZE];

    int bytes_read = GLOBAL_mdot->readUserFile(file, (void *)dataBuf, sizeof(dataBuf));
    if ( bytes_read != sizeof(dataBuf) ) {
        logError("%s: failed to read %d bytes from %s", __func__, sizeof(dataBuf), controlFile.c_str());
        // caller should destroy the object
        return false;
    }

    // close the file
    GLOBAL_mdot->closeUserFile(file);

    // parse the json data
    cJSON * root = cJSON_Parse(dataBuf);

    if ( !cJSON_HasObjectItem(root, "id") ||
            !cJSON_HasObjectItem(root, "priority") ||
            !cJSON_HasObjectItem(root, "input") ||
            !cJSON_HasObjectItem(root, "output") ||
            !cJSON_HasObjectItem(root, "setpoint") ||
            !cJSON_HasObjectItem(root, "prodfact") ||
            !cJSON_HasObjectItem(root, "actingDir") ||
            !cJSON_HasObjectItem(root, "halert") ||
            !cJSON_HasObjectItem(root, "lalert") ||
            !cJSON_HasObjectItem(root, "hfs") ||
            !cJSON_HasObjectItem(root, "lfs") ||
            !cJSON_HasObjectItem(root, "tol") ) {
        logError("Setpoint control is missing expected tags\n");
        cJSON_Delete(root);
        return false;
    }

    id              = cJSON_GetObjectItem(root,"id")->valuestring;
    priority        = atoi(cJSON_GetObjectItem(root,"priority")->valuestring);
    input           = cJSON_GetObjectItem(root,"input")->valuestring;
    output          = cJSON_GetObjectItem(root,"output")->valuestring;
    setpoint        = atof(cJSON_GetObjectItem(root,"setpoint")->valuestring);
    productFactor   = atof(cJSON_GetObjectItem(root, "prodfact")->valuestring);
    actingDir       = atoi(cJSON_GetObjectItem(root, "actingDir")->valuestring);
    highAlert       = atof(cJSON_GetObjectItem(root, "halert")->valuestring);
    lowAlert        = atof(cJSON_GetObjectItem(root, "lalert")->valuestring);
    highFailsafe    = atof(cJSON_GetObjectItem(root, "hfs")->valuestring);
    lowFailsafe     = atof(cJSON_GetObjectItem(root, "lfs")->valuestring);
    tolerance       = atof(cJSON_GetObjectItem(root, "tol")->valuestring);
    
    cJSON_Delete(root);

    return true;        // object created successfully
}

//
// method:          start
// description:     start the setpoint control
//
// @param           none
// @return          none
//
void SetpointControl::start(void)
{
    // this is the initial state; what else needs to be done??
    this->currentState = STATE_STARTUP;
}

//
// method:          update
// description:     based on the state of the control, check for
//                  under limit and over limit values, adjust the
//                  state accordingly
//
// @param           none
// @return          none
//
void SetpointControl::update(void)
{
    switch (this->currentState) {
        case STATE_INIT:
            // do nothing
            break;
        case STATE_STARTUP:
            if ( this->underLimit() ) {
                // start the feed right away
                this->startFeed();
                this->currentState = STATE_CONTROL_ON;
            } else {
                this->currentState = STATE_CONTROL_OFF;
                this->stopFeed();
            }
            break;
        case STATE_CONTROL_ON:
            if ( this->overLimit() ) {
                // stop the feed
                this->stopFeed();
                this->currentState = STATE_CONTROL_OFF;
            } else {
                // do nothing
            }
            break;
        case STATE_CONTROL_OFF:
            if ( this->underLimit() ) {
                // start the feed
                this->startFeed();
                this->currentState = STATE_CONTROL_ON;
            } else {
                // do nothing
            }
            break;
            //case STATE_CONTROL_DISABLED:
            //case STATE_CONTROL_PAUSED:
        default:
            break;
    }
}

//
// method:              overLimit
// description:         (see @return)
//
// @param               none
// @return              true if product is over the upper limit for normal mode
//                      or under the limit for reverse mode; false otherwise
//
bool SetpointControl::overLimit(void)
{
    ModbusValue value;
    ModbusMasterReadRegister( input, &value );
    float flimit;

    if ( !actingDir ) {
        flimit = setpoint + tolerance;
        return (value.value > flimit);
    } else {
        flimit = setpoint - tolerance;
        return (value.value < flimit);
    }
}

//
// method:              underLimit
// description:         (see @return)
//
// @param               none
// @return              true if product is under lower limit for normal mode or
//                      over the upper limit for reverse mode; false otherwise
//
bool SetpointControl::underLimit(void)
{
    ModbusValue value;
    ModbusMasterReadRegister( input, &value );
    float  flimit;

    if ( !actingDir ) {
        flimit = setpoint - tolerance;
        return (value.value < flimit);
    } else {
        flimit = setpoint + tolerance;
        return (value.value > flimit);
    }
}

//
// method:              startFeed()
// description:         send ON indication to Output Master for this control's
//                      relay
//
// @param               none
// @return              none
//
void SetpointControl::startFeed(void)
{
    logInfo("%s: %s attempting to start feed on relay %s\n",
            __func__, controlFile.c_str(), output.c_str());

    OutputControlMsg_t *output_mail = OutputMasterMailBox.alloc();
    memset(output_mail, 0, sizeof(OutputControlMsg_t));

    output_mail->action = ACTION_CONTROL_ON;
    output_mail->controlType = CONTROL_SETPOINT;
    strncpy(output_mail->input_tag,  this->input.c_str(),  sizeof(output_mail->input_tag)-1);
    strncpy(output_mail->output_tag, this->output.c_str(), sizeof(output_mail->output_tag)-1);
    output_mail->priority = this->priority;
    strncpy(output_mail->id, this->id.c_str(), sizeof(output_mail->id)-1);
    OutputMasterMailBox.put(output_mail);
}

//
// method:              stopFeed
// description:         send OFF indication to Output Master for this control's
//                      relay
//
// @param               none
// @return              none
//
void SetpointControl::stopFeed(void)
{
    logInfo("%s: %s attempting to stop feed on relay %s\n",
            __func__, controlFile.c_str(), output.c_str());

    OutputControlMsg_t *output_mail = OutputMasterMailBox.alloc();
    memset(output_mail, 0, sizeof(OutputControlMsg_t));

    output_mail->action = ACTION_CONTROL_OFF;
    output_mail->controlType = CONTROL_SETPOINT;
    strncpy(output_mail->input_tag,  this->input.c_str(),  sizeof(output_mail->input_tag)-1);
    strncpy(output_mail->output_tag, this->output.c_str(), sizeof(output_mail->output_tag)-1);
    output_mail->priority = this->priority;
    strncpy(output_mail->id, this->id.c_str(), sizeof(output_mail->id)-1);
    OutputMasterMailBox.put(output_mail);
}

//
// dethod:              unregisterControl
// description:         send OFF indication to Output Master for this control's
//                      relay
//
// @param               none
// @return              none
//
void SetpointControl::unregisterControl(void)
{
    logInfo("%s: %s attempting to unregister %s\n",
            __func__, controlFile.c_str());

    OutputControlMsg_t *output_mail = OutputMasterMailBox.alloc();
    memset(output_mail, 0, sizeof(OutputControlMsg_t));

    output_mail->action      = ACTION_CONTROL_UNREGISTER;
    output_mail->controlType = CONTROL_MANUAL;
    output_mail->priority     = this->priority;
    strncpy(output_mail->output_tag, this->output.c_str(), sizeof(output_mail->output_tag)-1);
    strncpy(output_mail->id, this->id.c_str(), sizeof(output_mail->id)-1);

    OutputMasterMailBox.put(output_mail);
}

//
// method:              display
// description:         display the control's information
//
// @param               none
// @return              none
//
void SetpointControl::display(void)
{
    // NOTE: this mapping must be 1:1 with "State" enumeration in SetpointControl.h
    string mapper[] = { "INIT",
                        "STARTUP",
                        "CONTROL_OFF",
                        "CONTROL_ON",
                        "CONTROL_DISABLE",
                        "CONTROL_PAUSE",
                        "CONTROL_MAX"
                      };


    ModbusValue inputValue;
    ModbusMasterReadRegister(input, &inputValue);

    printf("\r\n");
    std::cout << left << setw(10) << setfill(' ') << "setpoint: ";
    std::cout << left << setw(40) << setfill(' ') << controlFile;
    std::cout << left << setw(20) << setfill(' ') << id;
    std::cout << left << setw(6)  << setfill(' ') << priority;
    std::cout << left << setw(20) << setfill(' ') << input;
    std::cout << left << setw(20) << setfill(' ') << output;
    std::cout << left << setw(8)  << setfill(' ') << setpoint;
    std::cout << left << setw(12) << setfill(' ') << (actingDir ? "direct" : "indirect");
    std::cout << left << setw(16) << setfill(' ') << mapper[currentState];
    std::cout << right << setw(8)  << setfill(' ') << setpoint + tolerance << " <- ";
    std::cout << left << setw(8)  << setfill(' ') << inputValue.value << " -> ";
    std::cout << left << setw(8)  << setfill(' ') << setpoint - tolerance;
    std::cout << left << setw(10) << setfill(' ') << highFailsafe << " : " << lowFailsafe;

    std::cout.flush();
}