#ifndef FAILSAFECONTROL_H
#define FAILSAFECONTROL_H

#include <stdio.h>
#include <string>
#include "global.h"

class FailsafeControl
{
private:
    std::string     controlFile;            // the control file
    std::string     id;                     // unique control identifier
    std::string     input;                  // sensor to read
    std::string     output;                 // output to manipulate
    unsigned int    priority;               // control priority (typically 700)

    typedef struct  hfs_tag {
        float       value;                  // high-failsafe trigger value
        float       dutyCycle;              // percentage of on-time
        float       interval;               // in minutes
    } hfs;
    typedef struct lfs_tag {
        float       value;                  // low-failsafe trigger value
        float       dutyCycle;              // percentag of on-time
        float       interval;               // in minutes
    } lfs;

    lfs             lfs_data;               // Low Failsafe data
    hfs             hfs_data;               // High Failsafe data

    typedef struct timer_tag {
        unsigned long   offTime;            // epoch time
        unsigned long   expirationTime;     // epoch time
    } Timer_t;

    Timer_t         duty_timer;             // the duty timer

    enum State {
        STATE_INIT,                         // object instantiated
        STATE_START,                        // control has been started
        STATE_CONTROL_OFF,                  // control is not above/below limit
        STATE_CONTROL_ON_LFS,               // low-failsafe with output on
        STATE_CONTROL_OFF_LFS,              // low-failsafe with output off
        STATE_CONTROL_ON_HFS,               // high-failsafe with output on
        STATE_CONTROL_OFF_HFS,              // high-failsafe with output off
        STATE_CONTROL_SENSOR_ERROR,         // input sensor is faulted
        STATE_MAX
    };
    State           currentState;

    bool aboveHighFailsafe(void);           // boolean check if above HFS watermark
    bool belowLowFailsafe(void);            // boolean check if below LFS watermark

    void startHfsDutyTimer(void);           // start the HFS duty timer
    void stopHfsDutyTimer(void);            // stop the HFS duty timer

    void startLfsDutyTimer(void);           // start the LFS duty timer
    void stopLfsDutyTimer(void);            // stop the LFS duty timer

    bool dutyOnExpired(void);               // boolean check if duty ON-time expired
    bool dutyOffExpired(void);              // boolean check if duty OFF-time expired
    
    bool sensorError(void);                 // boolean check if input is faulted

    void sendMailToOutput(OutputAction action);

public:
    // ctor
    FailsafeControl() {
        currentState = STATE_INIT;
    }
    // dtor
    ~FailsafeControl() {
        // "erased...from existence!" -- Doc Brown
        printf("\r%s invoked\n", __func__);
    }

    // load the control data from the configuration file
    bool load(std::string filename);

    // start the control
    void start(void);

    // update the control
    void update(void);

    // unregister the control with the output task
    void unregisterControl(void);


    // diplay the pertinents
    void display(void);
};

#endif