/******************************************************************************
 *
 * File:                global.h
 * Desciption:          global data
 *
 *****************************************************************************/
#ifndef GLOBAL_H
#define GLOBAL_H

#include "mbed.h"
#include "rtos.h"
#include "ntshell.h"
#include "mDot.h"

// file naming prefix conventions
#define CONTROL_SP_STR      "control_sp_"
#define CONTROL_TM_STR      "control_tm_"
#define CONTROL_MN_STR      "control_mn_"
#define CONTROL_COMP_STR    "control_comp_"
#define CONTROL_CA_STR      "control_ca_"
#define CONTROL_FS_STR      "control_fs_"

// declare max file sizes 
#define MAX_FILE_SIZE       350  

// declare stack sizes
#define MODBUS_MASTER_STACK_SIZE        (1024 * 2)
#define OUTPUT_TASK_STACK_SIZE          (1024 * 4)
#define CONFIG_HANDLER_STACK_SIZE       (1024*4 + 512)
#define CONTROL_TASK_STACK_SIZE         (1024)
#define BLE_DATA_HANDLER_STACK_SIZE     (1024 * 2)
#define CLOUD_DATA_HANDLER_STACK_SIZE   (1024 * 4)  
#define ANALYTICS_LOGGER_STACK_SIZE     (1024 * 2)
 
extern int sig_output_continue;
extern int sig_config_continue;

extern osThreadId mainThreadId;

typedef enum action_tag {
    ACTION_CREATE,
    ACTION_MODIFY,
    ACTION_DESTROY,
    ACTION_READ_FILE,
    ACTION_EXEC_CMD
} Action_t;

typedef enum control_tag {
    CONTROL_TIMER       = 0,
    CONTROL_PID         = 1,
    CONTROL_SETPOINT    = 2,
    CONTROL_COMPOSITE   = 3,
    CONTROL_MANUAL      = 4,
    CONTROL_ALGORITHM   = 5,
    CONTROL_FAILSAFE    = 6
} Control_t;

typedef enum thread_name_tag {
    ANALYTICS_LOGGER        = 0,
    BLE_HANDLER             = 1,
    CLOUD_DATA_HANDLER      = 2,
    CONFIG_HANDLER          = 3,
    CONTROL_TASK            = 4,
    OUTPUT_TASK             = 5,
} ThreadName_t;


// message data type
typedef struct message_tag {
    Action_t        action;
    Control_t       control;
    char            controlFile[64];
} Message_t;

typedef enum {
    OFF = 0,
    ON = 1
} State;

typedef enum output_action {
    ACTION_NEW,
    ACTION_CONTROL_ON,
    ACTION_CONTROL_OFF,
    ACTION_CONTROL_REGISTER,
    ACTION_CONTROL_UNREGISTER
} OutputAction;

// output control request message
typedef struct output_control_req_t {
    OutputAction    action;
    Control_t       controlType;
    char            controlFile[32];
    char            input_tag[32];
    char            output_tag[32];
    char            id[32];
    unsigned int    priority;
} OutputControlMsg_t;

// analytics logger request message
typedef struct analytics_logger_req_t {
    char            timestamp[32];
    char            log_entry[96];
} AnalyticsLoggerReq_t;

// analytics logger request message
typedef struct ble_handler_req_t {
    char            reply[500];
} BLEHandlerReq_t;

// message data type
typedef struct modbus_master_tag {
    Action_t        action;
    Control_t       control;
    ThreadName_t    replyThread;
    char            controlFile[256];
} ModbusMasterReq_t;

extern Mail<Message_t, 16> MailBox;
extern Mail<ModbusMasterReq_t, 2> ModbusMasterMailBox;
extern Mail<OutputControlMsg_t, 16> OutputMasterMailBox;
extern Mail<BLEHandlerReq_t, 1> BLEHandlerMailBox;

extern ntshell_t ntshell;

extern int  func_read(char *buf, int cnt);
extern int  func_write(const char *buf, int cnt);
extern int  func_cb_ntshell(const char *text);
extern void func_cb_ntopt(int argc, char **argv);
extern mDot *GLOBAL_mdot;

typedef enum register_type_tag {
    REG_TYPE_NONE,
    REG_TYPE_INPUT,
    REG_TYPE_OUTPUT,
    REG_TYPE_VINPUT,
    REG_TYPE_VOUTPUT,
} RegisterType_t;

struct ModbusRegister {
    std::string name;
    std::string units;
    double min;
    double max;
    int node;
    int reg;
    int rtype;
    int type;
    int size;
    int order;
    int rfreq;
    RegisterType_t regType;
    std::string vcmd;
    std::string fmt;
    float float_value;
    uint32_t uint32_value;
    uint16_t uint16_value;
    uint32_t errflag;
    bool simulated;
};

struct VirtualCommand {
    float c;
    std::string tag;
    std::string opl;
    std::string opr;
    std::string op;
};

struct SimulateInput {
    float start_value;
    float min;
    float max;
    float up_step;
    float down_step;
    bool moving_up;
};

typedef enum byte_order {
    BigEndian                   = 0,
    LittleEndian                = 1,
    BigEndianReverseWord        = 2,
    LittleEndianReversWord      = 3,
} Byte_Order;

typedef enum event_reason_tag {
    EVENT_REASON_AUTO,
    EVENT_REASON_MANUAL,
    EVENT_REASON_TIMER,
    EVENT_REASON_FLOW,
    EVENT_REASON_FAILSAFE,
    EVENT_REASON_NO_CONTROL
} EventReason_t;

typedef struct event_reason_struct_t {
    EventReason_t   eventReason;
    char            inputTag[32];
    char            outputTag[32];
    float           inputValue;
    float           outputValue;
} EventReasonStruct_t;

extern std::map<std::string,ModbusRegister> ModbusRegisterMap;
extern std::map<std::string,SimulateInput> SimulateInputMap;

extern I2C* i2c;

extern Thread *GLOBAL_analyticsLogger_thread;
extern Thread *GLOBAL_modbusMaster_thread;
extern Thread *GLOBAL_BLE_thread;
extern Thread *GLOBAL_CDH_thread;
extern Thread *GLOBAL_configHandler_thread;
extern Thread *GLOBAL_controlTask_thread;
extern Thread *GLOBAL_outputTask_thread;

#define RELAY_STATUS_AUTO_ON             129
#define RELAY_STATUS_AUTO_OFF            128
#define RELAY_STATUS_MANUAL_ON           65
#define RELAY_STATUS_MANUAL_OFF          64
#define RELAY_STATUS_FAILSAFE_ON         33
#define RELAY_STATUS_FAILSAFE_OFF        32
#define RELAY_STATUS_FORCED_OFF          16
#define RELAY_STATUS_FLOW_FAILSAFE_ON    9
#define RELAY_STATUS_FLOW_FAILSAFE_OFF   8
#define RELAY_STATUS_NOT_CONTROLLED      0

// ---------------------------------------------------------------------------------------------------------------------
// pinout

#define XBEE_P01 VIN    // pin 01
#define MOD_TX   PA_2   // pin 02
#define MOD_RX   PA_3   // pin 03
#define SPI_MISO PA_6   // pin 04
#define XBEE_P05 NRST   // pin 05
#define I2C_SCL  PA_8   // pin 06
#define I2C_SDA  PC_9   // pin 07
#define XBEE_P08 NC     // pin 08
#define DMOD     PA_11  // pin 09
#define XBEE_P10 GND    // pin 10

#define MOD_RE   PB_1   // pin 20
#define MOD_DE   PB_0   // pin 19
#define SPI_SCK  PA_5   // pin 18
#define SPI_CS   PA_4   // pin 17
#define DOUT2    PA_1   // pin 16
#define DOUT1    PC_1   // pin 15
#define XBEE_P14 NC     // pin 14
#define DINP2    PC_13  // pin 13
#define DINP1    PA_0   // pin 12
#define SPI_MOSI PA_7   // pin 11

#define XBEE_P21 VOUT
#define XBEE_P22 GND
#define XBEE_P23 PA_14
#define XBEE_P24 PA_13
#define XBEE_P25 PB_4
#define XBEE_P26 PB_3
#define CON_RX   PA_10
#define CON_TX   PA_9

// ---------------------------------------------------------------------------------------------------------------------
// globals

#define CON_BAUD 115200
#define MOD_BAUD  19200

extern I2C* i2c;

// -------------------------------------------------------------------------------------------------
// typedefs

typedef struct word2byte { union
{
    unsigned short int value;
    struct { unsigned char byte0; unsigned char byte1; };
};}
    WORD2BYTE;

typedef struct sint2byte { union
{
    signed long int value;
    struct { unsigned short int word0; unsigned short int word1; };
    struct { unsigned char byte0; unsigned char byte1; unsigned char byte2; unsigned char byte3; };
};}
    SINT2BYTE;

typedef struct uint2byte { union
{
    unsigned long int value;
    struct { unsigned short int word0; unsigned short int word1; };
    struct { unsigned char byte0; unsigned char byte1; unsigned char byte2; unsigned char byte3; };
};}
    UINT2BYTE;

typedef struct float2byte { union
{
    float value;
    struct { unsigned short int word0; unsigned short int word1; };
    struct { unsigned char byte0; unsigned char byte1; unsigned char byte2; unsigned char byte3; };
};}
    FLOAT2BYTE;

typedef struct float2uint { union
{
    float value;
    unsigned long int uint_value;
};}
    FLOAT2UINT;
// -------------------------------------------------------------------------------------------------

#define SUCCESS             0
#define ORDER_NOT_SUPPORTED 500
#define DATA_ARRAY_MISSING  501
#define UNKNOWN_OPERATION   502

#endif
