/******************************************************************************
 *
 * File:                ControlTask.cpp
 * Desciption:          source for the ICE Control task
 *
 *****************************************************************************/
 
#include "ControlTask.h"
#include <stdio.h>
#include "rtos.h"
#include "ConfigurationHandler.h"
#include "MTSLog.h"
#include "global.h"

// control functions 
static void serviceManualControls(void);
static void serviceTimerControls(void);
static void serviceSetpointControls(void);
static void serviceCompositeControls(void);
static void serviceFailsafeControls(void);

/*****************************************************************************
 * Function:            ControlTask()
 * Description:         This task will loop through all of the configured
 *                      controls
 *
 * @param               msg
 * @return              none
 *****************************************************************************/
void ControlTask(void const *args)
{
    printf("\r%s has started...\n", __func__);

    while ( true ) {
        serviceManualControls();

        serviceTimerControls();

        serviceSetpointControls();

        serviceCompositeControls();
        
        serviceFailsafeControls();

        Thread::wait(1000);     // do we need to wait?
    }
}

/*****************************************************************************
 * Function:            serviceManualControls()
 * Description:         service the manual controls
 *                      controls
 *
 * @param               none
 * @return              none
 *****************************************************************************/
static void serviceManualControls(void)
{
    // service the manual controls
    if ( !manualTable.empty() ) {
        StringManualMap::iterator pos;
        for ( pos = manualTable.begin(); pos != manualTable.end(); ++pos ) {
            pos->second->update();
        }
    }
}

/*****************************************************************************
 * Function:            serviceSetpointControls()
 * Description:         service the setpoint controls
 *                      controls
 *
 * @param               none
 * @return              none
 *****************************************************************************/
static void serviceSetpointControls(void)
{
    // service the setpoint controls
    if ( !setpointTable.empty() ) {
        StringSetpointMap::iterator pos;
        for ( pos = setpointTable.begin(); pos != setpointTable.end(); ++pos ) {
            pos->second->update();
        }
    }
}

/*****************************************************************************
 * Function:            serviceTimerControls()
 * Description:         service the timer controls
 *                      controls
 *
 * @param               none
 * @return              none
 *****************************************************************************/
static void serviceTimerControls(void)
{
    // service the timer controls
    if ( !timerTable.empty() ) {
        StringVectorTimerMap::iterator pos;
        for ( pos = timerTable.begin(); pos != timerTable.end(); ++pos ) {
            // the timer vector is sorted by starttime, so we only need
            // to service the first timer
            TimerError_t rc = pos->second.front()->update();
            if ( rc == TIMER_CONTROL_DESTROY ) {
                GLOBAL_mdot->deleteUserFile(pos->second.front()->getControlFile().c_str());
                logInfo("%s: deleted %s", __func__, pos->second.front()->getControlFile().c_str());
                // free the memory
                delete pos->second.front();
                pos->second.erase(pos->second.begin());
            }
        }
    }

}

/*****************************************************************************
 * Function:            serviceCompositeControls()
 * Description:         service the composite controls
 *                      controls
 *
 * @param               none
 * @return              none
 *****************************************************************************/
static void serviceCompositeControls(void)
{
    // service the setpoint controls
    if ( !compositeTable.empty() ) {
        StringCompositeMap::iterator pos;
        for ( pos = compositeTable.begin(); pos != compositeTable.end(); ++pos ) {
            pos->second->update();
        }
    }
}

/*****************************************************************************
 * Function:            serviceFailsafeControls()
 * Description:         service the failsafe controls
 *                      controls
 *
 * @param[in]           none
 * @return              none
 *****************************************************************************/
static void serviceFailsafeControls(void)
{
    // service the setpoint controls
    if ( !failsafeTable.empty() ) {
        StringFailsafeMap::iterator pos;
        for ( pos = failsafeTable.begin(); pos != failsafeTable.end(); ++pos ) {
            pos->second->update();
        }
    }
}