/******************************************************************************
 *
 * File:                ManualControl.cpp
 * Desciption:          ICE Manual Control Class implementation
 *
 *****************************************************************************/
#include "ManualControl.h"
#include "mDot.h"
#include "cJSON.h"
#include "ModbusMasterApi.h"
#include "global.h"
#include <string>
#include <iostream>
#include <iomanip>

extern mDot *GLOBAL_mdot;

//
// method:          load
// description:     open the configuration file and assign data to the
//                  setpoint control object
//
// @param           controlFile -> name of the control file
// @return          true if data is assigned; false on error
//
bool ManualControl::load(string _controlFile)
{
    controlFile = _controlFile;

    // open and read from the control file
    mDot::mdot_file file = GLOBAL_mdot->openUserFile(controlFile.c_str(), mDot::FM_RDONLY);
    if ( file.fd < 0 )
        return false;

    // read the data into a buffer
    char dataBuf[MAX_FILE_SIZE];

    int bytes_read = GLOBAL_mdot->readUserFile(file, (void *)dataBuf, sizeof(dataBuf));
    if ( bytes_read != sizeof(dataBuf) ) {
        logError("%s: failed to read %d bytes from %s", __func__, sizeof(dataBuf), controlFile.c_str());
        // caller should destroy the object
        return false;
    }

    // close the file
    GLOBAL_mdot->closeUserFile(file);

    // parse the json data
    cJSON * root = cJSON_Parse(dataBuf);

    if ( !cJSON_HasObjectItem(root, "id") ||
            !cJSON_HasObjectItem(root, "output") ||
            !cJSON_HasObjectItem(root, "type") ||
            !cJSON_HasObjectItem(root, "priority") ||
            !cJSON_HasObjectItem(root, "duration") ||
            !cJSON_HasObjectItem(root, "setpoint") ||
            !cJSON_HasObjectItem(root, "state") ||
            !cJSON_HasObjectItem(root, "percent") ) {
        logError("Manual control is missing expected tags\n");
        cJSON_Delete(root);
        return false;
    }

    id          = cJSON_GetObjectItem(root,"id")->valuestring;
    output      = cJSON_GetObjectItem(root,"output")->valuestring;
    type        = atoi(cJSON_GetObjectItem(root,"type")->valuestring);
    priority    = atoi(cJSON_GetObjectItem(root, "priority")->valuestring);
    duration    = atoi(cJSON_GetObjectItem(root, "duration")->valuestring);
    setpoint    = atof(cJSON_GetObjectItem(root, "setpoint")->valuestring);
    state       = atoi(cJSON_GetObjectItem(root, "state")->valuestring);
    percent     = atoi(cJSON_GetObjectItem(root, "percent")->valuestring);
    
    cJSON_Delete(root);

    return true;
}

//
// method:          start
// description:     start the manual control
//
// @param           none
// @return          none
//
void ManualControl::start(void)
{
    currentState = STATE_STARTUP;
}

//
// method:          update
// description:     the setpoint control's state machine
//
// @param           none
// @return          none
//
int ManualControl::update(void)
{
    int rc = 0;
    switch ( this->currentState ) {
        case STATE_INIT:
            // do nothing
            break;
        case STATE_STARTUP:
            if ( state ) {
                this->currentState = STATE_CONTROL_ON;
            } else {
                this->currentState = STATE_CONTROL_OFF;
            }
            this->powerOutput();
            break;
        case STATE_CONTROL_ON:
            if ( !state ) {
                this->currentState = STATE_CONTROL_OFF;
                this->powerOutput();
            }
            break;
        case STATE_CONTROL_OFF:
            if ( state ) {
                this->currentState = STATE_CONTROL_ON;
                this->powerOutput();
            }
            break;
        default:
            logError("%s unknown state %d\n", __func__, this->currentState);
            rc = -1;
            break;
    }
    return rc;
}

//
// method:              unregisterControl
// description:         unregister this control with the output master
//
// @param               none
// @return              none
//
int ManualControl::unregisterControl(void)
{
    logInfo("%s: Attempting to unregister %s\n",
            __func__, controlFile.c_str());

    OutputControlMsg_t *output_mail = OutputMasterMailBox.alloc();
    memset(output_mail, 0, sizeof(OutputControlMsg_t));

    output_mail->controlType = CONTROL_MANUAL;
    output_mail->action      = ACTION_CONTROL_UNREGISTER;
    output_mail->priority    = this->priority;
    strncpy(output_mail->output_tag, this->output.c_str(), sizeof(output_mail->output_tag)-1);
    strncpy(output_mail->id, this->id.c_str(), sizeof(output_mail->id)-1);

    OutputMasterMailBox.put(output_mail);
    return 0;
}

//
// method:          powerOutput
// description:     send a message to the output thread to power the relay
//
// @param           none
// @return          -1 on error; 0 otherwise
//
int ManualControl::powerOutput(void)
{
    printf("%s: %s attempting to manually turn %s relay %s\n",
           __func__, controlFile.c_str(), (state) ? "ON" : "OFF", id.c_str());

    OutputControlMsg_t *output_mail = OutputMasterMailBox.alloc();
    memset(output_mail, 0, sizeof(OutputControlMsg_t));

    output_mail->controlType    = CONTROL_MANUAL;
    output_mail->action         = (state) ? ACTION_CONTROL_ON : ACTION_CONTROL_OFF;
    output_mail->priority       = this->priority;
    strncpy(output_mail->output_tag, this->output.c_str(), sizeof(output_mail->output_tag)-1);
    strncpy(output_mail->id, this->id.c_str(), sizeof(output_mail->id)-1);

    OutputMasterMailBox.put(output_mail);
    return 0;
}

//
// method:          display
// description:     display the pertinents
//
// @param           none
// @return          none
//
void ManualControl::display(void)
{
    string mapper[] = { "INIT",
                        "STARTUP",
                        "CONTROL_ON",
                        "CONTROL_OFF"
                      };

    printf("\r\n");
    std::cout << left << setw(10) << setfill(' ') << "manual: ";
    std::cout << left << setw(40) << setfill(' ') << controlFile;
    std::cout << left << setw(20) << setfill(' ') << id;
    std::cout << left << setw(20) << setfill(' ') << output;
    std::cout << left << setw(6)  << setfill(' ') << type;
    std::cout << left << setw(6)  << setfill(' ') << priority;
    std::cout << left << setw(8)  << setfill(' ') << duration;
    std::cout << left << setw(8)  << setfill(' ') << percent;
    std::cout << left << setw(16) << setfill(' ') << mapper[currentState];
    std::cout.flush();
}