/*
 * ===============================================================
 *  Natural Tiny Shell (NT-Shell) Application example.
 *  Version 0.0.6
 * ===============================================================
 * Copyright (c) 2010-2011 Shinichiro Nakamura
 *
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation
 * files (the "Software"), to deal in the Software without
 * restriction, including without limitation the rights to use,
 * copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following
 * conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
 * OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
 * HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
 * WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
 * OTHER DEALINGS IN THE SOFTWARE.
 * ===============================================================
 */

#include "cmd.h"
#include <mbed.h>
#include <utility>
#include "ntshell.h"
#include "ntopt.h"
#include "global.h"
#include "ConfigurationHandler.h"
#include "ModbusMasterApi.h"
#include "LogLocalApi.h"
#include "LoggerApi.h"
#include "OutputTask.h"
#include "mDot.h"
#include "rtos.h"
#include "rtc.h"
#include "cJSON.h"

Serial serial(USBTX, USBRX);
ntshell_t ntshell;

extern mDot *GLOBAL_mdot;

typedef struct {
    char *command;                              // command (from shell)
    char *description;                          // descrption
    void (*func)(int argc, char **argv);        // callback function
} command_table_t;

// see cmd.h
const command_table_t cmdlist[] = {
    {"?",                   "help command",                                     cmd_help            },
    //{"create-control",      "create a control",                                 cmd_create          },
    {"cat",                 "cat a file",                                       cmd_cat             },
    {"cif",                 "create a test input file",                         cmd_cif             },
    {"cmf",                 "create a manual control file",                     cmd_cmf             },
    {"cmt",                 "create multiple timers",                           cmd_cmt             },
    {"cof",                 "create a test output file",                        cmd_cof             },
    {"create-fs",           "create a failsafe file",                           cmd_create_fs       },
    {"create-ca",           "create a control algorithm file",                  cmd_createCAlg      },
    {"create-comp",         "craete a composite control file",                  cmd_createComposite },
    {"create-mn",           "create a manual control",                          cmd_createManual    },
    {"create-sp",           "create a setpoint control",                        cmd_createSetpoint  },
    {"create-tm",           "create a timer control",                           cmd_createTimer     },
    {"create-temp",         "create temperature input",                         cmd_createTemp      },
    {"deep",                "dump EEP",                                         cmd_deep            },
    {"preload",             "pre-load phase-1 demo files",                      cmd_preload         },
    {"destroy-control",     "destroy a control",                                cmd_destroy         },
    {"heap",                "show heap statistics",                             cmd_heap            },
    {"help",                "help command",                                     cmd_help            },
    {"ins-log",             "insert log event",                                 cmd_inslog          },
    {"log-level",           "get/set mDot log level",                           cmd_logLevel        },
    {"ls",                  "list user files",                                  cmd_ls              },
    {"modify-mn",           "modify a manual control",                          cmd_modifyManual    },
    {"modify-sp",           "modify a setpoint control",                        cmd_modifySetpoint  },
    {"modmap",              "dump modbus register map",                         cmd_modmap          },
    {"mod-cmd",             "send command to modbus master",                    cmd_modbusCmd       },
    {"peep",                "push EEP",                                         cmd_peep            },
    {"regcmd",              "create command to execute on register",            cmd_regcmd          },
    {"reset",               "reset the controller",                             cmd_reset           },
    {"reset-stats",         "reset current mDot statistics",                    cmd_resetStats      },
    {"rm",                  "remove a user file",                               cmd_rm              },
    {"rssi-stats",          "get current rssi stats",                           cmd_rssiStats       },
    {"set-time",            "set current time",                                 cmd_settime         },
    {"show-ca",             "show composite algorithms",                        cmd_ShowAlgorithms  },
    {"show-controls",       "display active controls",                          cmd_ShowControls    },
    {"show-outputs",        "dump outputs",                                     cmd_outputs         },
    {"snr-stats",           "get current SNR stats",                            cmd_snrStats        },
    {"sout",                "set output",                                       cmd_sout            },
    {"simin",               "simulate input",                                   cmd_simin           },
    {"show-simin",          "show simulated inputs",                            cmd_showSimin       },
    {"stack",               "get thread stack usage stats",                     cmd_stack           },
    {"stats",               "get current mDot statistics",                      cmd_stats           },
    {"time",                "get current time",                                 cmd_time            },
    {"simall",              "simulate multiple inputs",                         cmd_simall          },
    {"trim-temp",           "adjust temperature from ICE Tube",                 cmd_trimTemp        },
    {NULL, NULL, NULL}
};

int func_read(char *buf, int cnt);
int func_write(const char *buf, int cnt);
int func_cb_ntshell(const char *text);
void func_cb_ntopt(int argc, char **argv);

/**
 * Serial read function.
 */
int func_read(char *buf, int cnt)
{
    for (int i = 0; i < cnt; i++) {
        buf[i] = serial.getc();
    }
    return 0;
}

/**
 * Serial write function.
 */
int func_write(const char *buf, int cnt)
{
    for (int i = 0; i < cnt; i++) {
        serial.putc(buf[i]);
    }
    return 0;
}

/**
 * Callback function for ntshell module.
 */
int func_cb_ntshell(const char *text)
{
    return ntopt_parse(text, func_cb_ntopt);
}

/**
 * Callback function for ntopt module.
 */
void func_cb_ntopt(int argc, char **argv)
{
    if (argc == 0) {
        return;
    }
    int execnt = 0;
    const command_table_t *p = &cmdlist[0];
    while (p->command != NULL) {
        if (strcmp(argv[0], p->command) == 0) {
            p->func(argc, argv);
            execnt++;
        }
        p++;
    }
    if (execnt == 0) {
        printf("Command not found.\r\n");
    }
    wait_ms(250);
}

/************************* callback functions *******************************/

void cmd_cat(int argc, char **argv)
{
    if ( argc != 2 ) {
        printf("\rusage: cat <filename>\n");
        return;
    }
    mDot::mdot_file file = GLOBAL_mdot->openUserFile(argv[1], mDot::FM_RDONLY);
    if ( file.fd < 0 ) {
        printf("\rFailed to open %s\n", argv[1]);
        return;
    }
    char *data_buf = (char*) malloc(file.size);
    bool rc = GLOBAL_mdot->readUserFile(file, data_buf, file.size);
    if ( rc != true ) {
        printf("\rFailed to read %s\n", argv[1]);
        goto cleanup;
    }
    printf("%s\n", data_buf);

cleanup:
    free(data_buf);
    GLOBAL_mdot->closeUserFile(file);
}
/*****************************************************************************
 * Function:        cmd_help
 * Description:     displays the list of commands available
 *
 * @param           argc (not used)
 * @param           argv (not used)
 * @return          none
 *****************************************************************************/
void cmd_help(int argc, char **argv)
{
    UNUSED(argc);
    UNUSED(argv);

    const command_table_t *tblPtr = cmdlist;

    while (tblPtr->command) {
        printf("\r%-32s:\t%s\n", tblPtr->command, tblPtr->description);
        tblPtr++;
    }
    printf("\r\n");
}


/*****************************************************************************
 * Function:        cmd_logLevel
 * Description:     get or set the current log-level
 *
 * @param           argc (not used)
 * @param           argv (not used)
 * @return          none
 *****************************************************************************/
void cmd_logLevel(int argc, char **argv)
{
    uint8_t logLevel = 0;

    const char *mapper[] = { "NONE",
                             "FATAL",
                             "ERROR",
                             "WARNING",
                             "INFO",
                             "DEBUG",
                             "TRACE"
                           };

    if ( argc == 1 ) {
        printf("\r  current log-level set to %s\r\n",
               mapper[GLOBAL_mdot->getLogLevel()]);
        goto usage;
    }

    if ( argc != 2 ) {
usage:
        printf("\rusage: log-level [0-6]\n");
        printf("\r   0 = NONE\n");
        printf("\r   1 = FATAL\n");
        printf("\r   2 = ERROR\n");
        printf("\r   3 = WARNING\n");
        printf("\r   4 = INFO\n");
        printf("\r   5 = DEBUG\n");
        printf("\r   6 = TRACE\r\n");
        return;
    }

    logLevel = atoi(argv[1]);
    if ( logLevel > 6 )
        goto usage;

    // reassign the log level
    printf("...setting log-level to %s\r\n", mapper[logLevel]);
    GLOBAL_mdot->setLogLevel(logLevel);
    printf("\r\n");
}

/*****************************************************************************
 * Function:        cmd_ls
 * Description:     list the user files on flash
 *
 * @param           argc (not used)
 * @param           argv (not used)
 * @return          none
 *****************************************************************************/
void cmd_ls(int argc, char **argv)
{
    UNUSED(argc);
    UNUSED(argv);

    vector<mDot::mdot_file> userFiles;
    userFiles = GLOBAL_mdot->listUserFiles();
    vector<mDot::mdot_file>::iterator pos;

    for ( pos = userFiles.begin(); pos != userFiles.end(); ++pos ) {
        printf("\r  %-33s %d\n", pos->name, pos->size);
    }
    printf("\r\n");
}

/*****************************************************************************
 * Function:        cmd_ShowControls
 * Description:     show active controls
 *
 * @param           argc (not used)
 * @param           argv (not used)
 * @return          none
 *****************************************************************************/
void cmd_ShowControls(int argc, char **argv)
{
    UNUSED(argc);
    UNUSED(argv);
    ConfigurationHandler_showControls();
    printf("\r\n");

}

void cmd_ShowAlgorithms(int argc, char **argv)
{
    UNUSED(argc);
    UNUSED(argv);
    ConfigurationHandler_showAlgorithms();
    printf("\r\n");
}

/*****************************************************************************
 * Function:        cmd_reset
 * Description:     reset the cpu
 *
 * @param           argc (not used)
 * @param           argv (not used)
 * @return          none
 *****************************************************************************/
void cmd_reset(int argc, char **argv)
{
    UNUSED(argc);
    UNUSED(argv);
    GLOBAL_mdot->resetCpu();
}

/*****************************************************************************
 * Function:        cmd_rm
 * Description:     removes a user file from flash
 *
 * @param           argc-> number of args
 * @param           argv-> filename
 * @return          none
 *****************************************************************************/
void cmd_rm(int argc, char **argv)
{
    UNUSED(argc);
    UNUSED(argv);

    if ( argc != 2 ) {
        printf("\rusage: rm <filename>\n");
        return;
    }
    if ( strcmp(argv[1], "*") == 0 ) {
        vector<mDot::mdot_file> userFiles;
        userFiles = GLOBAL_mdot->listUserFiles();
        vector<mDot::mdot_file>::iterator pos;
        for ( pos = userFiles.begin(); pos != userFiles.end(); ++pos ) {
            GLOBAL_mdot->deleteUserFile(pos->name);
        }
    } else {
        GLOBAL_mdot->deleteUserFile(argv[1]);
    }
}

/*****************************************************************************
 * Function:        cmd_create
 * Description:     create a control
 *
 * @param           argc-> number of args
 * @param           argv-> control name, control type
 * @return          none
 *****************************************************************************/
void cmd_create(int argc, char **argv)
{
    if ( argc != 3 ) {
        printf("\r\nusage: create [controlName] [controlType]\n");
        printf("\rcontrolType-> 0=timer, 1=PID, 2=setpoint, 3=composite, 4=manual\r\n");
        return;
    }
    // send a message to the configuration handler to create the control
    Message_t *msg  = MailBox.alloc();
    memset(msg, 0, sizeof(Message_t));
    msg->action  = ACTION_CREATE;
    msg->control = (Control_t) atoi(argv[2]);
    strncpy(msg->controlFile, argv[1], sizeof(msg->controlFile)-1);

    printf("%s: Sending a create request for control %s type = %u\r\n",
           __func__, msg->controlFile, msg->control);

    MailBox.put(msg);
    printf("\r\n");
    return;
}

/*****************************************************************************
 * Function:        cmd_destroy
 * Description:     reset the cpu
 *
 * @param           argc-> number of arguments
 * @param           argv-> control name, control type
 * @return          none
 *****************************************************************************/
void cmd_destroy(int argc, char **argv)
{
    if ( argc != 3 ) {
        printf("\r\nusage: destroy [controlName] [controlType]\n");
        printf("\rcontrolType-> 0=timer, 1=PID, 2=setpoint, 3=composite, 4=manual\r\n");
        return;
    }

    // send a message to the configuration handler to destroy the control
    Message_t *msg  = MailBox.alloc();
    memset(msg, 0, sizeof(Message_t));
    msg->action  = ACTION_DESTROY;
    msg->control = (Control_t) atoi(argv[2]);
    strncpy(msg->controlFile, argv[1], sizeof(msg->controlFile)-1);

    printf("%s: Sending a destroy request for control %s\r\n",
           __func__, msg->controlFile);

    MailBox.put(msg);
    printf("\r\n");
    return;
}

void cmd_creatCAlg(int argc, char **argv)
{

}

/*****************************************************************************
 * Function:        cmd_createSetpoint
 * Description:     create control file
 *
 * @param           argc-> number of args
 * @param           argv-> filename
 * @return          none
 *****************************************************************************/
void cmd_createSetpoint(int argc, char **argv)
{
    if ( argc != 8 ) {
        printf("\rusage:   create-sp <filename> <id> <input> <output> <sp> <dir> <tol>\n");
        printf("\rexample: create-sp control_sp_1.json bd i_cond o_rly1 2000 1 15\n");
        return;
    }

    if ( strncmp(argv[1], CONTROL_SP_STR, strlen(CONTROL_SP_STR)) != 0 ) {
        printf("\rFilename must be prefixed with control_sp_*\n");
        return;
    }

    char data_buf[MAX_FILE_SIZE];
    snprintf(data_buf, sizeof(data_buf),
             "{ "
             "\"id\":           \"%s\", "
             "\"priority\":     \"800\","
             "\"input\":        \"%s\", "
             "\"output\":       \"%s\", "
             "\"setpoint\":     \"%s\","
             "\"prodfact\":     \"100\","
             "\"actingDir\":    \"%s\", "
             "\"halert\":       \"115\","
             "\"lalert\":       \"85\", "
             "\"hfs\":          \"130\","
             "\"lfs\":          \"70\", "
             "\"tol\":          \"%s\"  }", argv[2], argv[3], argv[4], argv[5], argv[6], argv[7]);

    bool status = GLOBAL_mdot->saveUserFile(argv[1], (void *)data_buf, MAX_FILE_SIZE);
    if( status != true ) {
        logInfo("(%d)save file failed, status=%d", __LINE__, status);
        return;
    }

    // send a message to the configuration handler to create the control
    Message_t *msg  = MailBox.alloc();
    memset(msg, 0, sizeof(Message_t));
    msg->action  = ACTION_CREATE;
    msg->control = CONTROL_SETPOINT;
    strncpy(msg->controlFile, argv[1], sizeof(msg->controlFile)-1);

    printf("%s: Sending a create request for control %s type = %u\r\n",
           __func__, msg->controlFile, msg->control);

    MailBox.put(msg);
    printf("\r\n");
    return;
}

/*****************************************************************************
 * Function:        cmd_createTimer
 * Description:     create control file
 *
 * @param           argc-> number of args
 * @param           argv-> filename
 * @return          none
 *****************************************************************************/
void cmd_createTimer(int argc, char **argv)
{
    if ( argc != 6 ) {
        printf("\rusage: create-timer <filename> <id> <output> <priority> <duration>\n");
        printf("\rexample: create-timer control_tm_1.json timer-1 o_rly1 750 60\n");
        printf("\r     <startTime> is epoch time\n");
        printf("\r     <duration>  is in seconds\r\n");
        return;
    }

    if ( strncmp(argv[1], CONTROL_TM_STR, strlen(CONTROL_TM_STR)) != 0 ) {
        printf("\rFilename must be prefixed with control_tm_*\n");
        return;
    }

    char time_buf[32];
    sprintf(time_buf, "%lu", time(NULL)+5);

    char data_buf[MAX_FILE_SIZE];
    snprintf(data_buf, sizeof(data_buf),
             "{ "
             "\"id\":           \"%s\", "
             "\"output\":       \"%s\", "
             "\"priority\":     \"%s\", "
             "\"starttime\":    \"%s\", "
             "\"duration\":     \"%s\"  ", argv[2], argv[3], argv[4], time_buf, argv[5]
            );


    bool status = GLOBAL_mdot->saveUserFile(argv[1], (void *)data_buf, MAX_FILE_SIZE);
    if( status != true ) {
        logInfo("(%d)save file failed, status=%d", __LINE__, status);
        return;
    }

    // send a message to the configuration handler to create the control
    Message_t *msg  = MailBox.alloc();
    memset(msg, 0, sizeof(Message_t));
    msg->action  = ACTION_CREATE;
    msg->control = CONTROL_TIMER;
    strncpy(msg->controlFile, argv[1], sizeof(msg->controlFile)-1);

    printf("%s: Sending a create request for control %s type = %u\r\n",
           __func__, msg->controlFile, msg->control);

    MailBox.put(msg);
    printf("\r\n");
    return;

}

void cmd_cmt(int argc, char **argv)
{
    char time_buf[32];
    unsigned int counter = 0;
    char filename[32];
    Message_t *msg;

    for ( counter = 0; counter < 5; counter++ ) {
        // stuff the file
        sprintf(time_buf, "%lu", time(NULL)+5 + (40 * counter));
        char data_buf[MAX_FILE_SIZE];
        snprintf(data_buf, sizeof(data_buf),
                 "{ "
                 "\"id\":           \"timer-%d\", "
                 "\"output\":       \"o_rly1\", "
                 "\"priority\":     \"750\", "
                 "\"starttime\":    \"%s\", "
                 "\"duration\":     \"30\"  }", counter, time_buf);

        sprintf(filename, "control_tm_%d_rly1.json", counter);

        bool status = GLOBAL_mdot->saveUserFile(filename, (void *)data_buf, MAX_FILE_SIZE);
        msg = MailBox.alloc();
        memset(msg, 0, sizeof(Message_t));
        msg->action  = ACTION_CREATE;
        msg->control = CONTROL_TIMER;
        strncpy(msg->controlFile, filename, sizeof(msg->controlFile)-1);

        printf("%s: Sending a create request for control %s type = %u\r\n",
               __func__, msg->controlFile, msg->control);

        MailBox.put(msg);
        Thread::wait(1000);
    }

    for ( counter = 0; counter < 5; counter++ ) {
        // stuff the file
        sprintf(time_buf, "%lu", time(NULL)+5 + (40 * counter));
        char data_buf[MAX_FILE_SIZE];
        snprintf(data_buf, sizeof(data_buf),
                 "{ "
                 "\"id\":           \"timer-%d\", "
                 "\"output\":       \"o_rly2\", "
                 "\"priority\":     \"750\", "
                 "\"starttime\":    \"%s\", "
                 "\"duration\":     \"30\"  ", counter, time_buf);

        sprintf(filename, "control_tm_%d_rly2.json", counter);

        bool status = GLOBAL_mdot->saveUserFile(filename, (void *)data_buf, MAX_FILE_SIZE);
        msg = MailBox.alloc();
        memset(msg, 0, sizeof(Message_t));
        msg->action  = ACTION_CREATE;
        msg->control = CONTROL_TIMER;
        strncpy(msg->controlFile, filename, sizeof(msg->controlFile)-1);

        printf("%s: Sending a create request for control %s type = %u\r\n",
               __func__, msg->controlFile, msg->control);

        MailBox.put(msg);
        Thread::wait(1000);
    }
}

void cmd_create_fs(int argc, char **argv)
{
    char buf[MAX_FILE_SIZE];
    bool status;

    std::string filename = "control_fs_rly01.json";
    snprintf(buf, sizeof(buf),
             "{"
             "\"id\":    \"BLOWDOWN-FS\", "
             "\"input\":   \"i_bdcond01\", "
             "\"output\":  \"o_rly01\", "
             "\"priority\":\"700\", "
             "\"lfsValue\":\"700\", "
             "\"lfsDutyCycle\":\"0\", "
             "\"lfsInterval\":\"1\", "
             "\"hfsValue\":\"2700\", "
             "\"hfsDutyCycle\":\"50\", "
             "\"hfsInterval\":\"1\" "
             "}");

    status = GLOBAL_mdot->saveUserFile(filename.c_str(), (void *)buf, MAX_FILE_SIZE);
    if ( status != true ) {
        printf("\rFailed to create %s\n", filename.c_str());
        return;
    } else {
        printf("\r   created %s\n", filename.c_str());
    }

    Message_t *msg = MailBox.alloc();
    memset(msg, 0, sizeof(Message_t));
    msg->action  = ACTION_CREATE;
    msg->control = CONTROL_FAILSAFE;
    strncpy(msg->controlFile, filename.c_str(), sizeof(msg->controlFile)-1);

    printf("%s: Sending a create request for control %s type = %u\r\n",
           __func__, msg->controlFile, msg->control);

    MailBox.put(msg);
    Thread::wait(1000);

    filename = "control_fs_rly02.json";
    snprintf(buf, sizeof(buf),
             " {"
             "\"id\":    \"TRASAR-FS\", "
             "\"input\":   \"i_tra01\", "
             "\"output\":  \"o_rly02\", "
             "\"priority\":\"700\", "
             "\"lfsValue\":\"70\", "
             "\"lfsDutyCycle\":\"20\", "
             "\"lfsInterval\":\"15\", "
             "\"hfsValue\":\"200\", "
             "\"hfsDutyCycle\":\"0\", "
             "\"hfsInterval\":\"15\" "
             "}");

    status = GLOBAL_mdot->saveUserFile(filename.c_str(), (void *)buf, MAX_FILE_SIZE);
    if ( status != true ) {
        printf("\rFailed to create %s\n", filename.c_str());
        return;
    } else {
        printf("\r   created %s\n", filename.c_str());
    }
    
    msg = MailBox.alloc();
    memset(msg, 0, sizeof(Message_t));
    msg->action  = ACTION_CREATE;
    msg->control = CONTROL_FAILSAFE;
    strncpy(msg->controlFile, filename.c_str(), sizeof(msg->controlFile)-1);

    printf("%s: Sending a create request for control %s type = %u\r\n",
           __func__, msg->controlFile, msg->control);

    MailBox.put(msg);
}

//
//
void cmd_createCAlg(int argc, char **argv)
{
    if ( argc != 1 ) {
        printf("\rusage: create-ca control_ca_equal.json");
        return;
    }

    char buf[MAX_FILE_SIZE];
    std::string filename = "control_ca_eq1.json";
    snprintf(buf, sizeof(buf),
             " {"
             "\"id\":    \"EQUAL_TO_1\", "
             "\"opr\":   \"1\", "
             "\"op\":    \"==\", "
             "\"true\":   \"responseA\", "
             "\"false\":  \"nothing\" "
             "}");

    bool rc = GLOBAL_mdot->saveUserFile(filename.c_str(), (void *)buf, MAX_FILE_SIZE);
    if ( rc != true ) {
        printf("\rFailed to save user file %s\n", filename.c_str());
        return;
    } else {
        printf("\r...generated %s\n", filename.c_str());
    }

// send a message to the configuration handler to create the control
    Message_t *msg  = MailBox.alloc();
    memset(msg, 0, sizeof(Message_t));
    msg->action  = ACTION_CREATE;
    msg->control = CONTROL_ALGORITHM;
    strncpy(msg->controlFile, filename.c_str(), sizeof(msg->controlFile)-1);

    MailBox.put(msg);

    Thread::wait(2000);

// == 129
    filename = "control_ca_eq129.json";
    snprintf(buf, sizeof(buf),
             "{"
             "\"id\":    \"EQUAL_TO_1\", "
             "\"opr\":   \"1\", "
             "\"op\":    \"==\", "
             "\"true\":   \"responseA\", "
             "\"false\":  \"nothing\" "
             "}");

    rc = GLOBAL_mdot->saveUserFile(filename.c_str(), (void *)buf, MAX_FILE_SIZE);
    if ( rc != true ) {
        printf("\rFailed to save user file %s\n", filename.c_str());
        return;
    } else {
        printf("\r...generated %s\n", filename.c_str());
    }

// send a message to the configuration handler to create the control
    msg  = MailBox.alloc();
    memset(msg, 0, sizeof(Message_t));
    msg->action  = ACTION_CREATE;
    msg->control = CONTROL_ALGORITHM;
    strncpy(msg->controlFile, "control_ca_greq1.json", sizeof(msg->controlFile)-1);

    MailBox.put(msg);
}

void cmd_createComposite(int argc, char **argv)
{
    if ( argc != 1 ) {
        printf("\rusage:    create-comp");
        return;
    }

    char data_buf[MAX_FILE_SIZE];
    bool status;
    Message_t *msg = NULL;

#if 1
    // FLOW
    snprintf(data_buf, sizeof(data_buf),
             "{ "
             "\"id\":  \"FLOW_OVERRIDE\",  "
             "\"tag\": \"i_flowsw01\",  "
             "\"ca\":  \"EQUAL_TO_1\",   "
             "\"priority\": \"400\", "
             "\"entries\":  \"2\", "
             "\"outputs\": [ "
             "{"
             "\"tag\": \"o_rly01\", "
             "\"responseA\": \"fixed off\" "
             "},"
             "{"
             "\"tag\": \"o_rly02\", "
             "\"responseA\": \"fixed off\" "
             "}"
             "]"
             "}"
            );
    status = GLOBAL_mdot->saveUserFile("control_comp_flow.json", (void *)data_buf, MAX_FILE_SIZE);
    if ( status != true ) {
        printf("\rFailed to save user file control_comp_flow.json\n");
        return;
    }

    // send a message to the configuration handler to create the control
    msg  = MailBox.alloc();
    memset(msg, 0, sizeof(Message_t));
    msg->action  = ACTION_CREATE;
    msg->control = CONTROL_COMPOSITE;
    strncpy(msg->controlFile, "control_comp_flow.json", sizeof(msg->controlFile)-1);

    printf("\rDEBUG: sending a request for composite control %s\n", msg->controlFile);

    MailBox.put(msg);

    Thread::wait(2000);

#endif

    // LOCKOUT
    snprintf(data_buf, sizeof(data_buf),
             "{ "
             "\"id\":  \"LOCKOUT_BLOWDOWN\",  "
             "\"tag\": \"o_rly01\",  "
             "\"ca\":  \"EQUAL_TO_129\",   "
             "\"priority\": \"400\", "
             "\"entries\":  \"1\", "
             "\"outputs\": [ "
             "{"
             "\"tag\": \"o_rly02\", "
             "\"responseA\": \"fixed off\" "
             "}"
             "]"
             "}"
            );
    status = GLOBAL_mdot->saveUserFile("control_comp_lockout.json", (void *)data_buf, MAX_FILE_SIZE);
    if ( status != true ) {
        printf("\rFailed to save user file control_comp_lockout.json\n");
        return;
    }

    // send a message to the configuration handler to create the control
    msg  = MailBox.alloc();
    memset(msg, 0, sizeof(Message_t));
    msg->action  = ACTION_CREATE;
    msg->control = CONTROL_COMPOSITE;
    strncpy(msg->controlFile, "control_comp_lockout.json", sizeof(msg->controlFile)-1);

    printf("\rDEBUG: sending a request for composite control %s\n", msg->controlFile);

    MailBox.put(msg);
}

/*****************************************************************************
 * Function:        cmd_createManual
 * Description:     create a manual control
 *
 * @param           argc-> number of args
 * @param           argv-> filename
 * @return          none
 *****************************************************************************/
void cmd_createManual(int argc, char **argv)
{
    std::string relayState;
    if ( argc != 5 ) {
        printf("\rusage:   create-manual <filename> <id> <output> <state>\n");
        printf("\rexample: create-manual control_mn_1.json man-1 o_rly1 1\r\n");
        return;
    }

    string state(argv[4]);

    if ( state == "on" ) {
        relayState = "1";
    } else if ( state == "off" ) {
        relayState = "0";
    } else {
        printf("\r<state> must be on or off\r\n");
        return;
    }

    char data_buf[MAX_FILE_SIZE];
    snprintf(data_buf, sizeof(data_buf),
             "{ "
             "\"id\":           \"%s\",  "
             "\"output\":       \"%s\",  "
             "\"type\":         \"1\",   "
             "\"priority\":     \"100\", "
             "\"duration\":     \"0\",   "
             "\"setpoint\":     \"0\",   "
             "\"state\":        \"%s\",  "
             "\"percent\":      \"100\"  }", argv[2], argv[3], relayState.c_str()
            );

    bool status = GLOBAL_mdot->saveUserFile(argv[1], (void *)data_buf, MAX_FILE_SIZE);
    if( status != true ) {
        logInfo("(%d)save file failed, status=%d", __LINE__, status);
        return;
    }

    // send a message to the configuration handler to create the control
    Message_t *msg  = MailBox.alloc();
    memset(msg, 0, sizeof(Message_t));
    msg->action  = ACTION_CREATE;
    msg->control = CONTROL_MANUAL;
    strncpy(msg->controlFile, argv[1], sizeof(msg->controlFile)-1);

    printf("%s: Sending a create request for control %s type = %u\r\n",
           __func__, msg->controlFile, msg->control);

    MailBox.put(msg);
    printf("\r\n");
    return;
}

/*****************************************************************************
 * Function:        cmd_cif
 * Description:     create input file
 *
 * @param           argc-> number of args
 * @param           argv-> filename
 * @return          none
 *****************************************************************************/
void cmd_cif(int argc, char **argv)
{
    if ( argc != 6 ) {
        printf("\rusage:   cif <fname> <input> <name> <node> <reg>\n");
        printf("\rexample: cif input_i_tra01.json i_tra01 Trasar 5 2\n");
        return;
    }

    char data_buf[MAX_FILE_SIZE];
    snprintf(data_buf, sizeof(data_buf),
             "{ "
             "\"id\":       \"%s\", "
             "\"name\":     \"%s\", "
             "\"units\":    \"PPM\", "
             "\"min\":      \"0\", "
             "\"max\":      \"300\", "
             "\"node\":     \"%s\", "
             "\"reg\":      \"%s\", "
             "\"rtype\":    \"1\", "
             "\"type\":     \"0\", "
             "\"size\":     \"2\", "
             "\"order\":    \"2\", "
             "\"rfreq\":    \"5\", "
             "\"fmt\":      \"%%.2f\" } ", argv[2], argv[3], argv[4], argv[5]);

    bool status = GLOBAL_mdot->saveUserFile(argv[1], (void *)data_buf, MAX_FILE_SIZE);
    if( status != true ) {
        logInfo("(%d)save file failed, status=%d", __LINE__, status);
    }

    logInfo("Sending Mail To ModbusMasterMailBox, filename=%s", argv[1]);
    ModbusMasterReq_t *mail = ModbusMasterMailBox.alloc();
    mail->action = ACTION_READ_FILE;
    strncpy( mail->controlFile, argv[1], (sizeof(mail->controlFile)-1));
    ModbusMasterMailBox.put(mail);
}

/*****************************************************************************
 * Function:        cmd_cmf
 * Description:     create manual control file
 *
 * @param           argc-> number of args
 * @param           argv-> filename
 * @return          none
 *****************************************************************************/
void cmd_cmf(int argc, char **argv)
{
    if ( argc != 2 ) {
        printf("\rusage: cmf <filename> <relay\r\n");
        printf("\rexmaple: cmd control_mn_1.json o_rly1");
        return;
    }

    char data_buf[MAX_FILE_SIZE];
    snprintf(data_buf, sizeof(data_buf),
             "{ "
             "\"id\":           \"%s\", "
             "\"type\":         \"1\", "
             "\"priority\":     \"100\", "
             "\"duration\":     \"30\", "       // seconds
             "\"setpoint\":     \"2000.0\", "
             "\"state\":        \"1\", "
             "\"percent\":      \"100\", }", argv[2]
            );

    bool status = GLOBAL_mdot->saveUserFile(argv[1], (void *)data_buf, MAX_FILE_SIZE);
    if( status != true ) {
        logInfo("(%d)save file failed, status=%d", __LINE__, status);
    }
}


/*****************************************************************************
 * Function:        cmd_cof
 * Description:     create output file
 *
 * @param           argc-> number of args
 * @param           argv-> filename
 * @return          none
 *****************************************************************************/
void cmd_cof(int argc, char **argv)
{
    if ( argc != 5 ) {
        printf("\rusage: cof <filename> <output> <name> <reg>\r\n");
        printf("\rexample: cof output_rly1.json o_rly1 Relay1 1\r\n");
        return;
    }

    char data_buf[MAX_FILE_SIZE];

    snprintf(data_buf, sizeof(data_buf),
             "{ "
             "\"id\":       \"%s\", "
             "\"name\":     \"%s\", "
             "\"units\":    \"\", "
             "\"min\":      \"0\", "
             "\"max\":      \"300\", "
             "\"node\":     \"0\", "
             "\"reg\":      \"%s\", "
             "\"rtype\":    \"1\", "
             "\"type\":     \"16\", "
             "\"size\":     \"2\", "
             "\"order\":    \"2\", "
             "\"fmt\":      \"%%.2f\", "
             "\"rfreq\":    \"5\", "
             "\"toperiod\": \"0\", "
             "\"scalelo\":  \"0\", "
             "\"scalehi\":  \"100\" }", argv[2], argv[3], argv[4]);

    bool status = GLOBAL_mdot->saveUserFile(argv[1], (void *)data_buf, MAX_FILE_SIZE);
    if( status != true ) {
        logInfo("(%d)save file failed, status=%d", __LINE__, status);
    }

    // send a message to the modbus master
    logInfo("Sending mail to ModbusMasterMailBox, filename=%s", argv[1]);
    ModbusMasterReq_t *modbus_mail = ModbusMasterMailBox.alloc();
    modbus_mail->action = ACTION_READ_FILE;
    strncpy( modbus_mail->controlFile, argv[1], (sizeof(modbus_mail->controlFile)-1));
    ModbusMasterMailBox.put(modbus_mail);

    // send a message to the output master
    logInfo("Sending mail to OutputMaster, filename = %s", argv[1]);
    OutputControlMsg_t *output_mail = OutputMasterMailBox.alloc();
    output_mail->action = ACTION_NEW;
    strncpy(output_mail->controlFile, argv[1], sizeof(output_mail->controlFile)-1);
    OutputMasterMailBox.put(output_mail);
}

/*****************************************************************************
 * Function:        cmd_heap
 * Description:     display heap statistics
 *
 * @param           argc (not used)
 * @param           argv (not used)
 * @return          none
 *****************************************************************************/
void cmd_heap(int argc, char **argv)
{
    UNUSED(argc), UNUSED(argv);
    __heapstats((__heapprt)fprintf,stderr);   // print initial free heap size
}


/*****************************************************************************
 * Function:        cmd_stats
 * Description:     display mDot stats
 *
 * @param           argc (not used)
 * @param           argv (not used)
 * @return          none
 *****************************************************************************/
void cmd_stats(int argc, char **argv)
{
    UNUSED(argc);
    UNUSED(argv);

    mDot::mdot_stats stats = GLOBAL_mdot->getStats();

    printf("\r                        Up: %u\n", stats.Up);
    printf("\r                      Down: %u\n", stats.Down);
    printf("\r                     Joins: %u\n", stats.Joins);
    printf("\r                 JoinFails: %u\n", stats.JoinFails);
    printf("\r                MissedAcks: %u\n", stats.MissedAcks);
    printf("\r                 CRCErrors: %u\n", stats.CRCErrors);
    printf("\r\n");
    printf("\r                 Freq band: %u\n", GLOBAL_mdot->getFrequencyBand());
    printf("\r              Freq subband: %u\n", GLOBAL_mdot->getFrequencySubBand());
    printf("\r         Session data rate: %u\n", GLOBAL_mdot->getSessionDataRate());
    printf("\r       Public Network Mode: %s\n", GLOBAL_mdot->getPublicNetwork() ? "yes" : "no");
    printf("\r   Application device port: %u\n", GLOBAL_mdot->getAppPort());
    printf("\r                     Class: %s\n", GLOBAL_mdot->getClass().c_str());
    printf("\r         Max packet length: %u\n", GLOBAL_mdot->getMaxPacketLength());
    std::vector<uint8_t> na = GLOBAL_mdot->getNetworkAddress();
    std::string str(na.begin(), na.end());
    printf("\r           Network address: %s\n", str.c_str());
    printf("\r              Network name: %s\n", GLOBAL_mdot->getNetworkName().c_str());
    std::vector<uint8_t> nid = GLOBAL_mdot->getNetworkId();
    std::string networkIdStr(nid.begin(), nid.end());
    printf("\r                Network ID: %s\n", networkIdStr.c_str());
    printf("\r          Join byter order: %s\n", GLOBAL_mdot->getJoinByteOrder() == 0 ? "LSB" : "MSB");
    printf("\r              Join retries: %u\n", GLOBAL_mdot->getJoinRetries());
    printf("\r                 Join mode: %u\n", GLOBAL_mdot->getJoinMode());
    printf("\r       Network join status: %s\n", GLOBAL_mdot->getNetworkJoinStatus() ? "yes" : "no");
    printf("\r           Link fail count: %u\n", GLOBAL_mdot->getLinkFailCount());
    printf("\r   Packets Tx'd to gateway: %u\n", GLOBAL_mdot->getUpLinkCounter());
    printf("\r Packets Rx'd from gateway: %u\n", GLOBAL_mdot->getDownLinkCounter());
    printf("\r            AES encryption: %s\n", GLOBAL_mdot->getAesEncryption() ? "enabled" : "disabled");
    printf("\r              Tx data rate: %u\n", GLOBAL_mdot->getTxDataRate());
    printf("\r          Datarate Details: %s\n", GLOBAL_mdot->getDateRateDetails(GLOBAL_mdot->getTxDataRate()).c_str());
    printf("\r                  Tx power: %u\n", GLOBAL_mdot->getTxPower());
    printf("\r              Antenna gain: %u\n", GLOBAL_mdot->getAntennaGain());
    printf("\r             Min frequency: %u\n", GLOBAL_mdot->getMinFrequency());
    printf("\r             Max frequency: %u\n", GLOBAL_mdot->getMaxFrequency());
    printf("\r               CRC enabled: %s\n", GLOBAL_mdot->getCrc() ? "yes" : "no");
    printf("\r               ACK enabled: %s\n", GLOBAL_mdot->getAck() ? "yes" : "no");

    printf("\r\n");
}

/*****************************************************************************
 * Function:        cmd_resetStats
 * Description:     resets the mDot stats
 *
 * @param           argc (not used)
 * @param           argv (not used)
 * @return          none
 *****************************************************************************/
void cmd_resetStats(int argc, char **argv)
{
    UNUSED(argc);
    UNUSED(argv);

    GLOBAL_mdot->resetStats();
}

/*****************************************************************************
 * Function:        cmd_rssiStats
 * Description:     displays mDot RSSI statistics
 *
 * @param           argc (not used)
 * @param           argv (not used)
 * @return          none
 *****************************************************************************/
void cmd_rssiStats(int argc, char **argv)
{
    UNUSED(argc);
    UNUSED(argv);

    mDot::rssi_stats s = GLOBAL_mdot->getRssiStats();

    printf("\r     Last: %d dB\n", s.last);
    printf("\r      Min: %d dB\n", s.min);
    printf("\r      Max: %d dB\n", s.max);
    printf("\r      Avg: %d dB\n", s.avg);

    printf("\r\n");
}

/*****************************************************************************
 * Function:        cmd_snrStats
 * Description:     displays signal-to-noise ratio stats
 *
 * @param           argc (not used)
 * @param           argv (not used)
 * @return          none
 *****************************************************************************/
void cmd_snrStats(int argc, char **argv)
{
    mDot::snr_stats s = GLOBAL_mdot->getSnrStats();

    printf("\r     Last: %d cB\n", s.last);
    printf("\r      Min: %d cB\n", s.min);
    printf("\r      Max: %d cB\n", s.max);
    printf("\r      Avg: %d cB\n", s.avg);

    printf("\r\n");
}

/*****************************************************************************
 * Function:        cmd_stack
 * Description:     display thread stack statisics
 *
 * @param           argc (not used)
 * @param           argv (not used)
 * @return          none
 *****************************************************************************/
void cmd_stack(int argc, char **argv)
{
    vector<pair<string, Thread*> > taskList;

    const char *mapper[] = { "Inactive",
                             "Ready",
                             "Running",
                             "WaitingDelay",
                             "WaitingInterval",
                             "WaitingOr",
                             "WaitingAnd",
                             "WaitingSempahore",
                             "WaitingMailbox",
                             "WaitingMutex"
                           };

    //simply add your task to the list...
    taskList.push_back(make_pair((string)"AnalyticsLogger",  GLOBAL_analyticsLogger_thread));
    taskList.push_back(make_pair((string)"BLEHandler",       GLOBAL_BLE_thread));
    taskList.push_back(make_pair((string)"CloudDataHandler", GLOBAL_CDH_thread));
    taskList.push_back(make_pair((string)"ConfigHandler",    GLOBAL_configHandler_thread));
    taskList.push_back(make_pair((string)"ControlTask",      GLOBAL_controlTask_thread));
    taskList.push_back(make_pair((string)"ModbusMaster",     GLOBAL_modbusMaster_thread));
    taskList.push_back(make_pair((string)"OutputTask",       GLOBAL_outputTask_thread));

    for ( vector<pair<string, Thread*> >::iterator pos = taskList.begin();
            pos != taskList.end(); ++ pos) {
        printf("\r %-32s size/free/used/max = %5u/%5u/%5u/%5u\tpri=%u state=%-20s\n",
               pos->first.c_str(),
               pos->second->stack_size(),
               pos->second->free_stack(),
               pos->second->used_stack(),
               pos->second->max_stack(),
               pos->second->get_priority(),
               mapper[pos->second->get_state()]);
    }

    printf("\r\n");
}

/*****************************************************************************
 * Function:        cmd_modmap
 * Description:     dump modbus register map
 *****************************************************************************/
void cmd_modmap(int argc, char **argv)
{
    UNUSED(argc);
    UNUSED(argv);
    ModbusValue value;

    std::map<std::string, ModbusRegister>::iterator iter;
    for (iter = ModbusRegisterMap.begin(); iter != ModbusRegisterMap.end(); ++iter) {
        ModbusMasterReadRegister( iter->first, &value );
        printf("tag=%s, name=%s, units=%s, node=%d, reg=%d, rtype=%d size=%d, order=%d, value=%2.2f, errflag=%d\r\n", iter->first.c_str(), iter->second.name.c_str(), iter->second.units.c_str(), iter->second.node, iter->second.reg, iter->second.rtype, iter->second.size, iter->second.order, value.value, value.errflag );
    }
}

/*****************************************************************************
 * Function:        cmd_modifyManual
 * Description:     modify a manual control
 *****************************************************************************/
void cmd_modifyManual(int argc, char **argv)
{
    std::string relayState;

    if ( argc != 3 ) {
        printf("\rusage:    modify-mn <controlFile> <state>\n");
        printf("\rexample:  modify-mn control_mn_rly1.json on\n");
        return;
    }

    string state(argv[2]);
    if ( state == "on" ) {
        relayState = "1";
    } else if ( state == "off" ) {
        relayState = "0";
    } else {
        printf("\r<state> must be ON or OFF\n");
        return;
    }

    char buf[MAX_FILE_SIZE];
    bool rc = GLOBAL_mdot->readUserFile(argv[1], (void*)buf, MAX_FILE_SIZE);
    if ( rc != true ) {
        printf("\rFailed to read %s\n", argv[1]);
        return;
    }
    
    cJSON * root = cJSON_Parse(buf);
    strncpy( cJSON_GetObjectItem(root,"state")->valuestring, relayState.c_str(), (strlen(relayState.c_str())-1) );
    std::string s = cJSON_Print(root);
    cJSON_Delete(root);
    
    printf("writing back: %s\r\n",s.c_str());

    rc = GLOBAL_mdot->saveUserFile(argv[1], (void*)s.c_str(), MAX_FILE_SIZE);
    if ( rc != true ) {
        printf("\rFailed to write %s\n", argv[1]);
        return;
    }

    // send a message to the configuration handler to create the control
    Message_t *msg  = MailBox.alloc();
    memset(msg, 0, sizeof(Message_t));
    msg->action  = ACTION_MODIFY;
    msg->control = CONTROL_MANUAL;
    strncpy(msg->controlFile, argv[1], sizeof(msg->controlFile)-1);

    printf("%s: Sending a create request for control %s type = %u\r\n",
           __func__, msg->controlFile, msg->control);

    MailBox.put(msg);
    printf("\r\n");
    return;
}

/*****************************************************************************
 * Function:        cmd_modifySetpoint
 * Description:     modify a setpoint control
 *****************************************************************************/
void cmd_modifySetpoint(int argc, char**argv)
{
    if ( argc != 4) {
        printf("\rusage:   modify-sp <controlFile> <setpoint> <tol>\n");
        printf("\rexample: modify-sp control_sp_1.json 2200 10\n");
        return;
    }

    // read the file
    char buf[MAX_FILE_SIZE];
    bool rc = GLOBAL_mdot->readUserFile(argv[1], (void*)buf, MAX_FILE_SIZE);
    if ( rc != true ) {
        printf("\rFailed to read %s\n", argv[1]);
        return;
    }

    cJSON * root = cJSON_Parse(buf);
    strncpy( cJSON_GetObjectItem(root,"setpoint")->valuestring, argv[2], (strlen(argv[2])-1) );
    strncpy( cJSON_GetObjectItem(root,"tol")->valuestring, argv[3], (strlen(argv[3])-1) );
    std::string s = cJSON_Print(root);
    cJSON_Delete(root);
    
    printf("writing back: %s\r\n",s.c_str());

    rc = GLOBAL_mdot->saveUserFile(argv[1], (void*)s.c_str(), MAX_FILE_SIZE);
    if ( rc != true ) {
        printf("\rFailed to write %s\n", argv[1]);
        return;
    }

    // send a message to the configuration handler to create the control
    Message_t *msg  = MailBox.alloc();
    memset(msg, 0, sizeof(Message_t));
    msg->action  = ACTION_MODIFY;
    msg->control = CONTROL_SETPOINT;
    strncpy(msg->controlFile, argv[1], sizeof(msg->controlFile)-1);

    printf("%s: Sending a MODIFY request for control %s type = %u\r\n",
           __func__, msg->controlFile, msg->control);

    MailBox.put(msg);
    printf("\r\n");
    return;

}

/*****************************************************************************
 * Function:        cmd_time
 * Description:     display real-time clock
 ****************************************************************************/
void cmd_time(int argc, char **argv)
{
    UNUSED(argc);
    UNUSED(argv);

    time_t rawtime;
    time(&rawtime);

    int iyr=0, imo=0, idy=0, ihr=0, imn=0, isc=0;
    rtc_get_time(&iyr, &imo, &idy, &ihr, &imn, &isc);
    printf("RTC time: %04d-%02d-%02d %02d:%02d:%02d\r\n", iyr, imo, idy, ihr, imn, isc);
    printf("\repoch timestamp: %lu\r\n", time(NULL));
}

/*****************************************************************************
 * Function:        cmd_outputs
 * Description:     display outputs
 ****************************************************************************/
void cmd_outputs(int argc, char **argv)
{
    UNUSED(argc);
    UNUSED(argv);

    DisplayOutputs();
}

/*****************************************************************************
 * Function:        cmd_sout
 * Description:     set output
 *
 * @param           argc-> number of args
 * @param           argv-> output
 * @return          none
 *****************************************************************************/
void cmd_sout(int argc, char **argv)
{
    float value = atof( argv[2] );

    if ( argc != 3 ) {
        printf("\rusage: sout <output> <value>\r\n");
        printf("\rexample: sout o_rly1 1\r\n");
        return;
    }
    ModbusMasterWriteRegister( argv[1], value );
}

/*****************************************************************************
 * Function:        cmd_settime
 * Description:     set real-time clock
 *
 * @param           argc-> number of args
 * @param           argv-> input
 * @return          none
 *****************************************************************************/
void cmd_settime(int argc, char **argv)
{

    if ( argc != 7 ) {
        printf("\rusage:   set-time <yyyy> <mm> <dd> <hh> <mm> <ss>\n");
        printf("\rexample: set-time 2016 12 25 12 0 0\r\n");
        return;
    }

    rtc_set_time(atoi(argv[1]),     // year
                 atoi(argv[2]),     // month
                 atoi(argv[3]),     // day
                 atoi(argv[4]),     // hr
                 atoi(argv[5]),     // min
                 atoi(argv[6]));    // sec
}

/*****************************************************************************
 * Function:        cmd_simin
 * Description:     simulat input
 *
 * @param           argc-> number of args
 * @param           argv-> input
 * @return          none
 *****************************************************************************/
void cmd_simin(int argc, char **argv)
{
    float value = atof( argv[2] );

    if ( argc < 3 ) {
        printf("\rusage: simin <input> <value> <low> <hi> <up_step> <down_step>\r\n");
        printf("\rexample: simin i_tra01 100 94 106 1 .25\r\n");
        printf("\rexample: simin i_bdcond01 2000 1990 2006 .25 1\r\n");
        return;
    }

    SimulateInputMap[argv[1]].start_value = atof(argv[2]);

    ModbusRegisterMap[argv[1]].simulated = true;

    if ( argc < 3 ) {
        SimulateInputMap[argv[1]].min = 0;
        SimulateInputMap[argv[1]].max = 0;
        SimulateInputMap[argv[1]].moving_up = false;
        return;
    }
    SimulateInputMap[argv[1]].min = atof(argv[3]);
    SimulateInputMap[argv[1]].max = atof(argv[4]);
    SimulateInputMap[argv[1]].up_step = atof(argv[5]);
    SimulateInputMap[argv[1]].down_step = atof(argv[6]);
    SimulateInputMap[argv[1]].moving_up = true;

}

/*****************************************************************************
 * Function:        cmd_simall
 * Description:     simulat multiple inputs
 *
 * @return          none
 *****************************************************************************/
void cmd_simall(int argc, char **argv)
{

    if ( argc > 1 ) {
        printf("\rsetting: simin i_tra01 100\r\n");
        printf("\rsetting: simin i_bdcond01 2000\r\n");

        ModbusRegisterMap["i_tra01"].simulated = true;
        SimulateInputMap["i_tra01"].start_value = 130;
        SimulateInputMap["i_tra01"].min = 0;
        SimulateInputMap["i_tra01"].max = 0;
        SimulateInputMap["i_tra01"].up_step = 0;
        SimulateInputMap["i_tra01"].down_step = 0;
        SimulateInputMap["i_tra01"].moving_up = true;

        ModbusRegisterMap["i_tag01"].simulated = true;
        SimulateInputMap["i_tag01"].start_value = 100;
        SimulateInputMap["i_tag01"].min = 0;
        SimulateInputMap["i_tag01"].max = 0;
        SimulateInputMap["i_tag01"].up_step = 0;
        SimulateInputMap["i_tag01"].down_step = 0;
        SimulateInputMap["i_tag01"].moving_up = true;

        ModbusRegisterMap["i_bdcond01"].simulated = true;
        SimulateInputMap["i_bdcond01"].start_value = 1800;
        SimulateInputMap["i_bdcond01"].min = 0;
        SimulateInputMap["i_bdcond01"].max = 0;
        SimulateInputMap["i_bdcond01"].up_step = 0;
        SimulateInputMap["i_bdcond01"].down_step = 0;
        SimulateInputMap["i_bdcond01"].moving_up = true;

        ModbusRegisterMap["i_cond_temp01"].simulated = true;
        SimulateInputMap["i_cond_temp01"].start_value = 25000.0;
        SimulateInputMap["i_cond_temp01"].min = 0;
        SimulateInputMap["i_cond_temp01"].max = 0;
        SimulateInputMap["i_cond_temp01"].up_step = 0;
        SimulateInputMap["i_cond_temp01"].down_step = 0;
        SimulateInputMap["i_cond_temp01"].moving_up = true;

        ModbusRegisterMap["i_ph01"].simulated = true;
        SimulateInputMap["i_ph01"].start_value = 8000;
        SimulateInputMap["i_ph01"].min = 0;
        SimulateInputMap["i_ph01"].max = 0;
        SimulateInputMap["i_ph01"].up_step = 0;
        SimulateInputMap["i_ph01"].down_step = 0;
        SimulateInputMap["i_ph01"].moving_up = true;

        ModbusRegisterMap["i_ph_temp01"].simulated = true;
        SimulateInputMap["i_ph_temp01"].start_value = 25000.0;
        SimulateInputMap["i_ph_temp01"].min = 0;
        SimulateInputMap["i_ph_temp01"].max = 0;
        SimulateInputMap["i_ph_temp01"].up_step = 0;
        SimulateInputMap["i_ph_temp01"].down_step = 0;
        SimulateInputMap["i_ph_temp01"].moving_up = true;

        ModbusRegisterMap["i_orp01"].simulated = true;
        SimulateInputMap["i_orp01"].start_value = 350;
        SimulateInputMap["i_orp01"].min = 0;
        SimulateInputMap["i_orp01"].max = 0;
        SimulateInputMap["i_orp01"].up_step = 0;
        SimulateInputMap["i_orp01"].down_step = 0;
        SimulateInputMap["i_orp01"].moving_up = true;

        return;
    }

    printf("\rsetting: simin i_tra01 130 120 140 1 .25\r\n");
    printf("\rsetting: simin i_tag01 100 90 110 1 .25\r\n");
    printf("\rsetting: simin i_bdcond01 1800 1725 1875 1 5\r\n");
    printf("\rsetting: simin i_ph01 8.0 7.25 8.75 .0. .1\r\n");
    printf("\rsetting: simin i_orp01 350 300 400 5 1\r\n");

    ModbusRegisterMap["i_tra01"].simulated = true;
    SimulateInputMap["i_tra01"].start_value = 130;
    SimulateInputMap["i_tra01"].min = 120;
    SimulateInputMap["i_tra01"].max = 140;
    SimulateInputMap["i_tra01"].up_step = 1;
    SimulateInputMap["i_tra01"].down_step = .25;

    ModbusRegisterMap["i_tag01"].simulated = true;
    SimulateInputMap["i_tag01"].start_value = 100;
    SimulateInputMap["i_tag01"].min = 90;
    SimulateInputMap["i_tag01"].max = 110;
    SimulateInputMap["i_tag01"].up_step = 1;
    SimulateInputMap["i_tag01"].down_step = .25;

    ModbusRegisterMap["i_bdcond01"].simulated = true;
    SimulateInputMap["i_bdcond01"].start_value = 1800;
    SimulateInputMap["i_bdcond01"].min = 1725;
    SimulateInputMap["i_bdcond01"].max = 1875;
    SimulateInputMap["i_bdcond01"].up_step = 1;
    SimulateInputMap["i_bdcond01"].down_step = 5;

    ModbusRegisterMap["i_ph01"].simulated = true;
    SimulateInputMap["i_ph01"].start_value = 8000;
    SimulateInputMap["i_ph01"].min = 7250;
    SimulateInputMap["i_ph01"].max = 8750;
    SimulateInputMap["i_ph01"].up_step = 5;
    SimulateInputMap["i_ph01"].down_step = 10;

    ModbusRegisterMap["i_orp01"].simulated = true;
    SimulateInputMap["i_orp01"].start_value = 350;
    SimulateInputMap["i_orp01"].min = 300;
    SimulateInputMap["i_orp01"].max = 400;
    SimulateInputMap["i_orp01"].up_step = 5;
    SimulateInputMap["i_orp01"].down_step = 1;

}

void cmd_deep( int argc, char **argv )
{
    UNUSED(argc);
    UNUSED(argv);

    char logString[LOG_BYTES_PER_ENTRY];

    LogLocalApi_PopEntry( logString );
    if( logString[0] != '\0' ) {
        printf("%s\r\n", logString );
    } else {
        printf("%s\r\n", "No String Found" );
    }
}

void cmd_peep( int argc, char **argv )
{
    UNUSED(argc);
    UNUSED(argv);

    std::string logString = "This is a string to log";

    LogLocalApi_PushEntry( logString.c_str() );
}

void cmd_inslog( int argc, char **argv )
{
    UNUSED(argc);
    UNUSED(argv);

    EventReasonStruct_t eventReason;
    eventReason.eventReason = EVENT_REASON_AUTO;
    eventReason.inputValue = 100.00;
    strncpy(eventReason.inputTag, "i_stub01", sizeof(eventReason.inputTag) );
    eventReason.outputValue = 0.0;
    strncpy(eventReason.outputTag, "o_stub01", sizeof(eventReason.outputTag) );
    EventLoggerApi( eventReason );
}

void cmd_showSimin( int argc, char **argv )
{
    std::map<std::string, SimulateInput>::iterator iter;
    for (iter = SimulateInputMap.begin(); iter != SimulateInputMap.end(); ++iter) {
        printf("simulated input=%s, min=%2.2f, max=%2.2f, start_value=%2.2f, up_step=%2.2f, down_step=%2.2f\r\n",iter->first.c_str(), SimulateInputMap[iter->first].min, SimulateInputMap[iter->first].max, SimulateInputMap[iter->first].start_value, SimulateInputMap[iter->first].up_step, SimulateInputMap[iter->first].down_step);
    }
}

void cmd_preload(int argc, char **argv)
{
    UNUSED(argc);
    UNUSED(argv);

    char buf[MAX_FILE_SIZE];
    bool rc;

    // PRELOADER:
    //
    // Inputs:
    //      input_i_bdcond.json         // conductivity probe
    //      input_i_bdcond_temp.json    // conductivity probe : temperature
    //      input_i_flowsw01.json       // flow switch
    //      input_i_orp01.json          // ORP probe
    //      input_i_ph01.json           // pH probe
    //      input_i_ph01_temp.json      // pH probe : temperature
    //      input_i_tag01.json          // tag (flu)
    //      input_i_tra01.json          // Trasar
    //
    // Outputs:
    //      output_o_rly01.json         // relay-1
    //      output_o_rly02.json         // relay-2
    //
    // Standard Controls:
    //      control_sp_i_bd01.json        // blowdown
    //      control_sp_i_tra01.json       // Trasar
    //
    // Control Algorithms:
    //      control_ca_eq1.json         // == 1
    //      control_ca_eq129.json       // == 129
    //
    // Composite Controls:
    //      control_comp_flow.json      // flow override
    //      control_comp_lockout.json   // lockout blowdown

    std::string filename = "input_i_bdcond01.json";

    // INPUT: Conductivity
    snprintf(buf, sizeof(buf),
             "{ "
             "\"id\":        \"i_bdcond01\","
             "\"name\":      \"Tower Conductivity\", "
             "\"units\":     \"uS\","
             "\"min\":       \"0\","
             "\"max\":       \"6000\","
             "\"node\":      \"21\","
             "\"reg\":       \"18\","
             "\"rtype\":     \"4\","
             "\"type\":      \"1\","
             "\"size\":      \"2\","
             "\"order\":     \"2\","
             "\"fmt\":       \"%%.2f\","
             "\"vcmd\":      \"\","
             "\"rfreq\":     \"5\" }");
    rc = GLOBAL_mdot->saveUserFile(filename.c_str(), (void *)buf, MAX_FILE_SIZE);
    if ( rc != true ) {
        printf("\rFailed to save %s\n", filename.c_str());
        return;
    } else {
        printf("\r...generated %s\n", filename.c_str());
    }

    Thread::wait(1000);

    // INPUT: Conductivity Temperature
    filename = "input_i_cond_temp01.json";
    snprintf(buf, sizeof(buf),
             "{ "
             "\"id\":        \"i_cond_temp01\","
             "\"name\":      \"Temp Conductivity\","
             "\"units\":     \"C\","
             "\"min\":       \"0\","
             "\"max\":       \"80\","
             "\"node\":      \"21\","
             "\"reg\":       \"10\","
             "\"rtype\":     \"4\","
             "\"type\":      \"1\","
             "\"size\":      \"2\","
             "\"order\":     \"2\","
             "\"fmt\":       \"%%.2f\","
             "\"vcmd\":      \"\","
             "\"rfreq\":     \"5\" }");
    rc = GLOBAL_mdot->saveUserFile(filename.c_str(), (void *)buf, MAX_FILE_SIZE);
    if ( rc != true ) {
        printf("\rFailed to save %s\n", filename.c_str());
        return;
    } else {
        printf("\r...generated %s\n", filename.c_str());
    }

    Thread::wait(1000);

    // INPUT: Flow Switch
    filename = "input_i_flowsw01.json";
    snprintf(buf, sizeof(buf),
             "{ "
             "\"id\":        \"i_flowsw01\","
             "\"name\":      \"Flow\","
             "\"units\":     \"\","
             "\"min\":       \"0\","
             "\"max\":       \"1\","
             "\"node\":      \"0\","
             "\"reg\":       \"1\","
             "\"rtype\":     \"0\","
             "\"type\":      \"3\","
             "\"size\":      \"2\","
             "\"order\":     \"0\","
             "\"fmt\":       \"%%d\","
             "\"vcmd\":      \"\","
             "\"rfreq\":     \"5\" }");
    rc = GLOBAL_mdot->saveUserFile(filename.c_str(), (void *)buf, MAX_FILE_SIZE);
    if ( rc != true ) {
        printf("\rFailed to save %s\n", filename.c_str());
        return;
    } else {
        printf("\r...generated %s\n", filename.c_str());
    }

    Thread::wait(1000);

    // INPUT: ORP
    filename = "input_i_orp01.json";
    snprintf(buf, sizeof(buf),
             "{ "
             "\"id\":        \"i_orp01\","
             "\"name\":      \"ORP\","
             "\"units\":     \"mV\","
             "\"min\":       \"0\","
             "\"max\":       \"10\","
             "\"node\":      \"23\","
             "\"reg\":       \"14\","
             "\"rtype\":     \"4\","
             "\"type\":      \"1\","
             "\"size\":      \"2\","
             "\"order\":     \"2\","
             "\"fmt\":       \"%%.2f\","
             "\"vcmd\":      \"\","
             "\"rfreq\":     \"5\" }");
    rc = GLOBAL_mdot->saveUserFile(filename.c_str(), (void *)buf, MAX_FILE_SIZE);
    if ( rc != true ) {
        printf("\rFailed to save %s\n", filename.c_str());
        return;
    } else {
        printf("\r...generated %s\n", filename.c_str());
    }

    Thread::wait(1000);

    // INPUT: pH
    filename = "input_i_ph01.json";
    snprintf(buf, sizeof(buf),
             "{ "
             "\"id\":        \"i_ph01\","
             "\"name\":      \"pH\","
             "\"units\":     \"pH\","
             "\"min\":       \"0\","
             "\"max\":       \"10\","
             "\"node\":      \"22\","
             "\"reg\":       \"18\","
             "\"rtype\":     \"4\","
             "\"type\":      \"1\","
             "\"size\":      \"2\","
             "\"order\":     \"2\","
             "\"fmt\":       \"%%.2f\","
             "\"vcmd\":      \"\","
             "\"rfreq\":     \"5\" }");
    rc = GLOBAL_mdot->saveUserFile(filename.c_str(), (void *)buf, MAX_FILE_SIZE);
    if ( rc != true ) {
        printf("\rFailed to save %s\n", filename.c_str());
        return;
    } else {
        printf("\r...generated %s\n", filename.c_str());
    }

    Thread::wait(1000);

    // INPUT: pH Temperature
    filename = "input_i_ph01_temp.json";
    snprintf(buf, sizeof(buf),
             "{ "
             "\"id\":        \"i_ph_temp01\","
             "\"name\":      \"Temp pH\","
             "\"units\":     \"C\","
             "\"min\":       \"0\","
             "\"max\":       \"10\","
             "\"node\":      \"22\","
             "\"reg\":       \"10\","
             "\"rtype\":     \"4\","
             "\"type\":      \"1\","
             "\"size\":      \"2\","
             "\"order\":     \"2\","
             "\"fmt\":       \"%%.2f\","
             "\"vcmd\":      \"\","
             "\"rfreq\":     \"5\" }");
    rc = GLOBAL_mdot->saveUserFile(filename.c_str(), (void *)buf, MAX_FILE_SIZE);
    if ( rc != true ) {
        printf("\rFailed to save %s\n", filename.c_str());
        return;
    } else {
        printf("\r...generated %s\n", filename.c_str());
    }

    Thread::wait(1000);

    // INPUT: Tag
    filename = "input_i_tag01.json";
    snprintf(buf, sizeof(buf),
             "{ "
             "\"id\":        \"i_tag01\","
             "\"name\":      \"Tag\", "
             "\"units\":     \"PPM\",    "
             "\"min\":       \"0\",      "
             "\"max\":       \"300\",    "
             "\"node\":      \"1\",      "
             "\"reg\":       \"11\",      "
             "\"rtype\":     \"4\",      "
             "\"type\":      \"0\",      "
             "\"size\":      \"2\",      "
             "\"order\":     \"2\",      "
             "\"fmt\":       \"%%.2f\",   "
             "\"vcmd\":      \"\",       "
             "\"rfreq\":     \"5\" }");
    rc = GLOBAL_mdot->saveUserFile(filename.c_str(), (void *)buf, MAX_FILE_SIZE);
    if ( rc != true ) {
        printf("\rFailed to save %s\n", filename.c_str());
        return;
    } else {
        printf("\r...generated %s\n", filename.c_str());
    }

    Thread::wait(1000);

    // INPUT: Trasar
    filename = "input_i_tra01.json";
    snprintf(buf, sizeof(buf),
             "{ "
             "\"id\":        \"i_tra01\","
             "\"name\":      \"Trasar\", "
             "\"units\":     \"PPM\",    "
             "\"min\":       \"0\",      "
             "\"max\":       \"300\",    "
             "\"node\":      \"1\",      "
             "\"reg\":       \"9\",      "
             "\"rtype\":     \"4\",      "
             "\"type\":      \"0\",      "
             "\"size\":      \"2\",      "
             "\"order\":     \"2\",      "
             "\"fmt\":       \"%%.2f\",   "
             "\"vcmd\":      \"\",       "
             "\"rfreq\":     \"5\" }");
    rc = GLOBAL_mdot->saveUserFile(filename.c_str(), (void *)buf, MAX_FILE_SIZE);
    if ( rc != true ) {
        printf("\rFailed to save %s\n", filename.c_str());
        return;
    } else {
        printf("\r...generated %s\n", filename.c_str());
    }

    Thread::wait(1000);

    // OUTPUT: Relay 01
    filename = "output_o_rly01.json";
    snprintf(buf, sizeof(buf),
             "{ "
             "\"id\": \"o_rly01\", "
             "\"name\": \"Blowdown\", "
             "\"units\": \"\", "
             "\"min\":  \"0\", "
             "\"max\": \"300\", "
             "\"node\": \"0\", "
             "\"reg\": \"1\", "
             "\"rtype\": \"1\", "
             "\"type\": \"0\", "
             "\"size\": \"2\", "
             "\"order\": \"2\", "
             "\"fmt\": \"%%.2f\", "
             "\"vcmd\": \"\", "
             "\"rfreq\": \"5\", "
             "\"toperiod\": \"0\", "
             "\"scalelo\": \"0\", "
             "\"scalehi\": \"100\" }");
    rc = GLOBAL_mdot->saveUserFile(filename.c_str(), (void *)buf, MAX_FILE_SIZE);
    if ( rc != true ) {
        printf("\rFailed to save %s\n", filename.c_str());
        return;
    } else {
        printf("\r...generated %s\n", filename.c_str());
    }

    Thread::wait(1000);

    // OUTPUT: Relay 02
    filename = "output_o_rly02.json";
    snprintf(buf, sizeof(buf),
             "{ "
             "\"id\": \"o_rly02\", "
             "\"name\": \"3DTS86\", "
             "\"units\": \"\", "
             "\"min\":  \"0\", "
             "\"max\": \"300\", "
             "\"node\": \"0\", "
             "\"reg\": \"2\", "
             "\"rtype\": \"1\", "
             "\"type\": \"0\", "
             "\"size\": \"2\", "
             "\"order\": \"2\", "
             "\"fmt\": \"%%.2f\", "
             "\"vcmd\": \"\", "
             "\"rfreq\": \"5\", "
             "\"toperiod\": \"0\", "
             "\"scalelo\": \"0\", "
             "\"scalehi\": \"100\" }");
    rc = GLOBAL_mdot->saveUserFile(filename.c_str(), (void *)buf, MAX_FILE_SIZE);
    if ( rc != true ) {
        printf("\rFailed to save %s\n", filename.c_str());
        return;
    } else {
        printf("\r...generated %s\n", filename.c_str());
    }

    Thread::wait(1000);

    // SETPOINT: Blowdown
    filename = "control_sp_BLOWDOWN_01.json";
    snprintf(buf, sizeof(buf),
             "{ "
             "\"id\": \"BLOWDOWN_01\", "
             "\"priority\": \"800\", "
             "\"input\": \"i_bdcond01\", "
             "\"output\": \"o_rly01\", "
             "\"setpoint\": \"1800\", "
             "\"prodfact\": \"\", "
             "\"actingDir\": \"1\", "
             "\"halert\": \"2400\", "
             "\"lalert\": \"800\", "
             "\"hfs\": \"2700\", "
             "\"lfs\": \"700\", "
             "\"tol\": \"30\" }");
    rc = GLOBAL_mdot->saveUserFile(filename.c_str(), (void *)buf, MAX_FILE_SIZE);
    if ( rc != true ) {
        printf("\rFailed to save %s\n", filename.c_str());
        return;
    } else {
        printf("\r...generated %s\n", filename.c_str());
    }

    Thread::wait(1000);

    // SETPOINT: Trasar
    filename = "control_sp_INH_TRA_01.json";
    snprintf(buf, sizeof(buf),
             "{ "
             "\"id\": \"INH_TRA_01\", "
             "\"priority\": \"800\", "
             "\"input\": \"i_tra01\", "
             "\"output\": \"o_rly02\", "
             "\"setpoint\": \"130\", "
             "\"prodfact\": \"100\", "
             "\"actingDir\": \"0\", "
             "\"halert\": \"190\", "
             "\"lalert\": \"92\", "
             "\"hfs\": \"200\", "
             "\"lfs\": \"70\", "
             "\"tol\": \"4\" }");
    rc = GLOBAL_mdot->saveUserFile(filename.c_str(), (void *)buf, MAX_FILE_SIZE);
    if ( rc != true ) {
        printf("\rFailed to save %s\n", filename.c_str());
        return;
    } else {
        printf("\r...generated %s\n", filename.c_str());
    }

    Thread::wait(1000);

    // ALGORITHM == 1
    filename = "control_ca_EQUAL_TO_1.json";
    snprintf(buf, sizeof(buf),
             "{"
             "\"id\":    \"EQUAL_TO_1\", "
             "\"opr\":   \"1\", "
             "\"op\":    \"==\", "
             "\"true\":   \"responseA\", "
             "\"false\":  \"nothing\" "
             "}");

    rc = GLOBAL_mdot->saveUserFile(filename.c_str(), (void *)buf, MAX_FILE_SIZE);
    if ( rc != true ) {
        printf("\rFailed to save user file %s\n", filename.c_str());
        return;
    } else {
        printf("\r...generated %s\n", filename.c_str());
    }

    Thread::wait(1000);

    // ALGORITHM == 129
    filename = "control_ca_EQUAL_TO_129.json";
    snprintf(buf, sizeof(buf),
             "{"
             "\"id\":    \"EQUAL_TO_129\", "
             "\"opr\":   \"129\", "
             "\"op\":    \"==\", "
             "\"true\":   \"responseA\", "
             "\"false\":  \"nothing\" "
             "}");

    rc = GLOBAL_mdot->saveUserFile(filename.c_str(), (void *)buf, MAX_FILE_SIZE);
    if ( rc != true ) {
        printf("\rFailed to save user file %s\n", filename.c_str());
        return;
    } else {
        printf("\r...generated %s\n", filename.c_str());
    }

    Thread::wait(1000);

    // FLOW
    filename = "control_comp_FLOW_OVERRIDE.json";
    snprintf(buf, sizeof(buf),
             "{ "
             "\"id\":  \"FLOW_OVERRIDE\",  "
             "\"tag\": \"i_flowsw01\",  "
             "\"ca\":  \"EQUAL_TO_1\",   "
             "\"priority\": \"400\", "
             "\"entries\":  \"2\", "
             "\"outputs\": [ "
             "{"
             "\"tag\": \"o_rly01\", "
             "\"responseA\": \"fixed off\" "
             "},"
             "{"
             "\"tag\": \"o_rly02\", "
             "\"responseA\": \"fixed off\" "
             "}"
             "]"
             "}"
            );
    rc = GLOBAL_mdot->saveUserFile(filename.c_str(), (void *)buf, MAX_FILE_SIZE);
    if ( rc != true ) {
        printf("\rFailed to save user file %s\n", filename.c_str());
        return;
    } else {
        printf("\r...generated %s\n", filename.c_str());
    }

    Thread::wait(1000);

    // LOCKOUT
    filename = "control_comp_LOCKOUT_BLOWDOWN.json";
    snprintf(buf, sizeof(buf),
             "{ "
             "\"id\":  \"LOCKOUT_BLOWDOWN\",  "
             "\"tag\": \"o_rly01\",  "
             "\"ca\":  \"EQUAL_TO_129\",   "
             "\"priority\": \"400\", "
             "\"entries\":  \"1\", "
             "\"outputs\": [ "
             "{"
             "\"tag\": \"o_rly02\", "
             "\"responseA\": \"fixed off\" "
             "}"
             "]"
             "}"
            );
    rc = GLOBAL_mdot->saveUserFile(filename.c_str(), (void *)buf, MAX_FILE_SIZE);
    if ( rc != true ) {
        printf("\rFailed to save user file %s\n", filename.c_str());
        return;
    } else {
        printf("\r...generated %s\n", filename.c_str());
    }
    
    filename = "control_fs_BLOWDOWN_FS.json";
    snprintf(buf, sizeof(buf),
             "{"
             "\"id\":    \"BLOWDOWN_FS\", "
             "\"input\":   \"i_bdcond01\", "
             "\"output\":  \"o_rly01\", "
             "\"priority\":\"700\", "
             "\"lfsValue\":\"700\", "
             "\"lfsDutyCycle\":\"0\", "
             "\"lfsInterval\":\"1\", "
             "\"hfsValue\":\"2700\", "
             "\"hfsDutyCycle\":\"50\", "
             "\"hfsInterval\":\"1\" "
             "}");

    rc = GLOBAL_mdot->saveUserFile(filename.c_str(), (void *)buf, MAX_FILE_SIZE);
    if ( rc != true ) {
        printf("\rFailed to create %s\n", filename.c_str());
        return;
    } else {
        printf("\r...generated %s\n", filename.c_str());
    }

    
    Thread::wait(1000);

    filename = "control_fs_TRASAR_FS.json";
    snprintf(buf, sizeof(buf),
             " {"
             "\"id\":    \"TRASAR_FS\", "
             "\"input\":   \"i_tra01\", "
             "\"output\":  \"o_rly02\", "
             "\"priority\":\"700\", "
             "\"lfsValue\":\"70\", "
             "\"lfsDutyCycle\":\"20\", "
             "\"lfsInterval\":\"15\", "
             "\"hfsValue\":\"200\", "
             "\"hfsDutyCycle\":\"0\", "
             "\"hfsInterval\":\"15\" "
             "}");

    rc = GLOBAL_mdot->saveUserFile(filename.c_str(), (void *)buf, MAX_FILE_SIZE);
    if ( rc != true ) {
        printf("\rFailed to create %s\n", filename.c_str());
        return;
    } else {
        printf("\r...generated %s\n", filename.c_str());
    }
    

    printf("\r\nIssue \"reset\" command to invoke changes\r\n\r\n\r\n");

}

/*****************************************************************************
 * Function:        cmd_createTemp
 * Description:     create input file
 *
 * @param           argc-> number of args
 * @param           argv-> filename
 * @return          none
 *****************************************************************************/
void cmd_createTemp(int argc, char **argv)
{
    char data_buf[MAX_FILE_SIZE];
    snprintf(data_buf, sizeof(data_buf),
             "{ "
             "\"id\":       \"i_temp01\", "
             "\"name\":     \"Temp Conductivity\", "
             "\"units\":    \"C\", "
             "\"min\":      \"0\", "
             "\"max\":      \"300\", "
             "\"node\":     \"21\", "
             "\"reg\":      \"10\", "
             "\"rtype\":    \"1\", "
             "\"type\":     \"1\", "
             "\"size\":     \"2\", "
             "\"order\":    \"2\", "
             "\"rfreq\":    \"5\", "
             "\"vcmd\":      \"\","
             "\"fmt\":      \"%%.2f\" } ");

    bool status = GLOBAL_mdot->saveUserFile("input_i_temp01.json", (void *)data_buf, MAX_FILE_SIZE);
    if( status != true ) {
        logInfo("(%d)save file failed, status=%d", __LINE__, status);
    }

    logInfo("Sending Mail To ModbusMasterMailBox, filename=%s", "input_i_temp01.json");
    ModbusMasterReq_t *mail = ModbusMasterMailBox.alloc();
    mail->action = ACTION_READ_FILE;
    strncpy( mail->controlFile, "input_i_temp01.json", (sizeof(mail->controlFile)-1));
    ModbusMasterMailBox.put(mail);
}

/*****************************************************************************
 * Function:        cmd_createBDCond
 * Description:     create input file
 *
 * @param           argc-> number of args
 * @param           argv-> filename
 * @return          none
 *****************************************************************************/
void cmd_modbusCmd(int argc, char **argv)
{
    if ( (argc != 7) && (argc != 8)) {
        printf("\r\nusage:   mod-cmd <node> <func> <sreg> <nreg> <dtype> <order> <value>\r\n");
        printf("example (read product factor): mod-cmd 1 3 40 2 0 2\r\n");
        printf("example (write product factor): mod-cmd 1 16 40 2 0 2 100.0\r\n");
        return;
    }

    char snd_buf[128];

    memset( snd_buf, '\0', 128 );
    snprintf(snd_buf, sizeof(snd_buf),
             "{"
             "\"id\":\"%s\","
             "\"node\":\"%s\","
             "\"func\":\"%s\","
             "\"sreg\":\"%s\","
             "\"nreg\":\"%s\","
             "\"dtype\":\"%s\","
             "\"order\":\"%s\","
             "\"value\":\"%s\"}",
             "READ_MODBUS",
             argv[1],
             argv[2],
             argv[3],
             argv[4],
             argv[5],
             argv[6],
             argv[7]
            );

    printf("Sending Command To ModbusMasterMailBox: %s : len=%d\r\n", snd_buf, strlen(snd_buf));
    ModbusMasterReq_t *mail = ModbusMasterMailBox.alloc();
    mail->action = ACTION_EXEC_CMD;
    strncpy( mail->controlFile, snd_buf, (sizeof(mail->controlFile)-1));
    ModbusMasterMailBox.put(mail);
}

/*****************************************************************************
 * Function:        cmd_regcmd
 * Description:     create register command
 *
 * @param           argc-> number of args
 * @param           argv-> filename
 * @return          none
 *****************************************************************************/
void cmd_regcmd(int argc, char **argv)
{
    char filename[64];

    printf("argc=%d\r\n",argc);

    if ( argc != 7 ) {
        printf("\rusage:   regcmd <if> <tag> <opl> <opr> <op> <c>\n");
        printf("\rexample: regcmd TEMP_CONV_PH01 i_ph_temp01 i_ph_temp01 - / 1000\n");
        return;
    }

    memset( filename, 0, 64 );
    snprintf( filename, 63, "%s%s%s", "vcmd_", argv[1], ".json" );
    printf("filename=%s\r\n",filename);

    std::string opl(argv[4]);

    printf("argv[4]=%s, opl=%s\r\n",argv[4], opl.c_str());
    if( opl.c_str()[0] == '-' ) {
        printf("argv[4]=%s, SETTING TO NULL\r\n",argv[4]);
        opl = "";
    }

    char data_buf[MAX_FILE_SIZE];
    snprintf(data_buf, sizeof(data_buf),
             "{ "
             "\"id\":\"%s\", "
             "\"tag\":\"%s\", "
             "\"opl\":\"%s\", "
             "\"opr\":\"%s\", "
             "\"op\":\"%s\", "
             "\"c\":\"%s\" } ", argv[1], argv[2], argv[3], opl.c_str(), argv[5], argv[6]);

    printf("JSON=%s\r\n",data_buf);

    bool status = GLOBAL_mdot->saveUserFile(filename, (void *)data_buf, MAX_FILE_SIZE);
    if( status != true ) {
        logInfo("(%d)save file failed, status=%d", __LINE__, status);
    }

    logInfo("Sending Mail To ModbusMasterMailBox, filename=%s", filename);
    ModbusMasterReq_t *mail = ModbusMasterMailBox.alloc();
    mail->action = ACTION_READ_FILE;
    strncpy( mail->controlFile, filename, (sizeof(mail->controlFile)-1));
    ModbusMasterMailBox.put(mail);
}

/*****************************************************************************
 * Function:        cmd_trimTemp
 * Description:     create register command
 *
 * @param           argc-> number of args
 * @param           argv-> filename
 * @return          none
 *****************************************************************************/
void cmd_trimTemp(int argc, char **argv)
{
    /*

        char filename[64];

        memset( filename, 0, 64 );
        snprintf( filename, 63, "%s%s%s", "vcmd_", "2_DIV_TEMP_PH01", ".json" );
        printf("filename=%s\r\n",filename);

        char data_buf[MAX_FILE_SIZE];
        snprintf(data_buf, sizeof(data_buf),
                 "{ "
                 "\"id\":\"%s\", "
                 "\"tag\":\"%s\", "
                 "\"opl\":\"%s\", "
                 "\"opr\":\"%s\", "
                 "\"op\":\"%s\", "
                 "\"c\":\"%s\" } ", "2_DIV_TEMP_PH01", "i_ph_temp01", "i_ph_temp01", "", "/", "1000.0");

        printf("JSON=%s\r\n",data_buf);

        bool status = GLOBAL_mdot->saveUserFile(filename, (void *)data_buf, MAX_FILE_SIZE);
        if( status != true ) {
            logInfo("(%d)save file failed, status=%d", __LINE__, status);
        }

        logInfo("Sending Mail To ModbusMasterMailBox, filename=%s", filename);
        Message_t *mail = ModbusMasterMailBox.alloc();
        mail->action = ACTION_READ_FILE;
        strncpy( mail->controlFile, filename, (sizeof(mail->controlFile)-1));
        ModbusMasterMailBox.put(mail);

        Thread::wait(5000);

        memset( filename, 0, 64 );
        snprintf( filename, 63, "%s%s%s", "vcmd_", "1_CONST_TEMP_PH01", ".json" );
        printf("filename=%s\r\n",filename);

        snprintf(data_buf, sizeof(data_buf),
                 "{ "
                 "\"id\":\"%s\", "
                 "\"tag\":\"%s\", "
                 "\"opl\":\"%s\", "
                 "\"opr\":\"%s\", "
                 "\"op\":\"%s\", "
                 "\"c\":\"%s\" } ", "1_CONST_TEMP_PH01", "i_ph_temp01", "i_ph_temp01", "", "=", "26000.0");

        printf("JSON=%s\r\n",data_buf);

        status = GLOBAL_mdot->saveUserFile(filename, (void *)data_buf, MAX_FILE_SIZE);
        if( status != true ) {
            logInfo("(%d)save file failed, status=%d", __LINE__, status);
        }

        logInfo("Sending Mail To ModbusMasterMailBox, filename=%s", filename);
        mail = ModbusMasterMailBox.alloc();
        mail->action = ACTION_READ_FILE;
        strncpy( mail->controlFile, filename, (sizeof(mail->controlFile)-1));
        ModbusMasterMailBox.put(mail);

        Thread::wait(5000);
    */
}