
// L. van der Kolk, ELVEDEKA, Holland //
// File:  FS2CDU_data_5.cpp 
// -- Message handling from FS to CDU --
// Data field analyzing and filling data structures

#include "mbed.h"
#include "MODSERIAL.h" 
#include "mbos.h"
#include "mbos_def2.h"

extern mbos CDU_OS;

// ---- FS-to-CDU data structures to be filled with received data ----------------
#define max_text0    48   //  max text position font 0  (pos 0 invalid )    
#define max_text1    24   //  max text position font 1  (pos 0 invalid )
#define max_lines    14   //  max nr of screen lines 
#define max_col_nr  255   //  highest possible R,G, or B colour number
#define fstyle_1     'S'  //  possible font style character 1
#define fstyle_2     'N'  //  possible font style character 2
#define max_keys    100   //  max total nr of select keys 0 - 99
#define max_leftkeys  6   //  max nr of used LEFT select keys ( 0 - 49 )
#define max_rightkeys 6   //  max nr of used RIGHT select keys ( 50 - 99 )

// Background colour of CDU display:
struct { int   BG_RED;            //  : RED value   0 - 255
         int   BG_GREEN;          //  : GREEN value 0 - 255
         int   BG_BLUE;           //  : BLUE value  0 - 255   
       } BACKGROUND_COL;

// Maintext lines at selectkeys:
struct { char  text[55];             //  : maintext string, ending with '\0'   
         int   font_size;            //  : fontsize of textline 0 or 1            
         char  font_style;           //  : style character S or N                       
         int   text_RED;             //  : RED value of textline
         int   text_GREEN;           //  : GREEN value of textline
         int   text_BLUE;            //  : BLUE value of textline
       } SELKEY_MAINTEXT[100];  //  : array of selectkey Maintext structures
        
// Subtext lines at selectkeys:
struct { char  text[55];             //  : subtext string, ending with '\0'   
         int   font_size;            //  : fontsize of textline 0 or 1                            
         char  font_style;           //  : style character S or N                        
         int   text_RED;             //  : RED value of textline
         int   text_GREEN;           //  : GREEN value of textline
         int   text_BLUE;            //  : BLUE value of textline
       } SELKEY_SUBTEXT[100];   //  : array of selectkey Subtext structures 

// Screen textlines:
struct { char  text[55];             //  : text line string, ending with '\0'   
         int   font_size;            //  : fontsize of textline 0 or 1         
         char  font_style;           //  : style character S or N                             
         int   text_RED;             //  : RED value of textline
         int   text_GREEN;           //  : GREEN value of textline
         int   text_BLUE;            //  : BLUE value of textline
         } TEXTLINE[16];             //  : array of textline structures
            
// CDU status:            
 struct { int msg_indicator;       //  : MSG  0 = light OFF, 1 = light ON
          int exec_indicator;      //  : EXEC 0 = indicator OFF, 1 = indicator ON
          int fail_indicator;      //  : FAIL 0 = indicator OFF, 1 = indicator ON
          int dspy_indicator;      //  : DSPY 0 = indicator OFF, 1 = indicator ON  
          int ofst_indicator;      //  : OFST 0 = indicator OFF, 1 = indicator ON                              
          int backlight;           //  : 0 = light OFF, 1 = light ON 
          int stby_mode;           //  : 0 = operational mode, 1 = standby mode
        } CDU_STATUS;           
        

// FS_data_update_ID:
// These global(!) flags indicate what data has been updated:
 int Background_Col_Update      = 0;    // : 1 when color was updated, must be reset to 0 when data has been read
 int CDU_Status_Update          = 0;    // : 1 when status was updated, must be reset to 0 when data has been read
 int DO_CLR_SCREEN              = 0;    // : 1 when screen should be cleared, must be reset to 0 when done
 int Text_Line_Update           = 0;    // : equal to line number whose text was updated, must be reset to 0 when text has been read 
 int Key_Maintext_Update        = -1;   // : equal to keynumber whose main text line was updated, must be reset to -1 (!) when text has been read
 int Key_Subtext_Update         = -1;   // : equal to keynumber whose sub text line was updated, must be reset to -1 (!) when text has been read
 int S_Exec                     = 0;    // WB 20-07-14 if 1 then S+EXEC is pressed together to enter setup mode 
                                        // -->No screen updates allowed
                                        // -->No key messages sent to FS
                                        // Setup is exited by pressing EXEC

//------------- debug only ---------------------------
extern MODSERIAL SERIAL_DEBUG; // : debug serial port
//----------------------------------------------------

extern MODSERIAL USB; 
extern char string_received[];  
extern int comma[];

// Common flag to signal that an update was stored in the datastructures:
int FSdata_received_flag = false;  //  : true when some FS-to-CDU data was updated 

void set_initial_data()
{  // Function to fill all FS-to-CDU datastructures with (dummy) data at initial power-up conditions.
   // Prevents non-printable chars in data structures !
   int key_nr, line_nr;  
  // Init background colour of screen :
   BACKGROUND_COL.BG_RED =   128;
   BACKGROUND_COL.BG_GREEN = 128;
   BACKGROUND_COL.BG_BLUE =  128;
   
  // Filling of all select key datastructures:
  // Init Maintext at left selectkeys starting from 0 :
  for ( key_nr = 0; key_nr < max_leftkeys; key_nr++ )
  {  SELKEY_MAINTEXT[key_nr].text_RED =   128;
     SELKEY_MAINTEXT[key_nr].text_GREEN = 128;
     SELKEY_MAINTEXT[key_nr].text_BLUE =  128;
     SELKEY_MAINTEXT[key_nr].font_size =    0;
     SELKEY_MAINTEXT[key_nr].font_style = fstyle_1;
     strcpy(SELKEY_MAINTEXT[key_nr].text, "   \0");  // some space chars
  }
  // Init Maintext at right selectkeys starting from 50 :
  for ( key_nr = 50; key_nr < (50+max_rightkeys); key_nr++ )
  {  SELKEY_MAINTEXT[key_nr].text_RED =   128;
     SELKEY_MAINTEXT[key_nr].text_GREEN = 128;
     SELKEY_MAINTEXT[key_nr].text_BLUE =  128;
     SELKEY_MAINTEXT[key_nr].font_size =    0;
     SELKEY_MAINTEXT[key_nr].font_style = fstyle_1; 
     strcpy(SELKEY_MAINTEXT[key_nr].text, "   \0"); // some space chars
  }
  // Init Subtext at left selectkeys starting from 0 :
  for ( key_nr = 0; key_nr < max_leftkeys; key_nr++ )
  {  SELKEY_SUBTEXT[key_nr].text_RED =   128;
     SELKEY_SUBTEXT[key_nr].text_GREEN = 128;
     SELKEY_SUBTEXT[key_nr].text_BLUE =  128;
     SELKEY_SUBTEXT[key_nr].font_size =    0;
     SELKEY_SUBTEXT[key_nr].font_style = fstyle_1; 
     strcpy(SELKEY_SUBTEXT[key_nr].text,"    \0"); // some space chars 
  }   
  // Init Subtext at right selectkeys starting from 50 :
  for ( key_nr = 50; key_nr < (50+max_rightkeys); key_nr++ )
  {  SELKEY_SUBTEXT[key_nr].text_RED =   128;
     SELKEY_SUBTEXT[key_nr].text_GREEN = 128;
     SELKEY_SUBTEXT[key_nr].text_BLUE =  128;
     SELKEY_SUBTEXT[key_nr].font_size =    0;
     SELKEY_SUBTEXT[key_nr].font_style = fstyle_1; 
     strcpy(SELKEY_SUBTEXT[key_nr].text, "    \0"); // some space chars
  } 
        
  // Init screen text lines with space chars :
  for ( line_nr = 1; line_nr <= 14; line_nr++ )
  {  TEXTLINE[line_nr].text_RED =   128;
     TEXTLINE[line_nr].text_GREEN = 128;
     TEXTLINE[line_nr].text_BLUE =  128;
     TEXTLINE[line_nr].font_size =    0;
     TEXTLINE[line_nr].font_style = fstyle_1; 
     strcpy(TEXTLINE[line_nr].text, "                                                \0");   // : all space chars                      
  }   
   // Init CDU status items:
   CDU_STATUS.msg_indicator = 0;
   CDU_STATUS.exec_indicator = 0;
   CDU_STATUS.backlight = 0;
   CDU_STATUS.stby_mode = 0;
   CDU_STATUS.dspy_indicator = 0;
   CDU_STATUS.fail_indicator = 0;
   CDU_STATUS.ofst_indicator = 0;
}

void read_datafields(int command_number) {
    // Function to read all comma seperated datafields in string_received[] from USB
    // and fill in the FS-to-CDU data structures.
    // Parameter is found command_number by decoder.
    int colour, size, pos, line, last, cnt, k_id, a ;
    int modified, char_pntr;
    char ch;
    
    modified = false; 
    //CDU_OS.LockResource(FS_DATA_RESOURCE); // : lock FS database  <<--- !!
      
    switch ( command_number )
    {   case 0:  
        {   // 0 is no valid commandnumber, do nothing
           break;
        }
        
        case 1:  // MSG command, message indicator control CDU
        {        
             // Get message indicator status:
             char_pntr = comma[1] + 1; // : set char pointer to char after comma 1
             if (string_received[char_pntr] == '0' || string_received[char_pntr] == '1')
             { CDU_STATUS.msg_indicator = atoi(&string_received[char_pntr]);
               CDU_Status_Update = 1;
             }
           break;
        }
        
        case 2: // EXC command, exec indicator control CDU
        {               
             // Get exec indicator status:
             char_pntr = comma[1] + 1; // : set char pointer to char after comma 1
             if (string_received[char_pntr] == '0' || string_received[char_pntr] == '1')
             { CDU_STATUS.exec_indicator = atoi(&string_received[char_pntr]);
               CDU_Status_Update = 1;
             }
           break;
        }

        case 3: // BLT command, backlight control CDU     
        {                 
             // Get backlight status:
             char_pntr = comma[1] + 1; // : set char pointer to char after comma 1
             if (string_received[char_pntr] == '0' || string_received[char_pntr] == '1')
             { CDU_STATUS.backlight = atoi(&string_received[char_pntr]);
               CDU_Status_Update = 1;
             }
           break;
        } 
        
        case 4: // SBY command, standby control CDU      
        {               
             // Get standby status:
             char_pntr = comma[1] + 1; // : set char pointer to char after comma 1
             if (string_received[char_pntr] == '0' || string_received[char_pntr] == '1')
             { CDU_STATUS.stby_mode = atoi(&string_received[char_pntr]);
               CDU_Status_Update = 1;
             }
           break;
        }        
        
        case 5: // CLS command, CDU clear screen control
        {    
             DO_CLR_SCREEN = 1;            // : tell to clear the screen
            break;
        }        


        case 6: // SBC command, screen background colour control       
        {                 
             char_pntr = comma[1] + 1; // : set char pointer to char after comma 1
             colour = atoi(&string_received[char_pntr]);
             if ( colour >= 0 && colour <= max_col_nr ) {
                BACKGROUND_COL.BG_RED = colour;
             }
             char_pntr = comma[2] + 1; // : set char pointer to char after comma 2
             colour = atoi(&string_received[char_pntr]);
             if ( colour >= 0 && colour <= max_col_nr ) {
                BACKGROUND_COL.BG_GREEN = colour;
             }
             char_pntr = comma[3] + 1; // : set char pointer to char after comma 3
             colour = atoi(&string_received[char_pntr]);
             if ( colour >= 0 && colour <= max_col_nr ) {
                BACKGROUND_COL.BG_BLUE = colour; 
             }                       

             Background_Col_Update = 1;    // : set update of background colour
           break;
        }        

        
        case 7: // WTX command, write textline to CDU screen   
        {  
          // Read linenumber:
          char_pntr = comma[1] + 1; // : set char pointer to char after comma 1   
          line = atoi(&string_received[char_pntr]);
          // Read char position:
          char_pntr = comma[2] + 1; // : set char pointer to char after comma 2
          pos = atoi(&string_received[char_pntr]);
           
          // Test if char X,Y position is within range:
          if ( line >= 1 && line <= max_lines && pos >= 1 && pos <= max_text0 )
          {  // Read font size :
                char_pntr = comma[3] + 1; // : set char pointer to char after comma 3
                size = atoi(&string_received[char_pntr]);
                // Test if size is valid:
                if ( size == 0 || size == 1 ) {
                    TEXTLINE[line].font_size = atoi(&string_received[char_pntr]);
                    modified = true;
                }
             // Read line font style:
             char_pntr = comma[4] + 1; // : set char pointer to char after comma 4 
             ch = string_received[char_pntr];
             // !! Font style check disabled:
             //if (ch == fstyle_1 || ch == fstyle_2 ) {
                 TEXTLINE[line].font_style = ch;                                     
                 modified = true;
             //}
             // Read RGB line colour:
             char_pntr = comma[5] + 1; // : set char pointer to char after comma 5
             colour = atoi(&string_received[char_pntr]);
             if ( colour >= 0 && colour <= max_col_nr ) {
                TEXTLINE[line].text_RED = colour;
                modified = true;
             }
             char_pntr = comma[6] + 1; // : set char pointer to char after comma 6
             colour = atoi(&string_received[char_pntr]);
             if ( colour >= 0 && colour <= max_col_nr ) {
                TEXTLINE[line].text_GREEN = colour;
                modified = true;
             }
             char_pntr = comma[7] + 1; // : set char pointer to char after comma 7
             colour = atoi(&string_received[char_pntr]);
             if ( colour >= 0 && colour <= max_col_nr ) {
                TEXTLINE[line].text_BLUE = colour; 
                modified = true;
             }                       
             // Read textfield :
             char_pntr = comma[8] + 1; // : set char pointer to first char of textfield
             last = 0;
             // read font size to determine last possible position:
             if ( TEXTLINE[line].font_size == 0 ) last = max_text0;
             else if ( TEXTLINE[line].font_size == 1 ) last = max_text1;
             
             if (last > 0 && pos <= last) {  // : test for valid area and valid fontsize
                a = pos-1;    // : !!!!!!
                while ( a <= last )
                 {  ch = string_received[char_pntr];
                    if (ch == '*') break; // : do not include * in text
                    TEXTLINE[line].text[a] = ch;
                    a++;
                    char_pntr++;
                 }
                modified = true;
                TEXTLINE[line].text[a] = '\0'; // : mark end of text 
             }            
          } 
           
          if ( modified == true )  {
            Text_Line_Update = line;          // : set update of text line
          }
          
          // --------------- Debug only -----------------------------------------------------------------------
          //SERIAL_DEBUG.printf("\rWTX command received :\r\n");          
          //SERIAL_DEBUG.printf("WTX line        : %d\r\n",line );                         // show line nr 
          //SERIAL_DEBUG.printf("WTX begin pos   : %d\r\n",pos );                          // show begin position   
          //SERIAL_DEBUG.printf("WTX fontsize is : %d\r\n",TEXTLINE[line].font_size );     // show fontsize 
          //SERIAL_DEBUG.printf("WTX fontstyle is: %c\r\n",TEXTLINE[line].font_style);     // show fontstyle 
          //SERIAL_DEBUG.printf("WTX R_colour is : %d\r\n",TEXTLINE[line].text_RED );      // show textcolour 
          //SERIAL_DEBUG.printf("WTX G_colour is : %d\r\n",TEXTLINE[line].text_GREEN );    // show textcolour 
          //SERIAL_DEBUG.printf("WTX B_colour is : %d\r\n",TEXTLINE[line].text_BLUE );     // show textcolour 
          //SERIAL_DEBUG.printf("WTX line text is: \r\n" );           
          //SERIAL_DEBUG.printf("%s\r\n",TEXTLINE[line].text );  // show line text
          //----------------------------------------------------------------------------------------------------
                   
          break;
        }        
        
        case 8: // ETX command, erase (part of) textline of CDU screen        
        { 
          // Read linenumber:
          char_pntr = comma[1] + 1; // : set char pointer to char after comma 1   
          line = atoi(&string_received[char_pntr]);
          // Read char position:
          char_pntr = comma[2] + 1; // : set char pointer to char after comma 2
          pos = atoi(&string_received[char_pntr]);
          // Read nr of char to be erased:
          char_pntr = comma[3] + 1; // : set char pointer to char after comma 3          
          cnt = atoi(&string_received[char_pntr]);
          // Test if char X,Y position is within range:
          if ( line >= 1 && line <= max_lines && pos >= 1 && pos <= max_text0 )
          {  
             last = 0;          
             // read font size to determine last possible position:
             if ( TEXTLINE[line].font_size == 0 )last = max_text0;
             else if ( TEXTLINE[line].font_size == 1 )last = max_text1;
             
             if (last > 0 && pos <= last && cnt > 0) {  // : test if in valid area and chars > 0
                a = pos-1; //: !!!!!!
                modified = true;
                while ( a <= last && cnt > 0)
                 {  
                    TEXTLINE[line].text[a] = ' '; //: write space
                    a++;
                    cnt--;
                 }
             }              
          } 
          if ( modified == true )  {
            Text_Line_Update = line;          // : set update of text line
          }
           
          // ---------------- Debug only -----------------------------------------------
          //SERIAL_DEBUG.printf("ETX line : %d\r\n",line );        // show line nr 
          //SERIAL_DEBUG.printf("ETX first erase pos : %d\r\n",pos );  // show begin pos   
          //SERIAL_DEBUG.printf("ETX modified text line :\r\n" );  
          //SERIAL_DEBUG.printf("%s\r\n",TEXTLINE[line].text );    // show text line
          // ---------------------------------------------------------------------------
        
          break; 
        }        
        
        case 9: // KTX command, write textline to select key area  
        {                    
           // Read key ID:
          char_pntr = comma[1] + 1; // : set char pointer to char after comma 1   
          k_id = atoi(&string_received[char_pntr]);
          
          // Read text type:
          char_pntr = comma[2] + 1; // : set char pointer to char after comma 2 
          ch = string_received[char_pntr];
          
          if (ch == 'M' && k_id < max_keys ) // : text type = MAIN text and keyID valid
          {   // Read font size :
              char_pntr = comma[3] + 1; // : set char pointer to char after comma 3
              size = atoi(&string_received[char_pntr]);
              // Test if size is valid:
              if ( size == 0 || size == 1 ) {
                  SELKEY_MAINTEXT[k_id].font_size = atoi(&string_received[char_pntr]);
                  modified = true;
              }
             // Read line font style:
             char_pntr = comma[4] + 1; // : set char pointer to char after comma 4 
             ch = string_received[char_pntr];
             // !! Font style check disabled:
             //if (ch == fstyle_1 || ch == fstyle_2 ) {
                 SELKEY_MAINTEXT[k_id].font_style = ch;                                     
                 modified = true;
             //}
             // Read RGB line colour:
             char_pntr = comma[5] + 1; // : set char pointer to char after comma 5
             colour = atoi(&string_received[char_pntr]);
             if ( colour >= 0 && colour <= max_col_nr ) {
                SELKEY_MAINTEXT[k_id].text_RED = colour;
                modified = true;
             }
             char_pntr = comma[6] + 1; // : set char pointer to char after comma 5
             colour = atoi(&string_received[char_pntr]);
             if ( colour >= 0 && colour <= max_col_nr ) {
                SELKEY_MAINTEXT[k_id].text_GREEN = colour;
                modified = true;
             }
             char_pntr = comma[7] + 1; // : set char pointer to char after comma 7
             colour = atoi(&string_received[char_pntr]);
             if ( colour >= 0 && colour <= max_col_nr ) {
                SELKEY_MAINTEXT[k_id].text_BLUE = colour; 
                modified = true;
             }                       
             // Read textfield :
             char_pntr = comma[8] + 1; // : set char pointer to first char of textfield
             last = 0;
             // read font size to determine last possible position:
             if ( SELKEY_MAINTEXT[k_id].font_size == 0 )last = max_text0;
             else if ( SELKEY_MAINTEXT[k_id].font_size == 1 )last = max_text1;
             
             if (last > 0) {  // : test on valid fontsize
                a = 0;  // : !!! pos 0 !!
                modified = true;
                while ( a <= last )
                 {  ch = string_received[char_pntr];
                    if (ch == '*') break; // : do not include * in text
                    SELKEY_MAINTEXT[k_id].text[a] = ch;
                    a++;
                    char_pntr++;
                 }
               SELKEY_MAINTEXT[k_id].text[a] = '\0'; // : mark end of text 
             }            
 
             if ( modified == true )  {
              Key_Maintext_Update = k_id;       // : set keynumber whose main text was updated 
              
             }
          
          // ----------------------- Debug only -----------------------------------------------------------------
          //SERIAL_DEBUG.printf("KTX decoded key nr is   : %d\r\n\n",k_id );                   // show key nr 
          //SERIAL_DEBUG.printf("KTX MAINTEXT is : %s\r\n",SELKEY_MAINTEXT[k_id].text );       // show text  
          //SERIAL_DEBUG.printf("KTX fontsize is : %d\r\n",SELKEY_MAINTEXT[k_id].font_size );  // show fontsize 
          //SERIAL_DEBUG.printf("KTX fontstyle is: %c\r\n",SELKEY_MAINTEXT[k_id].font_style);  // show fontstyle 
          //SERIAL_DEBUG.printf("KTX R_colour is : %d\r\n",SELKEY_MAINTEXT[k_id].text_RED );   // show textcolour 
          //SERIAL_DEBUG.printf("KTX G_colour is : %d\r\n",SELKEY_MAINTEXT[k_id].text_GREEN ); // show textcolour 
          //SERIAL_DEBUG.printf("KTX B_colour is : %d\r\n",SELKEY_MAINTEXT[k_id].text_BLUE );  // show textcolour 
          //----------------------------------------------------------------------------------------------------
          }
          
          else if ( ch == 'S' && k_id < max_keys ) // : text type = SUB text and keyID valid
          {   // Read font size 0 - 9:
              char_pntr = comma[3] + 1; // : set char pointer to char after comma 3
              size = atoi(&string_received[char_pntr]);
              // Test if size is valid:
              if ( size == 0 || size == 1 ) {
                  SELKEY_SUBTEXT[k_id].font_size = atoi(&string_received[char_pntr]);
                  modified = true;
              }
             // Read line font style:
             char_pntr = comma[4] + 1; // : set char pointer to char after comma 4 
             ch = string_received[char_pntr];
             if (ch == fstyle_1 || ch == fstyle_2 ) {
                 SELKEY_SUBTEXT[k_id].font_style = ch;                                     
                 modified = true;
             }
             // Read RGB line colour:
             char_pntr = comma[5] + 1; // : set char pointer to char after comma 5
             colour = atoi(&string_received[char_pntr]);
             if ( colour >= 0 && colour <= max_col_nr ) {
                 SELKEY_SUBTEXT[k_id].text_RED = colour;
                 modified = true;
             }
             char_pntr = comma[6] + 1; // : set char pointer to char after comma 6
             colour = atoi(&string_received[char_pntr]);
             if ( colour >= 0 && colour <= max_col_nr ) {
                 SELKEY_SUBTEXT[k_id].text_GREEN = colour;
                 modified = true;
             }
             char_pntr = comma[7] + 1; // : set char pointer to char after comma 7
             colour = atoi(&string_received[char_pntr]);
             if ( colour >= 0 && colour <= max_col_nr ) {
                 SELKEY_SUBTEXT[k_id].text_BLUE = colour; 
                 modified = true;
             }                       
             // Read textfield :
             char_pntr = comma[8] + 1; // : set char pointer to first char of textfield
             last = 0;
             // read font size to determine last possible position:
             if ( SELKEY_SUBTEXT[k_id].font_size == 0 )last = max_text0;
             else if ( SELKEY_SUBTEXT[k_id].font_size == 1 )last = max_text1;
             
             if (last > 0) {  // : test on valid fontsize
                a = 0;  // : !!! pos 0 !!
                modified = true;
                while ( a <= last )
                 {  ch = string_received[char_pntr];
                    if (ch == '*') break; // : do not include * in text
                    SELKEY_SUBTEXT[k_id].text[a] = ch;
                    a++;
                    char_pntr++;
                 }
               SELKEY_SUBTEXT[k_id].text[a] = '\0'; // : mark end of text 
             }            
 
             if ( modified == true )  {
              Key_Subtext_Update = k_id;       // : set keynumber whose sub text was updated 
             }
             
          // ------------------ Debug only ----------------------------------------------------------
          //SERIAL_DEBUG.printf("KTX key nr is   : %d\r\n",k_id );                            // show key nr 
          //SERIAL_DEBUG.printf("KTX SUBTEXT is : %s\r\n",SELKEY_SUBTEXT[k_id].text );        // show text  
          //SERIAL_DEBUG.printf("KTX fontsize is : %d\r\n",SELKEY_SUBTEXT[k_id].font_size );  // show fontsize 
          //SERIAL_DEBUG.printf("KTX fontstyle is: %c\r\n",SELKEY_SUBTEXT[k_id].font_style);  // show fontstyle 
          //SERIAL_DEBUG.printf("KTX R_colour is : %d\r\n",SELKEY_SUBTEXT[k_id].text_RED );   // show textcolour 
          //SERIAL_DEBUG.printf("KTX G_colour is : %d\r\n",SELKEY_SUBTEXT[k_id].text_GREEN ); // show textcolour 
          //SERIAL_DEBUG.printf("KTX B_colour is : %d\r\n",SELKEY_SUBTEXT[k_id].text_BLUE );  // show textcolour 
          // -----------------------------------------------------------------------------------------
          }
         break;
        }        

        case 10:  // FAI command, FAIL indicator control CDU
        {        
             // Get message indicator status:
             char_pntr = comma[1] + 1; // : set char pointer to char after comma 1
             if (string_received[char_pntr] == '0' || string_received[char_pntr] == '1')
             { CDU_STATUS.fail_indicator = atoi(&string_received[char_pntr]);
               CDU_Status_Update = 1;
             }
           break;
        }
        
        case 11:  // DPY command, DISPLAY indicator control CDU
        {        
             // Get message indicator status:
             char_pntr = comma[1] + 1; // : set char pointer to char after comma 1
             if (string_received[char_pntr] == '0' || string_received[char_pntr] == '1')
             { CDU_STATUS.dspy_indicator = atoi(&string_received[char_pntr]);
               CDU_Status_Update = 1;
             }
           break;
        }
        
        case 12:  // OFS command, OFFSET indicator control CDU
        {        
             // Get message indicator status:
             char_pntr = comma[1] + 1; // : set char pointer to char after comma 1
             if (string_received[char_pntr] == '0' || string_received[char_pntr] == '1')
             { CDU_STATUS.ofst_indicator = atoi(&string_received[char_pntr]);
               CDU_Status_Update = 1;
             }
           break;
        }
       
        default:
        {   
          // unknown commandnumber !
         break;
        }        

    }
    
    CDU_OS.SetEvent(FS_DATA_EVENT,CDU_DSP_CSS_TASK_ID); // : set event for CDU screen update task
}
