// L. van der Kolk,  W.Braat  
// File:  main.cpp  
// Version 21 
// ---------------------------- 

#include "mbed.h"
#include "MODSERIAL.h"
#include "mbos.h"
#include "debug_lvdk.h" // : debug mode control LvdK
#include "mbos_def.h"   // : Mbos tasks, timer and resource definitions

// Setup USB communication defined by Mbed USB TX and RX lines
// with TX and RX ringbuffers :
MODSERIAL USB(USBTX, USBRX, 256, 512);
DigitalOut alive_led(LED1); // : LED 1 on Mbed board toggles when CDU is alive

//DigitalOut test_led(LED4); // : LED 4 on Mbed board <<<<<<<< TEST LED !!!

void test_update_flags(); // << for testing only << !!!!!!!

// extra serial port to show debug info :
MODSERIAL SERIAL_DEBUG(p9, p10, 1024, 64);     //: tx, rx

int CDU_FS_interface = 0;   // : defines interface to FS: 0 = USB, 1 = Ethernet
extern int key_hit_ID;
extern PwmOut BGL_LED;

void init_USB(void);
void set_initial_data(); // : fill datastructures with initial data
void Send_KEY_message(int button);
void Send_ALIVE_message(int seconds);
void collect_FSdata();

mbos CDU_OS( 9, 5, 5); // : instantiate mbos O.S (tasks, timers, resources)

// Task function prototypes:
void ALIVE_TASK(void);
void RECEIVE_DECODE_TASK(void);
void SEND_KEYMESSAGE_TASK(void);
void CDU_DSP_CSS_TASK(void);
void READ_POT_TASK(void);

// keyboard function prototypes:
void CDU_KB_COMM_INIT(void);
void CDU_KB_INT_START(void);
void CDU_KB_GET_KEY(void);
void CDU_SET_BGL_INTENSITY(int);

// display function prototype:
void CDU_InitDisplay(void);
void CDU_ScreenAlign(void);
void CDU_StartScreen(void);
void CDU_Page(void);
void CDU_DSP_CSS(void);
void CDU_displayclear(void);


void char_received(MODSERIAL_IRQ_INFO *q)
{
    // -- This functiom will be called on EACH USB receive interrupt --
    // It only sets an mbos Event:
    CDU_OS.SetEvent(RX_EVENT,RECEIVE_DECODE_TASK_ID); // : set event flag for Read Serial Task
}

//------------ USB error interrupts ----------------------------------
void rx_buf_overflow(MODSERIAL_IRQ_INFO *q)
{
    // This functiom will be called when RX buffer overflows.
    // Blue-Leds-of-Death will be showed on Mbed board
    error("RX OVERFLOW !");
}

void tx_buf_overflow(MODSERIAL_IRQ_INFO *q)
{
    // This functiom will be called when TX buffer overflows.
    // Blue-Leds-of-Death will be showed on Mbed board
    error("RX OVERFLOW !");
}
//--------------------------------------------------------------------

void init_USB()
{
    // -- This function controls the setup for the CDU-FS USB communication --
    // Function char_received() will be attached to RX interrupt.
    USB.baud(38400);     // :  set default baudrate to 38400
    USB.rxBufferFlush(); // : empty USB rx buffer in case it is not empty:
    USB.txBufferFlush(); // : empty USB tx buffer in case it is not empty:
    USB.attach(&char_received, MODSERIAL::RxIrq);     // : attach function char_received to USB RX inerrupt
    USB.attach(&rx_buf_overflow, MODSERIAL::RxOvIrq); // : attach function to call when USB RX buffer overflows
    USB.attach(&tx_buf_overflow, MODSERIAL::TxOvIrq); // : attach function to call when USB TX buffer overflows
}

main() // : MBOS main() 
{
    init_USB();  // : setup USB communication to FS
     
    // Keyboard initialisation and keyboard interrrupt setup:
    CDU_KB_COMM_INIT();
    CDU_KB_INT_START();
    BGL_LED.period(0.01);
    
    // Display initialisation:
    CDU_InitDisplay();
    CDU_StartScreen();
    wait(1);
    CDU_Page();
    CDU_ScreenAlign();
    wait(1);
      
        
    set_initial_data(); //: fill FS datastructures with initial (dummy) data

#ifdef DEBUG_LVDK
    //------------------ Testing only ! ----------------------------------
    init_buttons_Mbed();
    USB.printf("\n--- CDU_Mbed_21 ---\r\n");
#endif
    //---------------------------------------------------------------------
      
    SERIAL_DEBUG.baud(38400); // : set baudrate to 38400
    SERIAL_DEBUG.printf("\n--- CDU_Mbed_21 ---\r\n");

    // Create all tasks and timers:
    CDU_OS.CreateTask(ALIVE_TASK_ID, ALIVE_TASK_PRIO, ALIVE_TASK_STACK_SZ, ALIVE_TASK);
    CDU_OS.CreateTask(RECEIVE_DECODE_TASK_ID, RECEIVE_DECODE_TASK_PRIO, RECEIVE_DECODE_TASK_STACK_SZ, RECEIVE_DECODE_TASK);
    CDU_OS.CreateTask(SEND_KEYMESSAGE_TASK_ID, SEND_KEYMESSAGE_TASK_PRIO, SEND_KEYMESSAGE_TASK_STACK_SZ, SEND_KEYMESSAGE_TASK);
    CDU_OS.CreateTask(CDU_DSP_CSS_TASK_ID, CDU_DSP_CSS_TASK_PRIO, CDU_DSP_CSS_TASK_STACK_SZ, CDU_DSP_CSS_TASK );
    CDU_OS.CreateTimer(ALIVE_TIMER_ID, ALIVE_TASK_ID, ALIVE_EVENT);
    CDU_OS.CreateTask(READ_POT_TASK_ID, READ_POT_TASK_PRIO, READ_POT_TASK_STACK_SZ, READ_POT_TASK );
    CDU_OS.CreateTimer(READ_POT_TIMER_ID, READ_POT_TASK_ID, READ_POT_TIMER_EVENT);
       
    // Create resources:
    CDU_OS.CreateResource(USB_TX_RESOURCE, USB_TX_PRIO);
    CDU_OS.CreateResource(FS_DATA_RESOURCE, FS_DATA_PRIO); // : highest prio !

    // Start mbos O.S with a final statement in main() :
    CDU_OS.Start(); // : MBOS is running tasks now....
    // end of main , program never reaches this point !
}

void RECEIVE_DECODE_TASK(void)
{   // Task waits for RX_EVENT, then
    // collects and decodes data from FS
    while(1) { // : loop forever because it is a mbos taskfunction
        CDU_OS.WaitEvent(RX_EVENT); // : wait for RX event
        collect_FSdata();
    }
}

void SEND_KEYMESSAGE_TASK(void)
{
    while(1) {  // : loop forever because it is a mbos taskfunction
        CDU_OS.WaitEvent(KEY_EVENT); // : wait for KEY event ( = "CDU key pressed" )
        CDU_KB_GET_KEY(); // : read key from CDU keyboard
        CDU_OS.LockResource(USB_TX_RESOURCE);
        Send_KEY_message(key_hit_ID);
        key_hit_ID = 0;   // : reset key_hit_ID flag
        CDU_OS.FreeResource(USB_TX_RESOURCE);
    }
}

void ALIVE_TASK(void)
{
    // Mbos task started by mbos timer to send ALIVE message to FS every 5 sec.
    CDU_OS.SetTimer( ALIVE_TIMER_ID, ALIVE_TIMER_PERIOD, ALIVE_TIMER_PERIOD );
    static int seconds = 0;
    while(1) { // : loop forever because it is a mbos taskfunction
        CDU_OS.WaitEvent(ALIVE_EVENT ); // : wait for Mbos timer event (5 sec)
        seconds += 5 ; // : increase seconds by 5
        if (seconds >= 60) seconds = 0 ; // : overflow handling
        CDU_OS.LockResource(USB_TX_RESOURCE);
        Send_ALIVE_message(seconds);
        CDU_OS.FreeResource(USB_TX_RESOURCE);
        alive_led = !alive_led;  // : toggle LED
    }
}

void CDU_DSP_CSS_TASK(void) // : read new screendata from FS datastructures
{
    while(1) { // : loop forever because it is a mbos taskfunction
        CDU_OS.WaitEvent(FS_DATA_EVENT);
        CDU_OS.LockResource(FS_DATA_RESOURCE); //lock resource to prevent intermediate updates
        
        //test_update_flags(); // : >>>>>>>>>>>>>>>>>>>>>   LvdK TEST ONLY !!!
        
        CDU_DSP_CSS();
                    
        CDU_OS.FreeResource(FS_DATA_RESOURCE); //free resource
    }
}

void READ_POT_TASK(void) 
{    // Mbos task started by mbos timer to scan potmeter every 100 msec
     CDU_OS.SetTimer( READ_POT_TIMER_ID, READ_POT_TIMER_PERIOD, READ_POT_TIMER_PERIOD );

    while(1) { // : loop forever because it is a mbos taskfunction
        CDU_OS.WaitEvent(READ_POT_TIMER_EVENT );   // : scan potmeter every 100msec
        CDU_SET_BGL_INTENSITY(255);
    }
}

