//Display control functions --
 
#include "mbed.h"
#include "TFT_4DGL.h"
#include "display.h"
#include <string> 
using namespace std;
 
DigitalOut VGA_SOURCE( p7 ); //control line for video switch between INT and EXT video
DigitalOut VGA_SELECT( p8 ); //control line to select/deselect video switch
 
/*
=====================================================
SGC (Serial Graphics Controller) PLATFORM OUTPUT FILE
=====================================================
 
*******************************************************
* Must set 'New image format' for usage on Picaso SGC *
* Data:                                               *
*  0x59, 0x06, 0x00                                   *
* 4DSL command:                                       *
*  Control(6,0)                                       *
*******************************************************
 
---------------------------------------------------------------------------------------
File "logo_flyengravity.jpg" (logo_flyengravity.jpg)
Sector Address 0x000000
X = 0 Y = 135 Width = 640 Height = 215 Bits = 16
 
Display Image from Memory Card (Serial Command):
Syntax:
@, I, x, y, SectorAdd(hi), SectorAdd(mid), SectorAdd(lo)
Picaso Data:
0x40, 0x49, 0x00, 0x00, 0x00, 0x87, 0x00, 0x00, 0x00
4DSL command:
NewUSDImage(0, 135, 0x000000)
 
---------------------------------------------------------------------------------------
File "Testscreen.png" (Testscreen.png)
Sector Address 0x00021A
X = 0 Y = 0 Width = 640 Height = 480 Bits = 16
 
Display Image from Memory Card (Serial Command):
Syntax:
@, I, x, y, SectorAdd(hi), SectorAdd(mid), SectorAdd(lo)
Picaso Data:
0x40, 0x49, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x1A
4DSL command:
NewUSDImage(0, 0, 0x00021A)
 
*/
 
/*Display Commands
 
$PCDUCLS Erase to active background colour (default black) or white
===================================================================
Syntax: $PCDUCLS, <mode>*<checksum>CRLF
 <mode>=0 : Clear screen to current background colour (default=black)
 <mode>=1 : Clear screen to white
 
$PCDUSBC Set Background Color
====================================================================
Syntax: $PCDUSBC,<Rcolour>,<Gcolour>,<Bcolour>*<checksum>CRLF
 < Rcolour > : RED (000‐255)
 < Gcolour > : GREEN (000‐255)
 < Bcolour > : BLUE (000‐255)
 Set background colour to RGB as defined in the three data fields.
 
$PCDUWTX WTX=WRITE TEXT Write text on any X,Y position on the screen.
=====================================================================
Syntax: $PCDUWTX,<Fsize>,<Fstyle>,<Rcolour>,<Gcolour>,<Bcolour>,<Text><Col>,<Row>*<checksum>CRLF
 <Fsize> : Font size (F0..F9). Currently F0 and F1 are implemented.
 <Fstyle> : Font style (S=Standard, B=Bold, I=Italic, U=Underline, N=Negative)
 < Rcolour > : RED (000‐255)
 < Gcolour > : GREEN (000‐255)
 < Bcolour > : BLUE (000‐255)
 <Text> : Any printable ASCII character except the * character as this is used as the text field delimiter. 
  Maximum length is 24 characters, minimum length is 1 character for font F0.
  Maximum length is 48 characters, minumum length is 1 character for font F1.
 <Col> : Horizontal position of the first character (1..24)
 <Row> : Vertical position of the line (1..14)
 
$PCDUETX ETX=ERASE TEXT
============================================================= 
Syntax: $PCDUETX, <Col>,<Row>,<n>*<checksum>CRLF
 <Col> : Horizontal position of the first character (1..24 for font F0 or 1..48 for font F1)
 <Row> : Vertical position of the line (1‐14)
 <n> : number of characters to be erased (1..24 for font F0 or 1..48 for font F1)
 
$PCDUKTX                Write text attached to a Select Key
===========================================================
Syntax: $PCDUKTX,<KeyID>,<Texttype>,<Fsize>,<Fstyle>,<Rcolour>,<Gcolour>,<Bcolour>,<Text>*< checksum>CRLF
 <KeyID> : Numbering is 00 – 49 for left keys and 50 – 99 for right keys. Top keys are 00 and 50.
 <Texttype> : M or S, meaning Main text or Subtext.
 <Fsize> : Font size (F0..F9). Currently F0 and F1 are implemented.
 <Fstyle> : Font style (S=Standard, B=Bold, I=Italic, U=Underline, N=Negative)
 < Rcolour > : RED (000‐255)
 < Gcolour > : GREEN (000‐255)
 < Bcolour > : BLUE (000‐255)
 <Text> : Any printable ASCII character within the character set.except the * character as this is used as the text field delimiter. 
 Maximum length is 24 characters, minimum length is 1 character for font F0
 Maximum length is 48 characters, minumum length is 1 character for font F1.
*/
 
//Control lines for VGA driver board
TFT_4DGL display(p13,p14,p15); // serial tx, serial rx, reset pin
 
//Character & String functions
 
char* str2char( string cString ) //convert a string to a character array
    {
    int nStrLen=cString.size();
    std::string cInput( cString );
    char* cText = new char[nStrLen+1];
    strncpy(cText, cInput.c_str(), nStrLen);
    cText[nStrLen] = '\0';
    return cText;
    }
    
int centertext( string cString, int nChars, int nCharWidth )
    //calculates the startposition on the screen to center the text
    //needs the actual string (cString), screenwidth in characters (nChars) and the character width (nCharWidth)
    {
    int nStart;
    nStart = nCharWidth*( nChars/2-( cString.size()/2 ));
    return nStart;
    }
    
int righttext( string cString, int nChars, int nCharWidth )
    //calculates the startposition on the screen to right-align the text
    //needs the actual string (cString), screenwidth in characters (nChars) and the character width (nCharWidth)
    {
    int nStart;
    nStart = nCharWidth*( nChars - cString.size());
    return nStart;
    }
 
int nFontSize( int nfont_number )
{
    int nFont = 0;
    switch ( nfont_number )
    {
        case 0:
        {
            nFont = FONT_12X34;
            break;
        }
        case 1:
        {
            nFont = FONT_24X34;
            break;
        }
    }
    return ( nFont );
}    
 
int nFontWidth (int nfont_number )
{
    int nFont = 12;
    switch ( nfont_number )
    {
        case 0:
        {
            nFont = 12;
            break;
        }
        case 1:
        {
            nFont = 24;
            break;
        }
    }
    return ( nFont );
}    
 
unsigned int cRGB( char cRED, char cGREEN, char cBLUE )
{
    //assemble separate colors into 1 integer value as 0xRRGGBB
    //Display driver requires this format
    unsigned int RGB = cBLUE + 256*cGREEN + 65536*cRED;
    return ( RGB );
}    
 
int LeftOrRight( int nTextLine, string cString, int nChars, int nCharWidth )
{
    //decide to print data left aligned or right aligned
    //00 - 49 is left  side of screen
    //50 - 99 is right side of screen
    int nHorPos = 0; 
    // nChars is number of characters on this line (24 or 48)
    // nCharWidth is the character width in pixels
    
    if ( nTextLine < 50 )
    // Left side adjust
    {
        nHorPos = 0;
    }
    else
    // Right side adjust
    {
        nHorPos = righttext( cString, nChars, nCharWidth );
    }
    return ( nHorPos );
}
 
int nLine2Pixel( int nLine )
{
    //calculate vertical pixelposition from linenumber
    int nPixel = 0;
    switch ( nLine )
    {
        case 1:
        {
            nPixel = LINE1;
            break;
        }
        case 2:
        {
            nPixel = LINE2;
            break;
        }
        
        case 3:
        {
            nPixel = LINE3;
            break;
        }
        
        case 4:
        {
            nPixel = LINE4;
            break;
        }
        
        case 5:
        {
            nPixel = LINE5;
            break;
        }
        
        case 6:
        {
            nPixel = LINE6;
            break;
        }
        
        case 7:
        {
            nPixel = LINE7;
            break;
        }
        
        case 8:
        {
            nPixel = LINE8;
            break;
        }
        
        case 9:
        {
            nPixel = LINE9;
            break;
        }
        
        case 10:
        {
            nPixel = LINE10;
            break;
        }
        
        case 11:
        {
            nPixel = LINE11;
            break;
        }
        
        case 12:
        {
            nPixel = LINE12;
            break;
        }
        
        case 13:
        {
            nPixel = LINE13;
            break;
        }
        
        case 14:
        {
            nPixel = LINE14;
            break;
        }
        
    }
    return ( nPixel ) ;
}
 
void VGA_SIGNAL( int Source, int On_Off)
{
    VGA_SOURCE = Source;
    VGA_SELECT = On_Off;
}
 
void CDU_InitDisplay()
{
    display.baudrate( 9600 );       //init uVGAIII card
    VGA_SIGNAL( VGA_INT, VGA_ON );  //select INTERNTAL video and set VGA switch ON
}
 
void CDU_StartScreen()
 {
    string cTitle1="ENGRAVITY";
    string cTitle2="CONTROL & DISPLAY UNIT";
    display.cls();   
    
    display.graphic_string( str2char( cTitle1 ), centertext( cTitle1, 24, LARGECHAR), LINE6, FONT_24X34, WHITE, 1, 1 );
    wait_ms(1000);
    display.graphic_string( str2char( cTitle2 ), centertext( cTitle2, 24, LARGECHAR), LINE8, FONT_24X34, RED, 1, 1 );
    wait_ms(1000);
    display.graphic_string( str2char( cTitle2 ), centertext( cTitle2, 24, LARGECHAR), LINE8, FONT_24X34, GREEN, 1, 1 );
    wait_ms(1000);
    display.graphic_string( str2char( cTitle2) , centertext( cTitle2, 24, LARGECHAR), LINE8, FONT_24X34, BLUE, 1, 1 );
    wait_ms(1000);
    display.graphic_string( str2char( cTitle2) , centertext( cTitle2, 24, LARGECHAR), LINE8, FONT_24X34, WHITE, 1, 1 );
    wait_ms(1000);
 }
 
void CDU_ScreenAlign()
//Draw a wireframe for aligning the screen on display with keys
{
    display.cls();
    
    display.pen_size(WIREFRAME);
    display.rectangle(XMIN,YMIN,XMAX,YMAX, WHITE);
    display.line(XMIN,LINE2,XMAX,LINE2, WHITE);
    display.line(XMIN,LINE3,XMAX,LINE3, WHITE);
    display.line(XMIN,LINE4,XMAX,LINE4, WHITE);
    display.line(XMIN,LINE5,XMAX,LINE5, WHITE);
    display.line(XMIN,LINE6,XMAX,LINE6, WHITE);
    display.line(XMIN,LINE7,XMAX,LINE7, WHITE);
    display.line(XMIN,LINE8,XMAX,LINE8, WHITE);
    display.line(XMIN,LINE9,XMAX,LINE9, WHITE);
    display.line(XMIN,LINE10,XMAX,LINE10, WHITE);
    display.line(XMIN,LINE11,XMAX,LINE11, WHITE);
    display.line(XMIN,LINE12,XMAX,LINE12, WHITE);
    display.line(XMIN,LINE13,XMAX,LINE13, WHITE);
    display.line(XMIN,LINE14,XMAX,LINE14, WHITE);
 
}
 
void CDU_TestScreen()
{
    display.display_control(IMAGE_FORMAT, NEW); //set correct image for reading from SD
    display.cls();
    display.showpicture(0x00, 0x00, 0x00, 0x14, 0x00, 0x01, 0xB5);    // Testscreen
}
 
void CDU_LogoScreen()
{
    display.display_control(IMAGE_FORMAT, NEW); //set correct image for reading from SD    
    display.cls();
    display.showpicture(0x00, 0x00, 0x00, 0x87, 0x00, 0x00, 0x00);    // Engravity logo
}

void CDU_Page()
{
    display.cls();
    display.graphic_string( "PERF INIT" , 8*24,  LINE1, FONT_24X34, WHITE, 1, 1 );
   
    display.graphic_string( "GW/CRZ CG" , 0,  LINE2, FONT_12X34, WHITE, 1, 1 );
    display.graphic_string( "CRZ ALT"   , 41*12, LINE2, FONT_12X34, WHITE, 1, 1 );
    
    display.graphic_string( "___._ / 26.2%" , 0,  LINE3, FONT_24X34, WHITE, 1, 1 );
    display.graphic_string( "_____"     , 19*24, LINE3, FONT_24X34, WHITE, 1, 1 );
 
    display.graphic_string( "FUEL" , 0,  LINE4, FONT_12X34, WHITE, 1, 1 );
    display.graphic_string( "CRZ/WIND"   , 40*12, LINE4, FONT_12X34, WHITE, 1, 1 );
        
    display.graphic_string( "0.0" , 0,  LINE5, FONT_24X34, WHITE, 1, 1 );
    display.graphic_string( "000$ /---" , 15*24,  LINE5, FONT_24X34, WHITE, 1, 1 );
   
    display.graphic_string( "ZFW" , 0,  LINE6, FONT_12X34, WHITE, 1, 1 );
    display.graphic_string( "___._ " , 0,  LINE7, FONT_24X34, WHITE, 1, 1 );
   
    display.graphic_string( "RESERVES" , 0,  LINE8, FONT_12X34, WHITE, 1, 1 );
    display.graphic_string( "__._ " , 0,  LINE9, FONT_24X34, WHITE, 1, 1 );
        
    display.graphic_string( "COST INDEX" , 0,  LINE10, FONT_12X34, WHITE, 1, 1 );
    display.graphic_string( "TRANS ALT"   , 39*12, LINE10, FONT_12X34, WHITE, 1, 1 ); 
    
    display.graphic_string( "___" , 0,  LINE11, FONT_24X34, WHITE, 1, 1 );
    display.graphic_string( "_____"     , 19*24, LINE11, FONT_24X34, WHITE, 1, 1 );
   
    display.graphic_string( "------------------------------------------------"   , 0, LINE12, FONT_12X34, WHITE, 1, 1 ); 
 
    display.graphic_string( "<INDEX" , 0,  LINE13, FONT_24X34, WHITE, 1, 1 );
    display.graphic_string( "N1 LIMIT>"     , 15*24, LINE13, FONT_24X34, WHITE, 1, 1 );
   
    display.graphic_string( "SCRATCHPAD DATA LINE", centertext("SCRATCHPAD DATA LINE", 24, LARGECHAR) , LINE14, FONT_24X34, RED, 1, 1 );
    
}

void  CDU_displayclear(){
    display.cls();
}
    